﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SandboxUtility.h"

namespace
{
const size_t SHAPE_USER_AREA_SIZE = 64;
} // anonymous namespace

nw::g3d::math::Vec3 s_SkyColor = nw::g3d::math::Vec3::Make(0.4f, 0.5f, 0.6f);
nw::g3d::math::Vec3 s_GroundColor = nw::g3d::math::Vec3::Make(0.4f, 0.6f, 0.4f);
nw::g3d::math::Vec3 s_DiffuseColor = nw::g3d::math::Vec3::Make(1.0f, 1.0f, 0.8f);
nw::g3d::math::Vec3 s_SpecularColor = nw::g3d::math::Vec3::Make(1.0f, 1.0f, 1.0f);
nw::g3d::math::Vec3 s_FogColor = nw::g3d::math::Vec3::Make(0.0f, 0.0f, 0.0f);

const char* ENV_NAMES[NUM_ENV_ITEMS] =
{
    "hemiDir",
    "skyColor",
    "groundColor",
    "lightDir",
    "diffuseColor",
    "specularColor",
    "fog_kind",
    "fogStart",
    "fogStartEndInv",
    "fogColor",
    "weather_kind"
};

const char* FOG_NAMES[NUM_FOG_KINDS] =
{
    "linear",
    "exp",
    "exp2"
};

const char* RENDERINFO_NAMES[NUM_RENDERINFO_KINDS] =
{
    "render_priority",
    "clearColor",
};

const char* WEATHER_NAMES[NUM_WEATHER_KINDS] =
{
    "mostly_sunny",
    "cloudiness",
    "rain"
};

CameraObj s_CameraObj;
EnvObj s_EnvObj;

nw::g3d::math::Vec3&
VEC3Transform(nw::g3d::math::Vec3* pDst, const nw::g3d::math::Mtx34& m)
{
    const f32 x = m.m00 * pDst->x + m.m01 * pDst->y + m.m02 * pDst->z + m.m03;
    const f32 y = m.m10 * pDst->x + m.m11 * pDst->y + m.m12 * pDst->z + m.m13;
    const f32 z = m.m20 * pDst->x + m.m21 * pDst->y + m.m22 * pDst->z + m.m23;

    pDst->x = x;
    pDst->y = y;
    pDst->z = z;

    return *pDst;
}

void BuildModel(nw::g3d::res::ResModel* rMdl, std::vector<AnimModelObj*>& models)
{
    NW_G3D_ASSERT_NOT_NULL(rMdl);

    nw::g3d::ModelObj* mdl =
        new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::ModelObj))) nw::g3d::ModelObj();
    nw::g3d::ModelObj::InitArg mdlArg(rMdl);
    mdlArg.EnableBounding();
    mdlArg.ShapeUserAreaSize(SHAPE_USER_AREA_SIZE);
    size_t bufferSize = nw::g3d::ModelObj::CalcBufferSize(mdlArg);
    bool result = mdl->Init(mdlArg, nw::g3d::demo::AllocMem2(bufferSize, nw::g3d::ModelObj::BUFFER_ALIGNMENT), bufferSize);
    NW_G3D_ASSERT(result);
    NW_G3D_UNUSED(result);

    nw::g3d::Mtx34 identity;
    identity.Identity();
    mdl->CalcWorld(identity);
    mdl->CalcBounding();
    for (int idxShape = 0, numShape = mdl->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = mdl->GetShape(idxShape);
        pShapeObj->CalcSubMeshBounding(mdl->GetSkeleton());
        const nw::g3d::Sphere* sphere = pShapeObj->GetBounding();
        nw::g3d::DebugPrint("sphere: %f, %f, %f, %f\n", sphere->center.x, sphere->center.y, sphere->center.z, sphere->radius);
        const nw::g3d::AABB* submeshBounding = pShapeObj->GetSubMeshBoundingArray();
        for (int idxSubmesh = 0, numSubmesh = pShapeObj->GetSubMeshCount(); idxSubmesh < numSubmesh; ++idxSubmesh)
        {
            const nw::g3d::AABB& bounding = submeshBounding[idxSubmesh];
            nw::g3d::DebugPrint("aabb min: %f, %f, %f\n", bounding.min.x, bounding.min.y, bounding.min.z);
            nw::g3d::DebugPrint("aabb max: %f, %f, %f\n", bounding.max.x, bounding.max.y, bounding.max.z);
        }
    }

    AnimModelObj* animModel = new(nw::g3d::demo::AllocMem2(sizeof(AnimModelObj))) AnimModelObj();
    animModel->mdl = mdl;
    models.push_back(animModel);
}

void BuildSklAnim(nw::g3d::res::ResSkeletalAnim* rSklAnim, int maxSklCount, std::vector<nw::g3d::SkeletalAnimObj*>& sklAnims)
{
    NW_G3D_ASSERT_NOT_NULL(rSklAnim);

    nw::g3d::SkeletalAnimObj* anim =
        new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::SkeletalAnimObj))) nw::g3d::SkeletalAnimObj();
    nw::g3d::SkeletalAnimObj::InitArg animArg;
    animArg.Reserve(rSklAnim);
    animArg.SetMaxBoneCount(maxSklCount);

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。
    size_t size = nw::g3d::SkeletalAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, nw::g3d::demo::AllocMem2(size, nw::g3d::SkeletalAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rSklAnim);

    sklAnims.push_back(anim);
}

void BuildBoneVisAnim(nw::g3d::res::ResVisibilityAnim* rVisAnim, int maxSklCount, std::vector<nw::g3d::VisibilityAnimObj*>& boneVisAnims)
{
    NW_G3D_ASSERT_NOT_NULL(rVisAnim);

    nw::g3d::VisibilityAnimObj* anim =
        new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::VisibilityAnimObj))) nw::g3d::VisibilityAnimObj();
    nw::g3d::VisibilityAnimObj::InitArg animArg;
    animArg.Reserve(rVisAnim);
    animArg.SetMaxBoneCount(maxSklCount);

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。
    size_t size = nw::g3d::VisibilityAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, nw::g3d::demo::AllocMem2(size, nw::g3d::VisibilityAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rVisAnim);

    boneVisAnims.push_back(anim);
}

void BuildSceneAnim(nw::g3d::res::ResSceneAnim* rScnAnim,
                    std::vector<nw::g3d::CameraAnimObj*>& cameraAnims,
                    std::vector<nw::g3d::LightAnimObj*>& lightAnims,
                    std::vector<nw::g3d::FogAnimObj*>& fogAnims)
{
    NW_G3D_ASSERT_NOT_NULL(rScnAnim);

    for (int i = 0; i < rScnAnim->GetCameraAnimCount(); ++i)
    {
        nw::g3d::CameraAnimObj* anim =
            new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::CameraAnimObj))) nw::g3d::CameraAnimObj();
        nw::g3d::CameraAnimObj::InitArg animArg;

        nw::g3d::ResCameraAnim* rCameraAnim = rScnAnim->GetCameraAnim(i);
        animArg.SetMaxCurveCount(rCameraAnim->GetCurveCount());
        size_t size = nw::g3d::CameraAnimObj::CalcBufferSize(animArg);
        anim->Init(animArg, nw::g3d::demo::AllocMem2(size, nw::g3d::CameraAnimObj::BUFFER_ALIGNMENT), size);
        anim->SetResource(rCameraAnim);

        cameraAnims.push_back(anim);
    }

    for (int i = 0; i < rScnAnim->GetLightAnimCount(); ++i)
    {
        nw::g3d::LightAnimObj* anim =
            new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::LightAnimObj))) nw::g3d::LightAnimObj();
        nw::g3d::LightAnimObj::InitArg animArg;

        nw::g3d::ResLightAnim* rLightAnim = rScnAnim->GetLightAnim(i);
        animArg.SetMaxCurveCount(rLightAnim->GetCurveCount());
        size_t size = nw::g3d::LightAnimObj::CalcBufferSize(animArg);
        anim->Init(animArg, nw::g3d::demo::AllocMem2(size, nw::g3d::LightAnimObj::BUFFER_ALIGNMENT), size);
        anim->SetResource(rLightAnim);

        lightAnims.push_back(anim);
    }

    for (int i = 0; i < rScnAnim->GetFogAnimCount(); ++i)
    {
        nw::g3d::FogAnimObj* anim =
            new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::FogAnimObj))) nw::g3d::FogAnimObj();
        nw::g3d::FogAnimObj::InitArg animArg;

        nw::g3d::ResFogAnim* rFogAnim = rScnAnim->GetFogAnim(i);
        animArg.SetMaxCurveCount(rFogAnim->GetCurveCount());
        size_t size = nw::g3d::FogAnimObj::CalcBufferSize(animArg);
        anim->Init(animArg, nw::g3d::demo::AllocMem2(size, nw::g3d::FogAnimObj::BUFFER_ALIGNMENT), size);
        anim->SetResource(rFogAnim);

        fogAnims.push_back(anim);
    }
}

void BuildTexPatternAnim(nw::g3d::res::ResTexPatternAnim* rTexAnim,
                         int maxMatCount,
                         std::vector<nw::g3d::TexPatternAnimObj*>& texAnims)
{
    NW_G3D_ASSERT_NOT_NULL(rTexAnim);

    nw::g3d::TexPatternAnimObj* anim =
        new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::TexPatternAnimObj))) nw::g3d::TexPatternAnimObj();
    nw::g3d::TexPatternAnimObj::InitArg animArg;
    animArg.Reserve(rTexAnim);
    animArg.SetMaxMatCount(maxMatCount);

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。
    size_t size = nw::g3d::TexPatternAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, nw::g3d::demo::AllocMem2(size, nw::g3d::TexPatternAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rTexAnim);

    texAnims.push_back(anim);
}

void BuildShapeAnim(nw::g3d::res::ResShapeAnim* rShapeAnim,
                    int maxShapeCount,
                    std::vector<nw::g3d::ShapeAnimObj*>& shapeAnims)
{
    NW_G3D_ASSERT_NOT_NULL(rShapeAnim);

    nw::g3d::ShapeAnimObj* anim =
        new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::ShapeAnimObj))) nw::g3d::ShapeAnimObj();
    nw::g3d::ShapeAnimObj::InitArg animArg(maxShapeCount, rShapeAnim->GetVertexShapeAnimCount(), rShapeAnim->GetKeyShapeAnimCount());

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。。
    size_t size = nw::g3d::ShapeAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, nw::g3d::demo::AllocMem2(size, nw::g3d::ShapeAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rShapeAnim);

    shapeAnims.push_back(anim);
}

extern void BuildParamAnim(nw::g3d::res::ResShaderParamAnim* rParamAnim,
                                 int maxMatCount,
                                 std::vector<nw::g3d::ShaderParamAnimObj*>& paramAnims)
{
    NW_G3D_ASSERT_NOT_NULL(rParamAnim);

    nw::g3d::ShaderParamAnimObj* anim =
        new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::ShaderParamAnimObj))) nw::g3d::ShaderParamAnimObj();
    nw::g3d::ShaderParamAnimObj::InitArg animArg;
    animArg.Reserve(rParamAnim);
    animArg.SetMaxMatCount(maxMatCount);

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。。
    size_t size = nw::g3d::ShaderParamAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, nw::g3d::demo::AllocMem2(size, nw::g3d::ShaderParamAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rParamAnim);

    paramAnims.push_back(anim);
}

void DeleteModel(AnimModelObj* animModel)
{
    NW_G3D_ASSERT_NOT_NULL(animModel);

    nw::g3d::ModelObj* mdl = animModel->mdl;

    void* blockBuffer = mdl->GetBlockBufferPtr();
    if (blockBuffer != NULL)
    {
        nw::g3d::demo::FreeMem2(blockBuffer);
    }

    nw::g3d::demo::FreeMem2(mdl->GetBufferPtr());

    nw::g3d::demo::FreeMem2(mdl);

    nw::g3d::demo::FreeMem2(animModel);
}

void SetupFile(const char* filename, std::vector<nw::g3d::res::ResFile*>& files)
{
    size_t size = 0;
    nw::g3d::res::ResFile* rFile = NULL;
    {
        void* fFile = nw::g3d::demo::LoadFile(filename, &size, 64 * 1024);

        NW_G3D_ASSERT(nw::g3d::res::ResFile::IsValid(fFile));
        rFile =
            nw::g3d::res::ResFile::ResCast(fFile);
    }

    files.push_back(rFile);

    nw::g3d::res::BindResult result =
        rFile->Bind(rFile, nw::g3d::res::BIND_TEXTURE);

    NW_G3D_WARNING(result.IsComplete(), "model binding was not complete.");

    rFile->Setup();
}

void CleanupFile(nw::g3d::res::ResFile* res)
{
    NW_G3D_ASSERT_NOT_NULL(res);
    res->Cleanup();
    nw::g3d::demo::FreeMem2(res);
}

void SetupShaderArchive(const char* filename, std::vector<nw::g3d::res::ResShaderArchive*>& shaderArchives)
{
    size_t size = 0;
    void* fFile = nw::g3d::demo::LoadFile(filename, &size, 0x100);

    NW_G3D_ASSERT(nw::g3d::res::ResShaderArchive::IsValid(fFile));
    nw::g3d::res::ResShaderArchive* pResShaderArchive = nw::g3d::res::ResShaderArchive::ResCast(fFile);
    pResShaderArchive->Setup();

    shaderArchives.push_back(pResShaderArchive);
}

void CleanupShaderArchive(nw::g3d::res::ResShaderArchive* res)
{
    NW_G3D_ASSERT_NOT_NULL(res);
    res->Cleanup();
    nw::g3d::demo::FreeMem2(res);
}

void BindSklAnim(AnimModelObj* animModel, std::vector<nw::g3d::SkeletalAnimObj*>& sklAnims)
{
    nw::g3d::SkeletonObj* skeleton = animModel->mdl->GetSkeleton();

    for (int i = 0; i < static_cast<int>(sklAnims.size()); ++i)
    {
        int animCount = sklAnims[i]->GetAnimCount();
        if (skeleton->GetBoneCount() >= animCount)
        {
            sklAnims[i]->Bind(skeleton);
            animModel->sklAnim = sklAnims[i];
        }
    }
}

void BindBoneVisAnim(AnimModelObj* animModel, std::vector<nw::g3d::VisibilityAnimObj*>& boneVisAnims)
{
    nw::g3d::SkeletonObj* skeleton = animModel->mdl->GetSkeleton();

    for (int i = 0; i < static_cast<int>(boneVisAnims.size()); ++i)
    {
        int animCount = boneVisAnims[i]->GetAnimCount();
        if (skeleton->GetBoneCount() >= animCount)
        {
            boneVisAnims[i]->Bind(animModel->mdl);
            animModel->boneVisAnim = boneVisAnims[i];
        }
    }
}

void BindTexAnim(AnimModelObj* animModel, std::vector<nw::g3d::TexPatternAnimObj*>& texAnims)
{
    for (int i = 0; i < static_cast<int>(texAnims.size()); ++i)
    {
        texAnims[i]->Bind(animModel->mdl);
        animModel->texAnim = texAnims[i];
    }
}

void BindShapeAnim(AnimModelObj* animModel, std::vector<nw::g3d::ShapeAnimObj*>& shapeAnims)
{
    for (int i = 0; i < static_cast<int>(shapeAnims.size()); ++i)
    {
        shapeAnims[i]->Bind(animModel->mdl);
        animModel->shapeAnim = shapeAnims[i];
    }
}

void BindParamAnim(AnimModelObj* animModel, std::vector<nw::g3d::ShaderParamAnimObj*>& paramAnims)
{
    for (int i = 0; i < static_cast<int>(paramAnims.size()); ++i)
    {
        paramAnims[i]->Bind(animModel->mdl);
        animModel->paramAnim = paramAnims[i];
    }
}

void DeleteRenderModel(RenderModel* renderModel)
{
    for (std::vector<RenderObj*>::iterator iterObj = renderModel->objArray.begin(); iterObj != renderModel->objArray.end(); ++iterObj)
    {
        RenderObj* renderObj = *iterObj;
        if (renderObj->clip)
        {
            ShaderClip* clip = renderObj->clip;
            nw::g3d::demo::FreeMem2(clip->fShader->GetShaderPtr());
            clip->fShader->~GfxFetchShader();
            nw::g3d::demo::FreeMem2(clip->fShader);
            clip->~ShaderClip();
            nw::g3d::demo::FreeMem2(clip);
        }

        for (std::vector<SamplerClip*>::iterator iter = renderObj->samplerTable.begin(); iter != renderObj->samplerTable.end(); ++iter)
        {
            SamplerClip* clip = *iter;
            clip->~SamplerClip();
            nw::g3d::demo::FreeMem2(clip);
        }
        renderObj->samplerTable.clear();
        std::vector<SamplerClip*>().swap(renderObj->samplerTable);

        if (renderObj->shaderAssign)
        {
            renderObj->shaderAssign->~ShaderAssign();
            nw::g3d::demo::FreeMem2(renderObj->shaderAssign);
            renderObj->shaderAssign = NULL;
        }

        renderObj->~RenderObj();
        nw::g3d::demo::FreeMem2(renderObj);
    }
    renderModel->objArray.clear();
    std::vector<RenderObj*>().swap(renderModel->objArray);

    renderModel->mdl = NULL;
    renderModel->~RenderModel();
    nw::g3d::demo::FreeMem2(renderModel);
}

void DeleteUniformBlockClip(UniformBlockClip& clip)
{
    nw::g3d::demo::FreeMem2(clip.uBlock->GetData());
    nw::g3d::demo::FreeMem2(clip.uBlock);
}


void SetupShaderAssign(nw::g3d::ResModel* pResModel, nw::g3d::ResShaderArchive** pShaderArchiveArray, std::vector<nw::g3d::ResShaderArchive*>& shaderArchives)
{
    for (int shapeIndex = 0; shapeIndex < pResModel->GetShapeCount(); ++shapeIndex)
    {
        nw::g3d::ResShape* rShape = pResModel->GetShape(shapeIndex);
        nw::g3d::ResMaterial* rMat = pResModel->GetMaterial(rShape->GetMaterialIndex());
        nw::g3d::ResShaderAssign* rShaderAssign = rMat->GetShaderAssign();

        nw::g3d::ResShadingModel* rShadingModel = NULL;
        if (pShaderArchiveArray != NULL)
        {
            // 最適化シェーダ・編集シェーダが存在する場合には使用する。
            nw::g3d::ResShaderArchive* pShaderArchive = pShaderArchiveArray[rShape->GetMaterialIndex()];
            if (pShaderArchive != NULL)
            {
                rShadingModel = pShaderArchive->GetShadingModel(rShaderAssign->GetShadingModelName());
            }
        }
        if (rShadingModel == NULL)
        {
            std::vector<nw::g3d::ResShaderArchive*>::iterator archive =
                std::find_if(shaderArchives.begin(), shaderArchives.end(), ShaderArchiveFinder(rShaderAssign->GetShaderArchiveName()));
            NW_G3D_ASSERTMSG(archive != shaderArchives.end(), "Not found shader archive.");

            rShadingModel = (*archive)->GetShadingModel(rShaderAssign->GetShadingModelName());
            NW_G3D_ASSERTMSG(rShadingModel != NULL, "Not found shading model.");
        }

        nw::g3d::ResShaderProgram* rProgram = NULL;
        {
            u32 keyLength = rShadingModel->GetKeyLength();
            u32* key = static_cast<u32*>(nw::g3d::demo::AllocMem2(keyLength * sizeof(u32)));
            nw::g3d::ShaderUtility::InitShaderKey(key, keyLength, rShadingModel, rShaderAssign);

            int programIndex = rShadingModel->FindProgramIndex(key);
            if (programIndex != -1)
            {
                rProgram = rShadingModel->GetShaderProgram(programIndex);
            }

            NW_G3D_ASSERT_NOT_NULL(rProgram);
            nw::g3d::demo::FreeMem2(key);
        }

        ShaderAssign* pShaderAssign = new(nw::g3d::demo::AllocMem2(sizeof(ShaderAssign))) ShaderAssign();
        pShaderAssign->shadingModel = rShadingModel;
        pShaderAssign->program = rProgram;

        rShape->SetUserPtr(pShaderAssign);
    }
}

void BindShader(RenderModel* renderModel, bool initBlock)
{
    for (int shapeIndex = 0; shapeIndex < renderModel->mdl->GetShapeCount(); ++shapeIndex)
    {
        nw::g3d::ShapeObj* shape = renderModel->mdl->GetShape(shapeIndex);
        nw::g3d::MaterialObj* mat = renderModel->mdl->GetMaterial(shape->GetMaterialIndex());

        ShaderAssign* pShaderAssign = shape->GetResource()->GetUserPtr<ShaderAssign>();
        shape->GetResource()->SetUserPtr(NULL);
        nw::g3d::ResShadingModel* rShadingModel = pShaderAssign->shadingModel;

        if (initBlock)
        {
            // UniformBlock のセットアップ
            int blockIndex = rShadingModel->GetSystemBlockIndex(nw::g3d::ResUniformBlockVar::TYPE_MATERIAL);
            if (blockIndex != -1)
            {
                nw::g3d::ResUniformBlockVar* rBlock = rShadingModel->GetUniformBlock(blockIndex);
                mat->GetResource()->SetRawParamSize(rBlock->GetSize());

                // shader param
                for (int paramIndex = 0; paramIndex < mat->GetShaderParamCount(); ++paramIndex)
                {
                    nw::g3d::ResShaderParam* rParam = mat->GetResShaderParam(paramIndex);
                    nw::g3d::ResUniformVar* rUniform = rBlock->GetUniform(rParam->GetId());

                    NW_G3D_ASSERT_NOT_NULL(rUniform);

                    rParam->SetOffset(rUniform->GetOffset());
                }
            }
        }

#if 0
        // depend のテスト用
        nw::g3d::ResShaderParam* rDiffuseShaderParam = mat->GetResource()->GetShaderParam("diffuse");
        if (rDiffuseShaderParam != NULL)
        {
            if (rDiffuseShaderParam->GetDependedIndex() != rDiffuseShaderParam->GetDependIndex())
            {
                rDiffuseShaderParam->SetConvertParamCallback(DiffuseConverter);
            }
        }
#endif

        nw::g3d::ResRenderInfo* rRenderPriority =
            mat->GetResource()->GetRenderInfo(RENDERINFO_NAMES[RENDER_PRIORITY]);

        // RenderObj を構築する
        RenderObj* obj = new(nw::g3d::demo::AllocMem2(sizeof(RenderObj))) RenderObj();
        obj->shape = shape;
        obj->mat = mat;
        obj->renderPriority = rRenderPriority ? *(rRenderPriority->GetInt()) : 0;
        obj->shaderAssign = pShaderAssign;
        obj->clip = NULL;
        obj->samplerTable.clear();

        renderModel->objArray.push_back(obj);
    }

    if (initBlock)
    {
        u32 bufferSize = renderModel->mdl->CalcBlockBufferSize();
        renderModel->mdl->SetupBlockBuffer(nw::g3d::demo::AllocMem2(bufferSize, nw::g3d::ModelObj::BLOCK_BUFFER_ALIGNMENT), bufferSize);
    }
}

void SetupShaderBinding(RenderModel* renderModel)
{
    for (int objIdx = 0; objIdx < static_cast<int>(renderModel->objArray.size()); ++objIdx)
    {
        RenderObj* obj = renderModel->objArray[objIdx];
        nw::g3d::ShapeObj* shape = obj->shape;
        nw::g3d::MaterialObj* mat = obj->mat;
        nw::g3d::ResMaterial* rMat = mat->GetResource();
        nw::g3d::ResShaderAssign* rShaderAssign = rMat->GetShaderAssign();
        nw::g3d::ResVertex* rVtx = shape->GetResVertex();
        nw::g3d::ResShadingModel* rShadingModel = obj->shaderAssign->shadingModel;

        // ShaderClip は必ず構築する
        // フェッチシェーダを共有する場合はちゃんと判定する。
        {
            ShaderClip* clip = new(nw::g3d::demo::AllocMem2(sizeof(ShaderClip))) ShaderClip();
            obj->clip = clip;
            clip->vertex = rVtx;

            // シェーダに存在するアトリビュートをカウントする。
            std::vector<int> locationArray;
            locationArray.reserve(rShadingModel->GetAttribCount());
            for (int attribIndex = 0; attribIndex < rShadingModel->GetAttribCount(); ++attribIndex)
            {
                const char* assignName = rShadingModel->GetAttribName(attribIndex);
                const char* attribName = rShaderAssign->GetAttribAssign(assignName);
                if (attribName == NULL)
                {
                    continue;
                }

                nw::g3d::ResVtxAttrib* rVtxAttrib = rVtx->GetVtxAttrib(attribName);
                if (rVtxAttrib == NULL)
                {
                    continue;
                }

                u32 location = rShadingModel->GetAttrib(attribIndex)->GetLocation();
                locationArray.push_back(location);
            }

            clip->fShader = new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::fnd::GfxFetchShader))) nw::g3d::fnd::GfxFetchShader();
            clip->fShader->SetAttribCount(locationArray.size());
            clip->fShader->CalcSize();

            u32 size = clip->fShader->GetShaderSize();
            clip->fShader->SetShaderPtr(nw::g3d::demo::AllocMem2(size, nw::g3d::fnd::GfxFetchShader::SHADER_ALIGNMENT));
            clip->fShader->SetDefault();

            // fetch shader を構築する
            int activeAttribIndex = 0;
            for (int attribIndex = 0; attribIndex < rShadingModel->GetAttribCount(); ++attribIndex)
            {
                const char* assignName = rShadingModel->GetAttribName(attribIndex);
                const char* attribName = rShaderAssign->GetAttribAssign(assignName);
                if (attribName == NULL)
                {
                    continue;
                }
                nw::g3d::ResVtxAttrib* rVtxAttrib = rVtx->GetVtxAttrib(attribName);
                if (rVtxAttrib == NULL)
                {
                    continue;
                }

                clip->fShader->SetLocation(activeAttribIndex, locationArray[activeAttribIndex]);
                clip->fShader->SetFormat(activeAttribIndex, rVtxAttrib->GetFormat());
                clip->fShader->SetOffset(activeAttribIndex, rVtxAttrib->GetOffset());
                clip->fShader->SetBufferSlot(activeAttribIndex, rVtxAttrib->GetBufferIndex());
                clip->fShader->SetVertexBuffer(activeAttribIndex, rVtx->GetVtxBuffer(rVtxAttrib->GetBufferIndex())->GetGfxBuffer());
                ++activeAttribIndex;
            }

            clip->fShader->Setup();

            clip->fShader->DCFlush();
        }
        // Sampler のセットアップ
        // TODO: シェーダドリブンに修正する
        for (int samplerIndex = 0; samplerIndex < rShaderAssign->GetSamplerAssignCount(); ++samplerIndex)
        {
            const char* name = rShaderAssign->GetSamplerAssignName(samplerIndex);
            int index = rShadingModel->GetSamplerIndex(name);
            nw::g3d::ResSampler* rSampler = mat->GetResSampler(rShaderAssign->GetSamplerAssign(samplerIndex));

            if (index == -1 || rSampler == NULL)
            {
                continue;
            }

            nw::g3d::ResTexture* rTexture = mat->GetResTexture(rSampler->GetIndex());
            if (rTexture == NULL)
            {
                // TODO: デフォルトテクスチャを挿入する。
                continue;
            }

            SamplerClip* clip = new(nw::g3d::demo::AllocMem2(sizeof(SamplerClip))) SamplerClip();
            clip->shaderSamplerIdx = index;
            clip->sampler = rSampler;

            obj->samplerTable.push_back(clip);
        }
    }
}
