﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nnt/gtest/gtest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnd/tmp451/tmp451.h>

namespace nnt { namespace tmp451 {

struct SettingValues
{
    nnd::tmp451::DigitalFilter  filter;
    nnd::tmp451::Temperature    offset;
    int8_t                      eta;
};

const SettingValues g_PorValues = {nnd::tmp451::DigitalFilter::FilterOff, {0x00, 0x00}, 0};
const SettingValues g_ChangeValues = {nnd::tmp451::DigitalFilter::Filter8Results, {0x01, nnd::tmp451::DecimalUnit}, 4};


// 計測中に設定の取得が可能なことを確認するテスト
TEST(RemoteSettingTest, Get)
{
    using namespace nnd::tmp451;
    Initialize();

    auto result = SetMeasurementMode(MeasurementMode::Continuous);
    EXPECT_TRUE(result.IsSuccess());

    SettingValues values;
    result = GetRemoteDigitalFilterControl(&values.filter);
    EXPECT_TRUE(result.IsSuccess());
    result = GetRemoteTemperatureOffset(&values.offset);
    EXPECT_TRUE(result.IsSuccess());
    result = GetRemoteEtaFactorCorrectionValueByN(&values.eta);
    EXPECT_TRUE(result.IsSuccess());

    // 初期値の確認
    EXPECT_EQ(g_PorValues.filter, values.filter);
    EXPECT_EQ(g_PorValues.offset.integer, values.offset.integer);
    EXPECT_EQ(g_PorValues.offset.decimal, values.offset.decimal);
    EXPECT_EQ(g_PorValues.eta, values.eta);

    Finalize();
}

// 設定の変更ができることを確認するテスト
TEST(RemoteSettingTest, Set)
{
    using namespace nnd::tmp451;
    Initialize();

    // 設定を変更
    auto result = SetRemoteDigitalFilterControl(g_ChangeValues.filter);
    EXPECT_TRUE(result.IsSuccess());
    result = SetRemoteTemperatureOffset(g_ChangeValues.offset);
    EXPECT_TRUE(result.IsSuccess());
    result = SetRemoteEtaFactorCorrectionValueByN(g_ChangeValues.eta);
    EXPECT_TRUE(result.IsSuccess());

    // 設定が変更されているかの確認
    SettingValues values;
    result = GetRemoteDigitalFilterControl(&values.filter);
    EXPECT_TRUE(result.IsSuccess());
    result = GetRemoteTemperatureOffset(&values.offset);
    EXPECT_TRUE(result.IsSuccess());
    result = GetRemoteEtaFactorCorrectionValueByN(&values.eta);
    EXPECT_TRUE(result.IsSuccess());


    // 比較
    EXPECT_EQ(g_ChangeValues.filter, values.filter);
    EXPECT_EQ(g_ChangeValues.offset.integer, values.offset.integer);
    EXPECT_EQ(g_ChangeValues.offset.decimal, values.offset.decimal);
    EXPECT_EQ(g_ChangeValues.eta, values.eta);

    Finalize();
}

// 計測中に設定の変更ができないことを確認するテスト
TEST(RemoteSettingTest, SetError)
{
    using namespace nnd::tmp451;
    Initialize();

    // 計測開始
    auto result = SetMeasurementMode(MeasurementMode::Continuous);
    EXPECT_TRUE(result.IsSuccess());

    // 計測中はエラーになるはず
    result = SetRemoteDigitalFilterControl(g_ChangeValues.filter);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);
    result = SetRemoteTemperatureOffset(g_ChangeValues.offset);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);
    result = SetRemoteEtaFactorCorrectionValueByN(g_ChangeValues.eta);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);
    Finalize();
}

}}
