﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nifm.h>
#include <nn/nn_Log.h>
#include <nn/ntc/shim/ntc_shim.h>
#include <nn/settings/system/settings_Clock.h>
#include <nn/time/time_ApiForRepair.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/util/util_ScopeExit.h>

TEST(TimeRepair, SteadyClockInternalOffset)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForRepair() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    const nn::TimeSpan previousInternalOffset = nn::time::GetStandardSteadyClockInternalOffset();
    const auto internalOffsetSettings = nn::TimeSpan::FromSeconds(nn::settings::system::GetExternalSteadyClockInternalOffset());

    // 永続データと StandardSteadyClockCore のキャッシュが同じであることをテストの前提とする
    EXPECT_EQ(previousInternalOffset.GetSeconds(), internalOffsetSettings.GetSeconds());

    nn::time::SteadyClockTimePoint timePoint;
    NNT_ASSERT_RESULT_SUCCESS( nn::time::StandardSteadyClock::GetCurrentTimePoint(&timePoint) );

    nn::time::PosixTime posixTime;
    NNT_ASSERT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );

    nn::time::SystemClockContext context;
    NNT_ASSERT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&context) );

    // 内部オフセットをプラス100秒
    nn::time::SetStandardSteadyClockInternalOffset(previousInternalOffset + nn::TimeSpan::FromSeconds(100));

    // SetStandardSteadyClockInternalOffset() 実行後もランタイムに内部オフセットは更新されない
    EXPECT_EQ(previousInternalOffset, nn::time::GetStandardSteadyClockInternalOffset());

    // 永続データは更新されているはず
    EXPECT_EQ(internalOffsetSettings.GetSeconds() + 100, nn::settings::system::GetExternalSteadyClockInternalOffset());

    // 内部オフセットを変更してもコンテキストに変化はない
    nn::time::SystemClockContext latestContext;
    NNT_ASSERT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&latestContext) );
    EXPECT_EQ(context, latestContext);
}

TEST(TimeRepair, CalibrateSystemClockWithInternalOffset)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::nifm::Initialize() );

    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForRepair() );
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        nn::time::SuspendAutonomicTimeCorrection();
        // Windowsでこのテスト抜けた直後に自動補正されるので Resume しない.NXではDevMenuCommandで自動補正OFF済.
        // NN_UTIL_SCOPE_EXIT{ nn::time::ResumeAutonomicTimeCorrection(); };

    }

    nn::nifm::SubmitNetworkRequestAndWait();

#if defined (NN_BUILD_CONFIG_OS_WIN)
    if(!nn::nifm::IsNetworkAvailable())
    {
        NN_LOG("Skip TimeRepair.CalibrateSystemClockWithInternalOffset. nn::nifm::IsNetworkAvailable()==false\n");
        return; // CI環境だとネットワーク利用できないかもしれないので終了
    }
#else
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());
#endif

    // デタラメなネットワーク時計にセット
    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        nn::time::PosixTime dummyNetworkTime = { 123456879 };
        NNT_ASSERT_RESULT_SUCCESS(nn::time::SetStandardNetworkSystemClockCurrentTime(dummyNetworkTime));
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForRepair() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    nn::time::SystemClockContext netContext1;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&netContext1));

    nn::time::SystemClockContext userContext1;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetSystemClockContext(&userContext1));

    auto internalOffset = nn::time::GetStandardSteadyClockInternalOffset();
    NN_LOG("(Before)StandardSteadyClockInternalOffset : %d sec.\n", internalOffset.GetSeconds());

    // 特別補正
#if defined (NN_BUILD_CONFIG_OS_WIN)
    auto result = nn::time::CalibrateSystemClockWithInternalOffset();
    if(result.IsFailure())
    {
        NN_LOG("Skip TimeRepair.CalibrateSystemClockWithInternalOffset. 0x%08x\n", result.GetInnerValueForDebug());
        return; // CI環境だと nifm::IsNetworkAvailable()==true でも外と通信できるか不明、なので終了
    }
#else
    NNT_ASSERT_RESULT_SUCCESS(nn::time::CalibrateSystemClockWithInternalOffset());
#endif

    internalOffset = nn::time::GetStandardSteadyClockInternalOffset();
    NN_LOG("(After)StandardSteadyClockInternalOffset : %d sec.\n", internalOffset.GetSeconds());

    // 特別補正の前後でコンテキストは変化しない
    nn::time::SystemClockContext netContext2;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&netContext2));
    EXPECT_EQ(netContext1, netContext2);

    nn::time::SystemClockContext userContext2;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetSystemClockContext(&userContext2));
    EXPECT_EQ(userContext1, userContext2);
}

// StandardNetworkSystemClock が正確な時刻を指しているかを確認するテスト.
// TimeRepair.CalibrateSystemClockWithInternalOffset テスト実行後、システムリセットを行った後に実行する.
TEST(TimeRepair, CheckNetworkClockAccuracy)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::nifm::Initialize() );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForRepair() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    nn::time::SuspendAutonomicTimeCorrection();

    // Windowsでこのテスト抜けた直後に自動補正されるので Resume しない.NXではDevMenuCommandで自動補正OFF済.
    // NN_UTIL_SCOPE_EXIT{ nn::time::ResumeAutonomicTimeCorrection(); };


    nn::nifm::SubmitNetworkRequestAndWait();

#if defined (NN_BUILD_CONFIG_OS_WIN)
    if(!nn::nifm::IsNetworkAvailable())
    {
        NN_LOG("Skip TimeRepair.CalibrateSystemClockWithInternalOffset. nn::nifm::IsNetworkAvailable()==false\n");
        return; // CI環境だとネットワーク利用できないかもしれないので終了
    }
#else
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());
#endif

    // サーバーから直接時刻を取得
    nn::ntc::shim::CorrectionNetworkClockAsyncTask task(
        nn::os::EventClearMode_AutoClear, nn::ntc::EnsureNetworkClockAvailabilityMode_GetServerTime);
    NNT_ASSERT_RESULT_SUCCESS(task.StartTask());
    task.GetFinishNotificationEvent().Wait();
#if defined (NN_BUILD_CONFIG_OS_WIN)
    auto result = task.GetResult();
    if(result.IsFailure())
    {
        NN_LOG("Skip TimeRepair.CheckNetworkClockAccuracy. 0x%08x\n", result.GetInnerValueForDebug());
        return; // CI環境だと nifm::IsNetworkAvailable()==true でも外と通信できるか不明、なので終了
    }
#else
    NNT_ASSERT_RESULT_SUCCESS(task.GetResult());
#endif

    auto serverTime = task.GetServerPosixTime();

    // StandardNetworkSystemClock の時刻とサーバー時刻がほぼ等しいことを確認
    nn::time::PosixTime networkTime;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardNetworkSystemClock::GetCurrentTime(&networkTime));

    auto diffSpan = serverTime - networkTime;
    auto diffSeconds = std::abs(diffSpan.GetSeconds());
    EXPECT_LE(diffSeconds , 30LL); // 通信遅延なども考慮して 30 秒誤差以内であればOKとする
}

