﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testTime_Common.h"

#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_ApiForRepair.h> // SuspendAutonomicTimeCorrection, ResumeAutonomicTimeCorrection
#include <nn/time/time_ApiForSystem.h>

#include <nn/time/detail/time_ClockSnapshotPrivateApi.h>
#include <nn/time/detail/time_ClaimPeriodicBenefit.h>
#include <nn/time/detail/time_SystemClockPrivateApi.h>
#include <nn/time/detail/time_TimeZonePrivateApi.h>

namespace nnt { namespace time { namespace util {

// ClockSnapshotのユーザー時計を操作する
// 自動補正は強制的にOFFになる
void DoUserClockOperation(nn::time::ClockSnapshot* pOut, int64_t add) NN_NOEXCEPT
{
    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);

    p->isAutomaticCorrectionEnabled = false; // 自動補正OFF

    // コンテキスト操作
    p->userSystemClockContext.offset += add;
    p->userSystemClockContext.timeStamp = p->steadyClockTimePoint;

    // 保持するPosixTimeも変化
    p->userSystemClockPosixTime += nn::TimeSpan::FromSeconds(add);
}

// 夏時間などの影響を想定し、 utcOffsetSeconds だけを操作(タイムゾーン変更はなし)
void DoUtcOffsetOperation(nn::time::ClockSnapshot* pOut, int32_t add) NN_NOEXCEPT
{
    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);
    p->userCalendarAdditionalInfo.timeZone.utcOffsetSeconds += add;
}

// タイムゾーンを変更
// pOut はタイムゾーンを変更した時点を保持する
// nn::time::InitializeForMenu() で初期化されている前提
void DoTimeZoneOperation(nn::time::ClockSnapshot* pOut, int32_t add) NN_NOEXCEPT
{
    static const nn::time::LocationName locationNames[2] =
    {
        {"UTC"},
        {"Asia/Tokyo"}
    };

    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);
    for(int i = 0 ; i < NN_ARRAY_SIZE(locationNames) ; i++)
    {
        if(p->locationName != locationNames[i])
        {
            p->locationName = locationNames[i];
            break;
        }
    }

    p->userCalendarAdditionalInfo.timeZone.utcOffsetSeconds += add;

    // タイムゾーン変更して、変更タイミングを SteadyClockTimePoint で取得
    nn::time::SetDeviceLocationName(p->locationName);
    nn::time::SteadyClockTimePoint updatedTime;
    {
        nn::time::LocationName name;
        nn::time::detail::GetDeviceLocationNameAndUpdatedTime(&name, &updatedTime);
    }

    // updatedTime は今現在の時刻になる.
    // システム側とのズレをなくすため、updatedTime を ClockSnapshot の現在時刻として上書きする
    p->steadyClockTimePoint = updatedTime;
    p->userSystemClockPosixTime.value = p->userSystemClockContext.offset + p->steadyClockTimePoint.value;
    if(p->netSystemClockContext.timeStamp.sourceId == p->steadyClockTimePoint.sourceId)
    {
        p->netSystemClockPosixTime.value = p->netSystemClockContext.offset + p->steadyClockTimePoint.value;
    }
}

// ClockSnapshotの時間を自然に進める
void AddElapsed(nn::time::ClockSnapshot* pOut, int64_t value) NN_NOEXCEPT
{
    NN_SDK_ASSERT(value >= 0); // SteadyClockTimePointはマイナスには進まない

    auto addValue = nn::TimeSpan::FromSeconds(value);

    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);
    p->steadyClockTimePoint += addValue;
    p->userSystemClockPosixTime += addValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(
        &p->userCalendarTime, nullptr, p->userSystemClockPosixTime));

    if(p->netSystemClockContext.timeStamp.sourceId == p->steadyClockTimePoint.sourceId)
    {
        // ネットワーク時計が有効であれば進める
        p->netSystemClockPosixTime += addValue;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(
            &p->netCalendarTime, nullptr, p->netSystemClockPosixTime));
    }
}

// ClockSnapshotに対してネットワーク自動時刻補正をかける
void DoAutomaticCorrection(nn::time::ClockSnapshot* pOut, int64_t add) NN_NOEXCEPT
{
    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);

    // ネットワーク時計を補正
    p->netSystemClockContext.offset += add;
    p->netSystemClockContext.timeStamp = p->steadyClockTimePoint;
    p->netSystemClockPosixTime += nn::TimeSpan::FromSeconds(add);

    // 0は未補正状態を表すので拒否
    NN_SDK_ASSERT(p->netSystemClockPosixTime.value != 0);

    // 自動補正ONであればユーザー時計も追従
    if(p->isAutomaticCorrectionEnabled)
    {
        p->userSystemClockContext = p->netSystemClockContext;
        p->userSystemClockPosixTime = p->netSystemClockPosixTime;
    }
}

// 自動補正ON. ネットワーク時計が有効であればユーザー時計は更新される.
void EnableAutomaticCorrection(nn::time::ClockSnapshot* pOut) NN_NOEXCEPT
{
    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);

    p->isAutomaticCorrectionEnabled = true;

    nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(true); // 自動補正ON

    // フラグ変更タイミングを pOut の時刻とする

    nn::time::SteadyClockTimePoint updatedTime;
    nn::time::detail::GetStandardUserSystemClockAutomaticCorrectionUpdatedTime(&updatedTime);

    p->steadyClockTimePoint = updatedTime;
    p->userSystemClockPosixTime.value = p->userSystemClockContext.offset + p->steadyClockTimePoint.value;

    if(p->netSystemClockContext.timeStamp.sourceId == p->steadyClockTimePoint.sourceId)
    {
        p->netSystemClockPosixTime.value = p->netSystemClockContext.offset + p->steadyClockTimePoint.value;

        // ネットワーク時計が有効であればユーザー時計を上書き
        p->userSystemClockContext = p->netSystemClockContext;
        p->userSystemClockPosixTime = p->netSystemClockPosixTime;
    }
}

// 自動補正OFF
void DisableAutomaticCorrection(nn::time::ClockSnapshot* pOut) NN_NOEXCEPT
{
    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);

    p->isAutomaticCorrectionEnabled = false;
}

// ネットワーク時計を未補正に
void UnadjustNetworkClock(nn::time::ClockSnapshot* pOut) NN_NOEXCEPT
{
    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);

    const nn::time::CalendarTime InitialCalendar = {1970, 0, 0, 0, 0, 0};
    p->netSystemClockPosixTime.value = 0;
    p->netCalendarTime = InitialCalendar;

    // 補正時の sourceId を適当に変更(未補正状態)
    p->netSystemClockContext.timeStamp.sourceId = nn::util::GenerateUuid();
}

// ネットワーク時計を補正済に
void AdjustNetworkClock(nn::time::ClockSnapshot* pOut, int64_t posixTimeValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT(posixTimeValue != 0); // 0 は未補正状態を表すので拒否

    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);

    p->netSystemClockPosixTime.value = posixTimeValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&p->netCalendarTime, nullptr, p->netSystemClockPosixTime));
    p->netSystemClockContext.timeStamp = p->steadyClockTimePoint;

    // offset は "context.offset + steadyClockTimePoint.value = PosixTime" になるよう調整
    p->netSystemClockContext.offset = posixTimeValue - p->steadyClockTimePoint.value;

    // 自動補正ONであればユーザー時計も追従
    if(p->isAutomaticCorrectionEnabled)
    {
        p->userSystemClockContext = p->netSystemClockContext;
        p->userSystemClockPosixTime = p->netSystemClockPosixTime;
    }
}

// RTCリセット相当
// ネットワーク時計は未補正、ユーザー時計は初期値に
void DoRtcReset(nn::time::ClockSnapshot* pOut) NN_NOEXCEPT
{
    auto p = nn::time::detail::GetSfClockSnapshotPtr(pOut);

    p->steadyClockTimePoint.value = 0;
    p->steadyClockTimePoint.sourceId = nn::util::GenerateUuid();

    // ユーザー時計初期値
    const nn::time::CalendarTime userCalendar = {2018, 1, 1, 0, 0, 0};
    p->userSystemClockPosixTime = nn::time::ToPosixTimeFromUtc(userCalendar);
    p->userCalendarTime = userCalendar;

    p->userSystemClockContext.timeStamp = p->steadyClockTimePoint;

    // offset は "context.offset + steadyClockTimePoint.value = PosixTime" になるよう調整
    p->userSystemClockContext.offset = p->userSystemClockPosixTime.value - p->steadyClockTimePoint.value;

    // ネットワーク時計初期値(未補正時の初期値)
    UnadjustNetworkClock(pOut);
}

}}} // nnt::time::util

class ClaimPeriodicBenefitTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize()); // win版だと SuspendAutonomicTimeCorrection() に必要

        NNT_ASSERT_RESULT_SUCCESS(nn::time::Initialize());
        nn::time::SuspendAutonomicTimeCorrection();
        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
    }

    static void TearDownTestCase()
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::time::Initialize());
        nn::time::ResumeAutonomicTimeCorrection();
        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
    }
};

const char* DumpCompare(const nn::time::ClockSnapshot& lhs, const nn::time::ClockSnapshot& rhs) NN_NOEXCEPT
{
    NN_LOG("---- user clock ----\n");
    NN_LOG("  posix                      : %lld , %lld\n",
        lhs.GetStandardUserSystemClockPosixTime().value, rhs.GetStandardUserSystemClockPosixTime().value);
    NN_LOG("  context offset             : %lld , %lld\n",
        lhs.GetStandardUserSystemClockContext().offset, rhs.GetStandardUserSystemClockContext().offset);
    NN_LOG("  context timestamp value    : %lld , %lld\n",
        lhs.GetStandardUserSystemClockContext().timeStamp.value, rhs.GetStandardUserSystemClockContext().timeStamp.value);

    char buffer[2][32];
    NN_LOG("  context timestamp sourceId : %s , %s\n",
        lhs.GetStandardUserSystemClockContext().timeStamp.sourceId.ToString(buffer[0], sizeof(buffer[0])),
        rhs.GetStandardUserSystemClockContext().timeStamp.sourceId.ToString(buffer[1], sizeof(buffer[1])));

    const nn::time::CalendarTime &c1 = lhs.GetStandardUserSystemClockCalendarTime();
    const nn::time::CalendarTime &c2 = rhs.GetStandardUserSystemClockCalendarTime();
    NN_LOG("  calendar : %04d/%02d/%02d %02d:%02d:%02d , %04d/%02d/%02d %02d:%02d:%02d\n",
        c1.year, c1.month, c1.day, c1.hour, c1.minute, c1.second,
        c2.year, c2.month, c2.day, c2.hour, c2.minute, c2.second);

    const nn::time::CalendarAdditionalInfo &info1 = lhs.GetStandardUserSystemClockCalendarAdditionalInfo();
    const nn::time::CalendarAdditionalInfo &info2 = rhs.GetStandardUserSystemClockCalendarAdditionalInfo();
    NN_LOG("  day of week          : %d , %d\n",
        static_cast<int>(info1.dayOfWeek), static_cast<int>(info2.dayOfWeek));
    NN_LOG("  year day             : %d , %d\n",
        info1.yearDay, info2.yearDay);

    const nn::time::TimeZone &zone1 = info1.timeZone;
    const nn::time::TimeZone &zone2 = info2.timeZone;
    NN_LOG("  std name             : %s , %s\n",
        zone1.standardTimeName, zone2.standardTimeName);
    NN_LOG("  isDaylightSavingTime : %d , %d\n",
        zone1.isDaylightSavingTime, zone2.isDaylightSavingTime);
    NN_LOG("  utcOffsetSeconds     : %d , %d\n",
        zone1.utcOffsetSeconds, zone2.utcOffsetSeconds);

    NN_LOG("---- steady clock ----\n");
    NN_LOG("  time point value    : %lld , %lld\n",
        lhs.GetStandardSteadyClockTimePoint().value, rhs.GetStandardSteadyClockTimePoint().value);
    NN_LOG("  time point sourceId : %s , %s\n",
        lhs.GetStandardSteadyClockTimePoint().sourceId.ToString(buffer[0], sizeof(buffer[0])),
        rhs.GetStandardSteadyClockTimePoint().sourceId.ToString(buffer[1], sizeof(buffer[1])));

    NN_LOG("----------------------\n");
    return "";
}

const char* DumpPenaltyInfo(const char* caption, const nn::time::SteadyClockTimePoint& tp) NN_NOEXCEPT
{
    NN_LOG("%s\n", caption);
    char buffer[32];
    NN_LOG("  _penaltyEndSteadyClockTimePoint.value    : %lld\n", tp.value);
    NN_LOG("  _penaltyEndSteadyClockTimePoint.sourceId : %s\n", tp.sourceId.ToString(buffer, sizeof(buffer)));

    return "";
}

template<bool isSameTiming, bool isMatchedTiming>
class TestCalendarChecker : public nn::time::CheckCalendarTimeCallback
{
public:
    virtual bool IsSameTiming(
        const nn::time::CalendarTime& lhs,
        const nn::time::CalendarTime& rhs) const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(lhs);
        NN_UNUSED(rhs);
        return isSameTiming;
    }

    virtual bool IsMatched(const nn::time::CalendarTime& calendarTime) const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(calendarTime);
        return isMatchedTiming;
    }
};

const char* GetResultString(nn::time::PeriodicBenefitClaimResult result) NN_NOEXCEPT
{
    switch(result)
    {
    case nn::time::PeriodicBenefitClaimResult_Success:              return "PeriodicBenefitClaimResult_Success";
    case nn::time::PeriodicBenefitClaimResult_NotReceivableTiming:  return "PeriodicBenefitClaimResult_NotReceivableTiming";
    case nn::time::PeriodicBenefitClaimResult_Penalty:              return "PeriodicBenefitClaimResult_Penalty";
    default:                                                        return "invalid";
    }
}

template<bool isSameTiming, bool isMatchedTiming>
void TestClaimPeriodicBenefit(
    nn::time::PeriodicBenefitClaimResult expectResult,
    const nn::TimeSpan& expectPenaltyEnd,
    nn::time::PeriodicBenefitClaimContext* pHandover,
    const nn::time::ClockSnapshot& currentSnapshot,
    const nn::TimeSpan& maxPenaltyTimeSpan,
    const nn::TimeSpan& acceptableOperationTimeSpan) NN_NOEXCEPT
{
    TestCalendarChecker<isSameTiming, isMatchedTiming> checker;

    bool isInitialCall = nn::time::detail::GetClockSnapshotInitialType(pHandover->_snapshot) == nn::time::detail::ClockSnapshotInitialType::Constructor;

    bool isSteadyClockContinuous =
        pHandover->_snapshot.GetStandardSteadyClockTimePoint().sourceId == currentSnapshot.GetStandardSteadyClockTimePoint().sourceId;

    auto result = nn::time::ClaimPeriodicBenefitWithUserSystemClock(
        pHandover, currentSnapshot, maxPenaltyTimeSpan, acceptableOperationTimeSpan, &checker);

    ASSERT_EQ(expectResult, result) << "expect:" << GetResultString(expectResult) << " != " << "actual:" << GetResultString(result);

    if(result == nn::time::PeriodicBenefitClaimResult_Success)
    {
        // 利益受け取り時 _lastBenefitReceivedSnapshot と current は必ず同じ
        EXPECT_EQ(currentSnapshot, pHandover->_lastBenefitReceivedSnapshot)
            << DumpCompare(currentSnapshot, pHandover->_snapshot);

        // pHandover->_snapshot (時計操作量計算の起点) も必ず更新される
        EXPECT_EQ(currentSnapshot, pHandover->_snapshot)
            << DumpCompare(currentSnapshot, pHandover->_snapshot);
    }

    if(!isSteadyClockContinuous || isInitialCall)
    {
        // 初回の判定、もしくは
        // SteadyClock連続性がない場合、 利益を得られなくても pHandover->_snapshot (時計操作量計算の起点) は必ず更新される
        EXPECT_EQ(currentSnapshot, pHandover->_snapshot)
            << DumpCompare(currentSnapshot, pHandover->_snapshot);
    }

    // ペナルティ確認
    if(expectPenaltyEnd == nn::TimeSpan(0))
    {
        EXPECT_FALSE(nn::time::detail::HasPenaltyInfo(*pHandover))
            << DumpPenaltyInfo("pHandover->_penaltyEndSteadyClockTimePoint", pHandover->_penaltyEndSteadyClockTimePoint);

        nn::time::SteadyClockTimePoint penaltyEnd;
        EXPECT_FALSE(pHandover->GetEndOfPenaltyAtLastClaimForDebug(&penaltyEnd));
    }
    else
    {
        EXPECT_TRUE(nn::time::detail::HasPenaltyInfo(*pHandover))
            << DumpPenaltyInfo("pHandover->_penaltyEndSteadyClockTimePoint", pHandover->_penaltyEndSteadyClockTimePoint);

        auto answer = currentSnapshot.GetStandardSteadyClockTimePoint() + expectPenaltyEnd;
        EXPECT_EQ(pHandover->_penaltyEndSteadyClockTimePoint, answer)
            << DumpPenaltyInfo("pHandover->_penaltyEndSteadyClockTimePoint", pHandover->_penaltyEndSteadyClockTimePoint)
            << DumpPenaltyInfo("answer", answer);

        nn::time::SteadyClockTimePoint penaltyEnd;
        EXPECT_TRUE(pHandover->GetEndOfPenaltyAtLastClaimForDebug(&penaltyEnd));
        EXPECT_EQ(pHandover->_penaltyEndSteadyClockTimePoint, penaltyEnd);
    }
}

// ClaimPeriodicBenefitTest で利用する time::detail::GetStandardUserSystemClockAutomaticCorrectionUpdatedTime() のテスト
TEST(ClaimPeriodicBenefitBaseTest, StandardUserSystemClockAutomaticCorrectionUpdatedTime)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    {
        nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(true);

        nn::time::SteadyClockTimePoint tp;
        nn::time::StandardSteadyClock::GetCurrentTimePoint(&tp);

        nn::time::SteadyClockTimePoint updatedTime;
        nn::time::detail::GetStandardUserSystemClockAutomaticCorrectionUpdatedTime(&updatedTime);

        ASSERT_EQ(updatedTime.sourceId, tp.sourceId);

        auto diff = tp.value - updatedTime.value;
        EXPECT_TRUE(diff >= 0 && diff <= 2) << diff;
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));

    {
        nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(false);

        nn::time::SteadyClockTimePoint tp;
        nn::time::StandardSteadyClock::GetCurrentTimePoint(&tp);

        nn::time::SteadyClockTimePoint updatedTime;
        nn::time::detail::GetStandardUserSystemClockAutomaticCorrectionUpdatedTime(&updatedTime);

        ASSERT_EQ(updatedTime.sourceId, tp.sourceId);

        auto diff = tp.value - updatedTime.value;
        EXPECT_TRUE(diff >= 0 && diff <= 2) << diff;
    }
}

// テスト用 nnt::time::util の簡易的なテスト
TEST_F(ClaimPeriodicBenefitTest, TestUtil)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::time::ClockSnapshot snapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&snapshot);

    {
        nn::time::ClockSnapshot value = snapshot;

        nnt::time::util::AddElapsed(&value, 100);

        nn::TimeSpan span;
        ASSERT_TRUE(nn::time::CalculateSpanBetween(&span, snapshot, value));
        EXPECT_EQ(100, span.GetSeconds());
    }
    {
        nn::time::ClockSnapshot value = snapshot;

        nnt::time::util::DoUserClockOperation(&value, 100);
        EXPECT_EQ(value.GetStandardSteadyClockTimePoint().sourceId, value.GetStandardUserSystemClockContext().timeStamp.sourceId);

        nn::TimeSpan span = nn::time::CalculateStandardUserSystemClockDifferenceByUser(snapshot, value);
        EXPECT_EQ(100, span.GetSeconds());
    }
    {
        nn::time::ClockSnapshot value = snapshot;

        const auto UserClockPosixTime = value.GetStandardUserSystemClockPosixTime();

        nnt::time::util::DisableAutomaticCorrection(&value);

        nnt::time::util::UnadjustNetworkClock(&value);
        EXPECT_EQ(value.GetStandardNetworkSystemClockPosixTime().value, 0);
        EXPECT_NE(value.GetStandardNetworkSystemClockContext().timeStamp.sourceId, value.GetStandardSteadyClockTimePoint().sourceId);
        EXPECT_EQ(value.GetStandardSteadyClockTimePoint().sourceId, value.GetStandardUserSystemClockContext().timeStamp.sourceId);

        nnt::time::util::AdjustNetworkClock(&value, 123456789LL);
        EXPECT_EQ(123456789LL, value.GetStandardNetworkSystemClockPosixTime().value);
        EXPECT_EQ(value.GetStandardNetworkSystemClockContext().timeStamp.sourceId, value.GetStandardSteadyClockTimePoint().sourceId);
        EXPECT_EQ(value.GetStandardSteadyClockTimePoint().sourceId, value.GetStandardUserSystemClockContext().timeStamp.sourceId);

        // 自動補正OFFなのでユーザー時計は変わらない
        EXPECT_EQ(UserClockPosixTime, value.GetStandardUserSystemClockPosixTime());

        // 自動補正ONにすると、ネットワーク時計につられてユーザー時計も変わる
        nnt::time::util::EnableAutomaticCorrection(&value);
        nnt::time::util::AdjustNetworkClock(&value, 987654321LL);
        EXPECT_EQ(987654321LL, value.GetStandardNetworkSystemClockPosixTime().value);
        EXPECT_EQ(value.GetStandardNetworkSystemClockContext().timeStamp.sourceId, value.GetStandardSteadyClockTimePoint().sourceId);
        EXPECT_EQ(987654321LL, value.GetStandardUserSystemClockPosixTime().value);
    }
    {
        nn::time::ClockSnapshot value = snapshot;

        nnt::time::util::DoRtcReset(&value);
        // sourceId 更新されているはず
        EXPECT_NE(value.GetStandardSteadyClockTimePoint().sourceId, snapshot.GetStandardSteadyClockTimePoint().sourceId);
        // ユーザー時計の SystemClockContext.timeStamp.sourceId は現在のと同じ
        EXPECT_EQ(value.GetStandardSteadyClockTimePoint().sourceId, value.GetStandardUserSystemClockContext().timeStamp.sourceId);
        // ネットワーク時計は未補正へ
        EXPECT_EQ(value.GetStandardNetworkSystemClockPosixTime().value, 0);
        EXPECT_NE(value.GetStandardSteadyClockTimePoint().sourceId, value.GetStandardNetworkSystemClockContext().timeStamp.sourceId);
    }
}

// コンストラクタのままの初回想定の ClaimPeriodicBenefitWithUserSystemClock
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock1)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromDays(1);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromMinutes(10);

    // PeriodicBenefitClaimResult_Success
    {
        nn::time::PeriodicBenefitClaimContext handover;
        nn::time::ClockSnapshot currentSnapshot;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }

    // PeriodicBenefitClaimResult_NotReceivableTiming
    // isSameTiming は初回は関係なく、マッチしてないので利益を得られない
    {
        nn::time::PeriodicBenefitClaimContext handover;
        nn::time::ClockSnapshot currentSnapshot;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }

    // PeriodicBenefitClaimResult_Success
    // isSameTiming は初回は関係なく、false であっても利益を得られる
    {
        nn::time::PeriodicBenefitClaimContext handover;
        nn::time::ClockSnapshot currentSnapshot;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }

    // PeriodicBenefitClaimResult_NotReceivableTiming
    // isSameTiming は初回は関係なく、マッチしてないので利益を得られない
    {
        nn::time::PeriodicBenefitClaimContext handover;
        nn::time::ClockSnapshot currentSnapshot;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }
} // NOLINT(impl/function_size)


// 時刻操作なしの ClaimPeriodicBenefitWithUserSystemClock
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock2)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromDays(1);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromMinutes(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::DisableAutomaticCorrection(&currentSnapshot); // 自動補正OFF

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 時刻操作なし、適当に進む
    nnt::time::util::AddElapsed(&currentSnapshot, 1000);

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 時刻操作なし、適当に進む
    nnt::time::util::AddElapsed(&currentSnapshot, 1000);

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 時刻操作なし、適当に進む
    nnt::time::util::AddElapsed(&currentSnapshot, 1000);

    // 同じタイミングはダメ
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 時刻操作なし、適当に進む
    nnt::time::util::AddElapsed(&currentSnapshot, 1000);

    // タイミング違うのでOK
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 時刻操作なし、適当に進む
    nnt::time::util::AddElapsed(&currentSnapshot, 1000);

    // 前回と同じタイミングかつ、利益を得られるタイミングでない、のは理論上ありえないがテストしとく
    // マッチしてないので利益得られない
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
} // NOLINT(impl/function_size)

// 未来に進む方向への時計操作
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock3)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::DisableAutomaticCorrection(&currentSnapshot); // 自動補正OFF

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::DoUserClockOperation(&currentSnapshot, 1); // AcceptableOperationTimeSpan以下なので許容されるはず
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::DoUserClockOperation(&currentSnapshot, 10); // AcceptableOperationTimeSpan以下なので許容されるはず
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::DoUserClockOperation(&currentSnapshot, 11); // AcceptableOperationTimeSpan を超える
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::AddElapsed(&currentSnapshot, 100); // ペナルティ100秒残ってる
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan - nn::TimeSpan::FromSeconds(100),
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::AddElapsed(&currentSnapshot, 99); // ペナルティ1秒残ってる
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan - nn::TimeSpan::FromSeconds(199),
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::AddElapsed(&currentSnapshot, 1); // ペナルティちょうど終わり
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 初回の利益受け取りなので isSameTiming=true でもOK. 本来 CheckCalendarTimeCallback::IsSameTiming が呼ばれもしないのでありえない.
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // タイミング同じなのでダメ
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::DoUserClockOperation(&currentSnapshot, 11); // AcceptableOperationTimeSpan 以上なのでMaxPenaltyTimeSpanかかる
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::AddElapsed(&currentSnapshot, 200); // ペナルティちょうど終わり
    // MaxPenaltyTimeSpan 後は同じタイミングでもOK
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    ::testing::Test::HasFatalFailure();

} // NOLINT(impl/function_size)

// 過去に戻る方向への時計操作
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock4)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::DisableAutomaticCorrection(&currentSnapshot); // 自動補正OFF

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    {
        nnt::time::util::DoUserClockOperation(&currentSnapshot, -5); // 5秒ペナルティのはず
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(5),
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        nnt::time::util::AddElapsed(&currentSnapshot, 4); // まだペナルティ中
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(1),
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        nnt::time::util::AddElapsed(&currentSnapshot, 1); // ペナルティちょうどあける
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }

    {
        nnt::time::util::DoUserClockOperation(&currentSnapshot, -500); // MaxPenaltyTimeSpanうける
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        nnt::time::util::AddElapsed(&currentSnapshot, 300); // ペナルティあける
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }

    {
        nnt::time::util::DoUserClockOperation(&currentSnapshot, -199); // 199秒ペナルティのはず
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(199),
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        nnt::time::util::AddElapsed(&currentSnapshot, 198); // まだペナルティ中
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(1),
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        nnt::time::util::AddElapsed(&currentSnapshot, 1); // ペナルティちょうどあける
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>( // MaxPenaltyTimeSpanうけたわけではないので同じタイミングだと利益得られない
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }

    {
        nnt::time::util::DoUserClockOperation(&currentSnapshot, -199); // 199秒ペナルティのはず
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(199),
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        nnt::time::util::AddElapsed(&currentSnapshot, 150); // まだペナルティ中

        nnt::time::util::DoUserClockOperation(&currentSnapshot, -100); // ペナルティ中に操作を重ねる. ここから100秒ペナルティのはず
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(100),
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        nnt::time::util::AddElapsed(&currentSnapshot, 100); // ペナルティちょうどあける
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>( // MaxPenaltyTimeSpanうけたわけではないので同じタイミングだと利益得られない
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }

    {
        nnt::time::util::DoUserClockOperation(&currentSnapshot, -199); // 199秒ペナルティのはず
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(199),
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        nnt::time::util::AddElapsed(&currentSnapshot, 100); // まだペナルティ中

        nnt::time::util::DoUserClockOperation(&currentSnapshot, -50); // ペナルティ中に操作を重ねる. 過去のペナルティのほうが長くあと99秒ペナルティのはず
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(99),
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        nnt::time::util::AddElapsed(&currentSnapshot, 99); // ペナルティちょうどあける
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>( // MaxPenaltyTimeSpanうけたわけではないので同じタイミングだと利益得られない
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }
} // NOLINT(impl/function_size)

// 時計を進める、戻すの混合
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock5)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::DisableAutomaticCorrection(&currentSnapshot); // 自動補正OFF

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 10進める. AcceptableOperationTimeSpan内なのでペナルティなし
    nnt::time::util::DoUserClockOperation(&currentSnapshot, 10);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 10戻す.
    nnt::time::util::DoUserClockOperation(&currentSnapshot, -10);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(10),
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // ペナルティあける
    nnt::time::util::AddElapsed(&currentSnapshot, 10);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
} // NOLINT(impl/function_size)

// タイムゾーン変更せず、夏時間などの影響で時計が操作された場合
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock6)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::DisableAutomaticCorrection(&currentSnapshot); // 自動補正OFF

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 100進める. LocationName 変更していないのでペナルティなし
    nnt::time::util::DoUtcOffsetOperation(&currentSnapshot, 100);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 100戻す. LocationName 変更していないのでペナルティなし
    nnt::time::util::DoUtcOffsetOperation(&currentSnapshot, -100);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 100進める. LocationName 変更していないのでペナルティなしだが、時計を直接1秒ずらす
    nnt::time::util::DoUtcOffsetOperation(&currentSnapshot, 100);
    nnt::time::util::DoUserClockOperation(&currentSnapshot, 1); // 時計操作.AcceptableOperationTimeSpan 以内なのでペナルティないはず
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

} // NOLINT(impl/function_size)

// タイムゾーン変更
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock7)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() ); // DoTimeZoneOperation()利用のため ForMenu で初期化
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);
    nnt::time::util::DisableAutomaticCorrection(&currentSnapshot); // 自動補正OFF

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // タイムゾーン変更によって3戻る
    nnt::time::util::DoTimeZoneOperation(&currentSnapshot, -3);

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(3),
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // ペナルティあける
    nnt::time::util::AddElapsed(&currentSnapshot, 3);

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // タイムゾーン変更によって3進む(AcceptableOperationTimeSpan以内なのでペナルティなし)
    nnt::time::util::DoTimeZoneOperation(&currentSnapshot, 3);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // タイムゾーン変更によって3進む(AcceptableOperationTimeSpan以内なのでペナルティなし)
    nnt::time::util::DoTimeZoneOperation(&currentSnapshot, 3);
    // 時計操作してぎりぎり AcceptableOperationTimeSpan を超える
    nnt::time::util::DoUserClockOperation(&currentSnapshot, 8); // 合計11進めた

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // ペナルティあける
    nnt::time::util::AddElapsed(&currentSnapshot, 200);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 普通に操作してペナルティをうけたあと、タイムゾーンをかえてもとに戻す
    // (ペナルティが始まるタイミングのテスト)
    {
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

        nnt::time::util::DoUserClockOperation(&currentSnapshot, 100); // MaxPenaltyTimeSpanをうける

        const auto PenaltyStartSteadyClockTimePoint = currentSnapshot.GetStandardSteadyClockTimePoint();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5)); // 以降のタイムゾーン変更タイミングをずらす

        // タイムゾーンを変更して戻す
        const nn::time::LocationName location = currentSnapshot.GetLocationName();
        nnt::time::util::DoTimeZoneOperation(&currentSnapshot, 1234);
        nnt::time::util::DoTimeZoneOperation(&currentSnapshot, -1234);
        ASSERT_EQ(location, currentSnapshot.GetLocationName());

        // タイムゾーンは前回の ClaimPeriodicBenefitWithUserSystemClock() からは変わっておらず、
        // DoUserClockOperation した瞬間からペナルティが始まっているはず

        int64_t elapsed;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::GetSpanBetween(
            &elapsed, PenaltyStartSteadyClockTimePoint, currentSnapshot.GetStandardSteadyClockTimePoint()));

        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan - nn::TimeSpan::FromSeconds(elapsed),
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }
} // NOLINT(impl/function_size)

// 単調増加クロックの連続性がないケース(ネットワーク時計未補正)
/*
 *  pHandover が保持する前回利益を得たタイミングからの経過時間取得の成否によって、ペナルティ期間を決める
 *
 *  - 利益を一度も受け取ったことがなく経過時間を取得できない
 *      ペナルティ期間なし
 *  - 経過時間の取得成功
 *      前回利益を受け取ったタイミングから maxPenaltyTimeSpan がペナルティ期間
 *      すでにペナルティ期間が過ぎ去っていることがある
 *  - 経過時間の取得失敗
 *      currentSnapshot から maxPenaltyTimeSpan がペナルティ期間
 *
 *  ただし、 過去の判定タイミングの時点と currentSnapshot がそれぞれ
 *  本体設定の「インターネットで時間をあわせる」がON、かつネットワーク時計が補正済の場合は、ペナルティなし.
 */
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock8)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::UnadjustNetworkClock(&currentSnapshot); // ネットワーク時計未補正
    nnt::time::util::DisableAutomaticCorrection(&currentSnapshot); // 自動補正OFF

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 利益を一度もGetしたことがないのでRTCリセットしてもペナルティなし
    nnt::time::util::DoRtcReset(&currentSnapshot);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::AddElapsed(&currentSnapshot, 1000);

    // 利益を一度受け取る
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::AddElapsed(&currentSnapshot, 10);

    // RTCリセットすると、前回利益をGetしたタイミングから MaxPenaltyTimeSpan のペナルティだが、
    // ネットワーク時計が補正されていないので、経過時間を計算できず、今から MaxPenaltyTimeSpan のペナルティ
    nnt::time::util::DoRtcReset(&currentSnapshot);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::AddElapsed(&currentSnapshot, 200);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
}

// 単調増加クロックの連続性がないケース(ネットワーク時計補正済)
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock9)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::DisableAutomaticCorrection(&currentSnapshot); // 自動補正OFF
    nnt::time::util::AdjustNetworkClock(&currentSnapshot, 1495257519LL); // ネットワーク時計補正済

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 利益を一度もGetしたことがないのでRTCリセットしてもペナルティなし
    nnt::time::util::DoRtcReset(&currentSnapshot);
    nnt::time::util::AdjustNetworkClock(&currentSnapshot, 1495257519LL + 10);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 利益を一度受け取る
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // RTCリセット後にネットワーク時計補正
    nnt::time::util::DoRtcReset(&currentSnapshot);
    nnt::time::util::AdjustNetworkClock(&currentSnapshot, 1495257519LL + 20); // 利益受け取りから10秒経過

    // RTCリセットすると、前回利益をGetしたタイミングから MaxPenaltyTimeSpan のペナルティ(残り190秒)
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_Penalty, nn::TimeSpan::FromSeconds(190),
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::AddElapsed(&currentSnapshot, 190);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // RTCリセット後にネットワーク時計補正
    nnt::time::util::DoRtcReset(&currentSnapshot);
    nnt::time::util::AdjustNetworkClock(&currentSnapshot, 1495257519LL + 2000); // 前回利益受け取りから MaxPenaltyTimeSpan 以上が経過

    // 前回利益をGetしたタイミングから MaxPenaltyTimeSpan のペナルティだが、既に過ぎ去っている
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
}

// 単調増加クロックの連続性がないケース(ネットワーク時計補正済,かつ自動補正ON)
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock10)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::EnableAutomaticCorrection(&currentSnapshot); // 自動補正ON
    nnt::time::util::AdjustNetworkClock(&currentSnapshot, 1495257519LL); // ネットワーク時計補正済

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // RTCリセット後にネットワーク時計補正
    nnt::time::util::DoRtcReset(&currentSnapshot);
    nnt::time::util::AdjustNetworkClock(&currentSnapshot, 1495257519LL + 10);

    // RTCリセットを跨いで、自動補正ON,かつネットワーク時計補正済なので、ペナルティなし
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
}

// handoverSnapshot を使いまわして、実際にあり得そうないくつかのケースでのテスト
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock11)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::EnableAutomaticCorrection(&currentSnapshot); // 自動補正ON
    nnt::time::util::AdjustNetworkClock(&currentSnapshot, 1495257519LL); // ネットワーク時計補正済

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // ネットワーク自動補正
    nnt::time::util::DoAutomaticCorrection(&currentSnapshot, 10);

    // 利益受け取りタイミングになった
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 適当に時間経過
    nnt::time::util::AddElapsed(&currentSnapshot, 1000LL);

    // ネットワーク時計の自動補正が発生(ユーザー時計もずれる)
    nnt::time::util::DoAutomaticCorrection(&currentSnapshot, 10LL);

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 利益受け取りタイミングになった
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 適当に時間経過
    nnt::time::util::AddElapsed(&currentSnapshot, 1000LL);

    // 時計を進める
    nnt::time::util::DoUserClockOperation(&currentSnapshot, 10); // 10は AcceptableOperationTimeSpan 内
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 時計を進める
    nnt::time::util::DoUserClockOperation(&currentSnapshot, 100);

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
        &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
        &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 適当に時間経過(ペナルティ明ける)
    nnt::time::util::AddElapsed(&currentSnapshot, 1000LL);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
        &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 時計を戻して MaxPenaltyTimeSpan をうける
    nnt::time::util::DoUserClockOperation(&currentSnapshot, -300);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
        nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
        &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nnt::time::util::AddElapsed(&currentSnapshot, 200);

    // 一度 MaxPenaltyTimeSpan をうけているので前回と同じタイミングでも利益得られる
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // 自動補正ON
    nnt::time::util::EnableAutomaticCorrection(&currentSnapshot);
    // 適当に時間経過
    nnt::time::util::AddElapsed(&currentSnapshot, 1000LL);

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
        nn::time::PeriodicBenefitClaimResult_Success, 0,
        &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    nn::time::PosixTime backupNetTime = currentSnapshot.GetStandardNetworkSystemClockPosixTime();

    {
        // RTCリセット、その後ネットワーク自動補正
        nnt::time::util::DoRtcReset(&currentSnapshot);
        nnt::time::util::DoAutomaticCorrection(&currentSnapshot, backupNetTime.value + 10);

        // RTCリセット後だけど、前後で自動補正ON、ネットワーク時計有効なのでペナルティでない
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        // 適当に時間経過
        nnt::time::util::AddElapsed(&currentSnapshot, 1000LL);

        // 時計操作する(RTCリセット後も、前後で自動補正ON/ネットワーク時計有効であれば時計操作を検知できる)
        nnt::time::util::DoUserClockOperation(&currentSnapshot, 100);
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        // ペナルティ明ける
        nnt::time::util::AddElapsed(&currentSnapshot, 200LL);

        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        // isSameTiming==trueでも利益を得られる
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }

    backupNetTime = currentSnapshot.GetStandardNetworkSystemClockPosixTime();

    {
        // RTCリセット(以前得た利益からの経過時間は分かる)
        nnt::time::util::DoRtcReset(&currentSnapshot);
        nnt::time::util::EnableAutomaticCorrection(&currentSnapshot);
        nnt::time::util::DoAutomaticCorrection(&currentSnapshot, backupNetTime.value + 100); // 自動補正ON(ネットワーク時計で100秒経過したことにする)
        nnt::time::util::DoUserClockOperation(&currentSnapshot, 0); // 自動補正OFF

        // 以前利益を得た瞬間から MaxPenaltyTimeSpan 発生
        // 利益を得てから 100 経過しているので、 MaxPenaltyTimeSpan - 100 のペナルティ中
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan - nn::TimeSpan::FromSeconds(100),
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan - nn::TimeSpan::FromSeconds(100),
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }

    {
        // RTCリセット(ネットワーク時計は未補正のまま)
        nnt::time::util::DoRtcReset(&currentSnapshot);

        // 今から MaxPenaltyTimeSpan かかる
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        // MaxPenaltyTimeSpan 継続中
        nnt::time::util::AddElapsed(&currentSnapshot, 10LL);
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan - nn::TimeSpan::FromSeconds(10),
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Penalty, MaxPenaltyTimeSpan - nn::TimeSpan::FromSeconds(10),
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        // MaxPenaltyTimeSpan明ける
        nnt::time::util::AddElapsed(&currentSnapshot, 190LL);
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

        // MaxPenaltyTimeSpan明けで、利益を受け取れるタミングがきた(前回と同じタイミングでも利益を得られるはず)
        NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<true, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));
    }
} // NOLINT(impl/function_size)

// ユーザー時計の自動補正フラグ変更タイミングがペナルティ開始となるかどうか
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock12)
{

    {
        // ネットワーク時計補正済
        NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        const nn::time::PosixTime value = {1520487966LL};
        NNT_ASSERT_RESULT_SUCCESS(nn::time::SetStandardNetworkSystemClockCurrentTime(value));
        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(false); // 自動補正OFF

    {
        // ユーザー時計をいじる.ネットワーク時計とは大きく異なる値にする.
        const nn::time::PosixTime value = {1234567890LL};
        nn::time::SetStandardLocalSystemClockCurrentTime(value);
    }

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // ネットワーク時計補正タイミングと、自動補正ONタイミングを適当にずらす
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(true); // 自動補正ON

    // 自動補正フラグをONにしたことによりユーザー時計が操作される
    // ONにしたタイミングからペナルティが開始されるはず
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nn::TimeSpan penalty = MaxPenaltyTimeSpan;
    {
        // SetStandardUserSystemClockAutomaticCorrectionEnabled(true)してから秒単位が変わっている可能性があるので
        // ペナルティを厳密に計算する

        nn::time::SteadyClockTimePoint updatedTime;
        nn::time::detail::GetStandardUserSystemClockAutomaticCorrectionUpdatedTime(&updatedTime);
        int64_t elapsed;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::GetSpanBetween(&elapsed, updatedTime, currentSnapshot.GetStandardSteadyClockTimePoint()));

        ASSERT_GE(elapsed, 0LL);

        penalty -= nn::TimeSpan::FromSeconds(elapsed);
    }

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_Penalty, penalty,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

} // NOLINT(impl/function_size)

// ユーザー時計の自動補正フラグ変更タイミングがペナルティ開始となるかどうか
// 自動補正フラグ変更後に、ネットワーク時計が補正されても、ペナルティ開始がフラグ変更～となるかどうかのてスト
TEST_F(ClaimPeriodicBenefitTest, ClaimPeriodicBenefitWithUserSystemClock13)
{
    {
        // ネットワーク時計補正済
        NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        const nn::time::PosixTime value = {1520487966LL};
        NNT_ASSERT_RESULT_SUCCESS(nn::time::SetStandardNetworkSystemClockCurrentTime(value));
        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(false); // 自動補正OFF

    {
        // ユーザー時計をいじる.ネットワーク時計とは大きく異なる値にする.
        const nn::time::PosixTime value = {1234567890LL};
        nn::time::SetStandardLocalSystemClockCurrentTime(value);
    }

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;
    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    // ネットワーク時計補正タイミングと、自動補正ONタイミングを適当にずらす
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(true); // 自動補正ON

    { // ClaimPeriodicBenefitWithUserSystemClock12 のテストとここのスコープのみが異なる

        // 自動補正フラグを操作した後にネットワーク時計が補正されるケース.
        // この場合でも、自動補正をONにした瞬間がペナルティ開始タイミングとなるはず.

        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
        NNT_ASSERT_RESULT_SUCCESS(nn::time::InitializeForSystem());

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));

        const nn::time::PosixTime value = {1520487966LL + 5LL};
        NNT_ASSERT_RESULT_SUCCESS(nn::time::SetStandardNetworkSystemClockCurrentTime(value));

        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
        NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    }

    // 自動補正フラグをONにしたことによりユーザー時計が操作される
    // ONにしたタイミングからペナルティが開始されるはず
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nn::TimeSpan penalty = MaxPenaltyTimeSpan;
    {
        // SetStandardUserSystemClockAutomaticCorrectionEnabled(true)してから秒単位が変わっている可能性があるので
        // ペナルティを厳密に計算する

        nn::time::SteadyClockTimePoint updatedTime;
        nn::time::detail::GetStandardUserSystemClockAutomaticCorrectionUpdatedTime(&updatedTime);
        int64_t elapsed;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::GetSpanBetween(&elapsed, updatedTime, currentSnapshot.GetStandardSteadyClockTimePoint()));

        ASSERT_GE(elapsed, 0LL);

        penalty -= nn::TimeSpan::FromSeconds(elapsed);
    }

    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_Penalty, penalty,
            &handover, currentSnapshot, MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

} // NOLINT(impl/function_size)

// PeriodicBenefitClaimContext::GetLastReceivedStandardUserSystemClockCalendarTime() のテスト
TEST_F(ClaimPeriodicBenefitTest, LastBenefitReceived)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

    const auto MaxPenaltyTimeSpan = nn::TimeSpan::FromSeconds(200);
    const auto AcceptableOperationTimeSpan = nn::TimeSpan::FromSeconds(10);

    nn::time::PeriodicBenefitClaimContext handover;

    nn::time::CalendarTime last;

    // まだ受け取ったことがない
    ASSERT_FALSE(nn::time::detail::HasLastBenefitReceivedInfo(handover)); // 内部APIもテストしとく
    ASSERT_FALSE(handover.GetLastReceivedStandardUserSystemClockCalendarTime(&last));

    nn::time::ClockSnapshot currentSnapshot;
    nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);

    nnt::time::util::EnableAutomaticCorrection(&currentSnapshot); // 自動補正ON
    nnt::time::util::AdjustNetworkClock(&currentSnapshot, 1495257519LL); // ネットワーク時計補正済

    // 初回
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, false>(
            nn::time::PeriodicBenefitClaimResult_NotReceivableTiming, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    ASSERT_FALSE(handover.GetLastReceivedStandardUserSystemClockCalendarTime(&last));

    // 利益受け取り
    nnt::time::util::AddElapsed(&currentSnapshot, 100);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    ASSERT_TRUE(handover.GetLastReceivedStandardUserSystemClockCalendarTime(&last));
    EXPECT_EQ(currentSnapshot.GetStandardUserSystemClockCalendarTime(), last);

    // 2度目で更新されるかどうか
    const auto BackupCalendarTime = last;
    nnt::time::util::AddElapsed(&currentSnapshot, 1000);
    NNT_TIME_ASSERT_NO_FAILURE((TestClaimPeriodicBenefit<false, true>(
            nn::time::PeriodicBenefitClaimResult_Success, 0,
            &handover, currentSnapshot,MaxPenaltyTimeSpan, AcceptableOperationTimeSpan)));

    ASSERT_TRUE(handover.GetLastReceivedStandardUserSystemClockCalendarTime(&last));
    EXPECT_EQ(currentSnapshot.GetStandardUserSystemClockCalendarTime(), last);
    EXPECT_NE(currentSnapshot.GetStandardUserSystemClockCalendarTime(), BackupCalendarTime); // 変わってるはず
}
