﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Log.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_Result.h>
#include <nn/os/os_Event.h>

#include <cstdlib>
#include <cstring>

#include <nn/tics/tics_Api.h>

#include "Plug.h"
#include "CrossBar.h"
#include "HotBridgeAPI.h"
#include "BridgeAPIInternal.h"

#include "../../Common/testHtc_BridgeAddressManager.h"
#include "../../Common/testHtc_FsLibraryAllocator.h"
#include "testHtc_EchoPlug.h"

/*
    libnn_htc.lib を利用して実装されたエコーサーバ。Flat API を使用する。
    1つのクライアントと通信した後、通信が切れるとプログラムも終了する。
*/

const char* g_BridgeName = "local_bridge";

namespace {
    // チャンネル数が変化したときのコールバック関数
    int ChannelConfigCallBack(void *ctx, ::tics::EOperation op, const char *name, bool HIO, int id, ::tics::UpdateChannelData *chanData);

    class TargetBridgeInstance : public ::tics::BridgeInstance
    {
    public:
        explicit TargetBridgeInstance( ::tics::BridgeHandle bridgeHandle, const char* tcpParameter );

        void OnConfigReset();
        int OnReleaseRequest( const ReleaseRequest* pRequest, size_t length );

    private:
        ::tics::BridgeHandle m_BridgeHandle;
    };


    int ChannelConfigCallBack(void *ctx, ::tics::EOperation op, const char *name, bool HIO, int id, ::tics::UpdateChannelData *chanData)
    {
        NN_LOG("*** ChannelConfigCallBack() Started\n");
        TargetBridgeInstance* pBridge = static_cast<TargetBridgeInstance*>( ctx );

        NN_LOG("*** ::tics::portability::stl::string()\n");
        ::tics::portability::stl::string sessionId;

        // SessionId == ChannelName
        sessionId.clear();
        sessionId += name;

        // EchoPlug を登録
        NN_LOG("*** EchoPlug()\n");
        ::EchoPlug* pEchoPlug = new ::EchoPlug();

        NN_LOG("*** ::tics::CrossBar()\n");
        ::tics::CrossBar* pCrossBar = pBridge->GetCrossBar();

        NN_LOG("*** ::tics::CrossBar::RegisterForSessionStart()\n");
        int intResult = pCrossBar->RegisterForSessionStart(sessionId, pEchoPlug);
        if( 0 != intResult )
        {
            NN_ABORT( "RegisterForSessionStart() failed. Result: 0x%08x.\n", intResult );
        }

        return 0;
    }


    TargetBridgeInstance::TargetBridgeInstance( ::tics::BridgeHandle bridgeHandle, const char* tcpParameter )
        : ::tics::BridgeInstance( bridgeHandle, true, tcpParameter ), m_BridgeHandle( bridgeHandle )
    {
    }

    void TargetBridgeInstance::OnConfigReset() // overriding from BridgeInstance
    {
        //  API_GUIDE_STEP:  4: here it is okay to call any of the flat APIs (high level),  or the Mux APIs (low-level)
        NN_LOG("*** Prepare HBNotifications\n");
        ::tics::BridgeNotifications  notifications;
        memset( &notifications, 0, sizeof(notifications) );
        notifications.OnChannelChange = ChannelConfigCallBack;

        NN_LOG("*** ::tics::HBRegister()\n");
        int intResult = ::tics::BridgeRegister( m_BridgeHandle, &notifications, this );
        if( 0 != intResult )
            {
            NN_ABORT( "RegisterForSessionStart() failed. Result: 0x%08x.\n", intResult );
        }
    }

    int TargetBridgeInstance::OnReleaseRequest( const ReleaseRequest* pRequest, size_t length )
    {
        return 0;
    }
}


extern "C" void nninitStartup()
{
}

extern "C" void nnMain()
{
    // メモリヒープ確保
    NN_LOG("*** nn::os::SetMemoryHeapSize()\n");
    size_t heapSize = 16 * 1024 * 1024; // 16MB
    nn::Result result = nn::os::SetMemoryHeapSize( heapSize );
    NN_ABORT_UNLESS( result.IsSuccess(), NN_TEXT("メモリヒープの確保に失敗しました。") );

    uintptr_t heapAddress;
    result = nn::os::AllocateMemoryBlock( &heapAddress, heapSize );
    NN_ABORT_UNLESS( result.IsSuccess(), NN_TEXT("メモリヒープの確保に失敗しました。") );

    // Siglo 初期化処理
    NN_LOG("*** nn::tics::Initialize()\n");
    nn::tics::Initialize(heapAddress, heapSize);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nnt::htc::detail::InitializeFsLibraryAllocator(); // for bsp0
#endif

    // Bridge の IP アドレスを取得
    BridgeAddressManager bridgeAddressManager;

    char bridgeAddress[20];
    bridgeAddressManager.GetAddress(bridgeAddress, sizeof(bridgeAddress));

    char bridgePort[8];
    bridgeAddressManager.GetPort(bridgePort, sizeof(bridgePort));

    char bridgeTcpParameter[32];
    strcpy(bridgeTcpParameter, bridgeAddress);
    strcat(bridgeTcpParameter, ":");
    strcat(bridgeTcpParameter, bridgePort);

    //  API_GUIDE_STEP:  0: Create Bridge Handle
    NN_LOG("*** ::tics::CreateBridgeHandleByName()\n");
    ::tics::BridgeHandle hBridge = ::tics::BridgeHandleCreateByName( g_BridgeName );
    if( nullptr == hBridge )
    {
        NN_ABORT( "BridgeHandleCreateByName() failed.\n" );
    }

    //  API_GUIDE_STEP:  1: Save this hBridge somewhere - you will need it later

    //  API_GUIDE_STEP:  2: Create Bridge Instance to allow it to run (note that this need bridge handle)
    NN_LOG("*** ::tics::VTargetBridgeInstance::VTargetBridgeInstance()\n");
    ::tics::BridgeInstance* pBridge = new ::TargetBridgeInstance(hBridge, bridgeTcpParameter);

    //  API_GUIDE_STEP:  3: Create Bridge Instance to allow it to run
    NN_LOG("*** ::tics::VTargetBridgeInstance::Run()\n");
    pBridge->Run();

    NN_LOG("*** ::tics::CrossBar::Run()\n");
    ::tics::CrossBar* pCrossBar = pBridge->GetCrossBar();
    pCrossBar->Run();

    NN_LOG("*** nn::tics::Finalize()\n");
    nn::tics::Finalize();

    NN_LOG("*** Finish!\n");
}
