﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <Tasks/OsMemoryTask.h>
#include <FileUtility.h>
#include <nn/nn_Log.h>
#include <nn/os/os_MemoryAttribute.h>

namespace nnt { namespace abuse {
    OsMemoryTask::OsMemoryTask(const String& typeName, const String& instanceName) :
        BaseTask(typeName, instanceName),
        m_threadArgs(nullptr),
        m_threads(nullptr),
        m_stacks(nullptr),
        m_memoryMutex(),
        m_heapSize(64 * MB),
        m_bytesAllocated(0),
        m_numThreads(1),
        m_maxThreads(64),
        m_numActualThreads(0),
        m_priority(nn::os::DefaultThreadPriority)
    {}

    OsMemoryTask::~OsMemoryTask()
    {}

    InitStatus OsMemoryTask::Initialize(const String& params)
    {
        ArgVector args;
        FileUtility::ParamsToArgs(params, args);

        for(ScriptArg& arg : args)
        {
            if(arg.argName == "MaxThreads")
                FileUtility::TryParseInt(arg, 1, 64, &m_maxThreads);
            else if(arg.argName == "Priority")
                FileUtility::TryParseInt(arg, nn::os::HighestThreadPriority, nn::os::LowestThreadPriority, &m_priority);
        }

        nn::os::InitializeMutex(&m_memoryMutex, false, 0);

        m_threads = (nn::os::ThreadType*) Platform::Allocate(sizeof(nn::os::ThreadType) * m_maxThreads);

        if(!m_threads)
        {
            LogError("Init Error - could not allocate threads\n");
            return INIT_ERROR;
        }

        for(int i = 0; i < m_maxThreads; ++i)
            ::new (&m_threads[i]) nn::os::ThreadType;

        LogInfo("Created thread array at %p\n", m_threads);
        m_stacks = (void**)Platform::Allocate(sizeof(void*) * m_maxThreads);

        if(!m_stacks)
        {
            LogError("Init Error - could not allocate stack array\n");
            Platform::Free(m_threads);
            m_threads = nullptr;
            return INIT_ERROR;
        }
        int i = 0;
        for(; i < m_maxThreads; ++i)
        {
            m_stacks[i] = Platform::AllocateAligned(4096, nn::os::StackRegionAlignment);

            if(!m_stacks[i])
                break;
        }

        if(i != m_maxThreads)
        {
            LogError("Init Error - could not allocate thread stacks\n");
            for(int j = 0; j <i; ++j)
            {
                Platform::Free(m_stacks[j]);
            }
            Platform::Free(m_stacks);
            Platform::Free(m_threads);
            m_threads = nullptr;
            m_stacks = nullptr;
            return INIT_ERROR;
        }

        m_threadArgs = (MemoryThreadArg*)Platform::Allocate( m_maxThreads * sizeof(MemoryThreadArg));

        if(!m_threadArgs)
            return INIT_ERROR;

        return INIT_OK;
    }

    StartStatus OsMemoryTask::Start()
    {
        if(!Abuse::AcquireResource(dynamic_cast<BaseTask*>(this), RESOURCE_HEAP_SIZE))
            return START_RESOURCE_DENIED;


        LogInfo("Heap size: 0x%x numThreads=%d\n", m_heapSize, m_numThreads);

        m_numThreads = m_numThreads << 1;

        if(m_numThreads > m_maxThreads)
            m_numThreads = 1;

        if(m_numThreads == 1)
        {
            m_heapSize += 64 * MB;
            if(m_heapSize + Platform::GetHeapSize() > (512 * MB))
                m_heapSize = 64  * MB;
        }


        if(nn::os::SetMemoryHeapSize(m_heapSize + Platform::GetHeapSize()).IsFailure())
        {
            LogWarning("Failed setting heapsize to 0x%x. Cancelling run\n", m_heapSize + Platform::GetHeapSize());
            return START_RESOURCE_DENIED;
        }

        return START_OK;
    }

    RunStatus OsMemoryTask::Run()
    {
        LogVerbose("++Run\n");

        m_numActualThreads = 0;
        int stepSize = 2 * MB;
        int numSteps = m_heapSize / stepSize;
        int start = 1;
        int end = numSteps;
        for(int i = 0; i < m_maxThreads; ++i)
        {
            m_threadArgs[i].task = this;
            m_threadArgs[i].stepSize = stepSize;
            m_threadArgs[i].startSize = start*stepSize;
            m_threadArgs[i].endSize   = end*stepSize;

            ++end;

            if(end > numSteps)
            {
                end = 2;
            }

            if(! CreateThread(&m_threads[i], workerFunc, &m_threadArgs[i], m_stacks[i], 4096, m_priority) )
                break;
            ++m_numActualThreads;
        }

        if(m_numActualThreads != m_maxThreads)
        {
            LogWarning("Could not create %d threads. Continuing this run with %d threads instead.\n", m_maxThreads, m_numActualThreads);
        }

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::StartThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::WaitThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
        {
            DestroyThread(&m_threads[i]);
        }

        LogVerbose("--Run\n");
        return RUN_OK;
    }

    StopStatus OsMemoryTask::Stop()
    {
        Abuse::ReleaseResource(dynamic_cast<BaseTask*>(this), RESOURCE_HEAP_SIZE);
        return STOP_OK;
    }

    ShutdownStatus OsMemoryTask::Shutdown()
    {
        if(m_threads)
            Platform::Free(m_threads);

        if(m_stacks)
        {
            for(int i = 0; i < m_maxThreads; ++i)
                Platform::Free(m_stacks[i]);
            Platform::Free(m_stacks);
        }

        if(m_threadArgs)
            Platform::Free(m_threadArgs);

        nn::os::FinalizeMutex(&m_memoryMutex);
        return SHUTDOWN_OK;
    }

    const char* OsMemoryTask::GetParamOptions()
    {
        return "MaxThreads=[1 - 64, default = 64] - maximum number of threads to run\n\
\tPriority=[0 - 31, default = 16] - priority of threads";
    }

    nn::Result OsMemoryTask::lockedAllocate(uintptr_t* addr, int size)
    {
        nn::os::LockMutex(&m_memoryMutex);
        nn::Result result = nn::os::AllocateMemoryBlock(addr, size);
        m_bytesAllocated += size;
        nn::os::UnlockMutex(&m_memoryMutex);
        return result;
    }

    void OsMemoryTask::lockedFree(uintptr_t addr, int size)
    {
        nn::os::LockMutex(&m_memoryMutex);
        nn::os::FreeMemoryBlock(addr, size);
        m_bytesAllocated -= size;
        nn::os::UnlockMutex(&m_memoryMutex);
    }

    void OsMemoryTask::workerFunc(void* vBarrierThreadArg)
    {
        MemoryThreadArg* arg = (MemoryThreadArg*) vBarrierThreadArg;
        for(int i = arg->startSize; i < arg->endSize; i += arg->stepSize)
        {
            uintptr_t addr = 0;

            while(arg->task->lockedAllocate(&addr, i).IsFailure())
            {
                nn::os::YieldThread();

                if(addr != 0)
                {
                    arg->task->LogWarning("allocate failed but addr=%p\n", addr);
                }
            }
            if(i % arg->stepSize == 0)
                nn::os::SetMemoryAttribute(addr, i, nn::os::MemoryAttribute_Normal);
            else
                nn::os::SetMemoryAttribute(addr, i, nn::os::MemoryAttribute_Uncached);

            memset((void*)addr, 'A', i);
            arg->task->lockedFree(addr, i);
            nn::os::YieldThread();
        }
    }

}    }
