﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <new>
#include <Tasks/NetworkTransferTask.h>
#include <Platform.h>
#include <Utils/md5.h>
#include <Common/testServerPki.h>


namespace nnt
{
    namespace abuse
    {
        NetworkTransferTask::NetworkTransferTask(const String& typeName, const String& instanceName)
            : BaseTask(typeName, instanceName)
        {
            m_HttpsHelper = HttpsHelperForCtxImport();
            m_pBuf = nullptr;
            m_jitterType = JITTER_NONE;
            m_jitterCurrentValue = MIN_JITTER;
        }

        NetworkTransferTask::~NetworkTransferTask()
        {
        }

        InitStatus NetworkTransferTask::Initialize(const String& params)
        {
            // process the input parameters from the script file
            if (!ProcessParams(params))
            {
                return INIT_ERROR;
            }

            // initialize the network
            if (!InitializeNetworkSettings())
            {
                return INIT_ERROR;
            }

            return INIT_OK;
        }

        StartStatus NetworkTransferTask::Start()
        {
            return START_OK;
        }

        RunStatus NetworkTransferTask::Run()
        {
            uint32_t    fileSize = 0;
            float       fAverageThroughput = 0.0f;

            // upload data
            for (uint32_t iUpload = 1; iUpload <= uploadOptions.timesToUploadOrDownload; iUpload++)
            {
                LogInfo(" * Starting upload number: %d\n", iUpload);
                UploadFile(uploadOptions);
            }

            // make sure we don't overload the server
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

            // download file
            if (!GetFileSize(downloadOptions, &fileSize))
            {
                LogError("Failed to get file size!\n");
                return RUN_ERROR;
            }

            for (uint32_t iDownload = 1; iDownload <= downloadOptions.timesToUploadOrDownload; iDownload++)
            {
                LogInfo(" * Starting download number: %d\n", iDownload);
                fAverageThroughput += DownloadFile(iDownload, downloadOptions, fileSize);
            }

            LogInfo("times to download: %d\n", downloadOptions.timesToUploadOrDownload);
            fAverageThroughput /= downloadOptions.timesToUploadOrDownload;

            LogInfo("\n******* Results *******\n");
            LogInfo("******* URL: %s\n", downloadOptions.url.c_str());
            LogInfo("******* Average Throughput: %.3f Mbits/sec\n", fAverageThroughput);

            return RUN_OK;
        }

        StopStatus NetworkTransferTask::Stop()
        {
            curl_global_cleanup();
            nn::nifm::CancelNetworkRequest();

            return STOP_OK;
        }

        ShutdownStatus NetworkTransferTask::Shutdown()
        {
            nn::ssl::SslContextId contextId;
            nn::Result result;

            /* cleanup SSL */
            delete[] m_pBuf;
            m_pBuf = nullptr;
            result = m_SslContext.GetContextId(&contextId);
            if (result.IsSuccess() && (contextId != 0))
            {
                result = m_SslContext.RemovePki(m_serverCertStoreId);
                m_SslContext.Destroy();
            }
            m_HttpsHelper.Finalize();

            /* cleanup libcurl library */
            curl_global_cleanup();

            /* this application no longer requires use of network interface */
            nn::nifm::CancelNetworkRequest();

            return SHUTDOWN_OK;
        }

        const char* NetworkTransferTask::GetParamOptions()
        {
            return "";
        }


        bool NetworkTransferTask::ProcessParams(const String& params)
        {
            // process the input parameters from the script file
            // eg. PARAMS="upload http 3,download http 2"
            std::string standardParams("");
            int tokenStartIndex = 0;
            int index = 0;
            std::string currentToken("");

            // convert params to standard string
            char * buffer = reinterpret_cast<char *>(Platform::Allocate(params.length() - 2));
            if (!buffer)
            {
                LogError("Memory allocation failed for param proccessing!");
                return false;
            }
            // remove the "" around the parameter value
            memcpy(buffer, &(params.c_str()[1]), params.length() - 2);
            buffer[params.length() - 2] = '\0';

            standardParams = buffer;

            // parse the params based on the delimiter ','
            while (tokenStartIndex < standardParams.length())
            {
                index = tokenStartIndex;
                while (index < standardParams.length())
                {
                    if (standardParams[index] == ',')
                    {
                        break;
                    }
                    index++;
                }
                currentToken = standardParams.substr(tokenStartIndex, index - tokenStartIndex);
                AssignParamValues(currentToken);
                tokenStartIndex = index + 1;
            }

            Platform::Free(buffer);
            return true;
        }

        bool NetworkTransferTask::AssignParamValues(std::string currentToken)
        {
            enum InputParamType commandType = INPUT_NONE;
            size_t currentSpace = 0;
            size_t currentStartIndex = 0;
            std::string currentWord("");
            bool fFinished = false;
            std::string httpsUrl("https");

            // parse the token string to find the list of commands and the associated values
            // sample list of commands:
            //    upload http 3
            //    download https 1
            //    jitter random
            //    jitter cycle
            while (!fFinished)
            {
                currentSpace = currentToken.find(" ", currentStartIndex);
                if (currentSpace != std::string::npos)
                {
                    currentWord = currentToken.substr(currentStartIndex, currentSpace - currentStartIndex);

                    if (currentWord.compare("upload") == 0)
                    {
                        commandType = UPLOAD;
                    }
                    else if (currentWord.compare("download") == 0)
                    {
                        commandType = DOWNLOAD;
                    }
                    else if (currentWord.compare("https") == 0)
                    {
                        // update the URL from http to https
                        if (commandType == UPLOAD)
                        {
                            uploadOptions.url.replace(0, 4, httpsUrl);
                        }
                        else if (commandType == DOWNLOAD)
                        {
                            downloadOptions.url.replace(0, 4, httpsUrl);
                        }
                    }
                    else if (currentWord.compare("jitter") == 0)
                    {
                        commandType = JITTER;
                    }

                    // move to the next word
                    currentStartIndex = currentSpace + 1;
                }
                else
                {
                    // process the last word in the command
                    currentWord = currentToken.substr(currentStartIndex, std::string::npos);

                    if (currentWord.length() > 0)
                    {
                        if (currentWord.compare("random") == 0)
                        {
                            m_jitterType = RANDOM;
                        }
                        else if (currentWord.compare("cycle") == 0)
                        {
                            m_jitterType = CYCLE;
                        }
                        else
                        {
                            // check if it's an integer value
                            std::string::size_type sz;
                            int currentWordInt = std::stoi(currentWord, &sz);
                            if (currentWordInt > 0)
                            {
                                // update the times to upload or download
                                if (commandType == UPLOAD)
                                {
                                    uploadOptions.timesToUploadOrDownload = currentWordInt;
                                }
                                else if (commandType == DOWNLOAD)
                                {
                                    downloadOptions.timesToUploadOrDownload = currentWordInt;
                                }
                            }
                        }
                    }

                    fFinished = true;
                }
            }

            return true;
        }


        // initialize all network settings
        bool  NetworkTransferTask::InitializeNetworkSettings()
        {
            CURLcode res = CURLE_OK;
            nn::Result result;

            /* since the following will utilize the system network interface, we must initialize
            network interface manager (NIFM) */
            nn::nifm::Initialize();

            // for DNS
            /*
            result = nn::nifm::SetExclusiveClient(nn::nifm::GetClientId());
            if (result.IsFailure())
            {
               LogInfo(" * Error: Failed to set exclusive client for nn::nifm. Desc: %d\n\n", result.GetDescription());
                return false;
            }
            */

            m_pNifmConnection = new nn::nifm::NetworkConnection;
            if (nullptr == m_pNifmConnection)
            {
                LogInfo("Failed to allocate nn::nifm::NetworkConnection.\n");
                return false;
            }

            nn::nifm::RequestHandle requestHandle = m_pNifmConnection->GetRequestHandle();
            nn::nifm::SetRequestConnectionConfirmationOption(requestHandle, nn::nifm::ConnectionConfirmationOption_Prohibited);
            nn::nifm::SetRequestPersistent(requestHandle, true);

            // Submit asynchronous request to NIFM
            m_pNifmConnection->SubmitRequest();

            // Wait while NIFM brings up the interface
            while (!m_pNifmConnection->IsAvailable())
            {
                LogInfo("Waiting for network interface to become available...\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            // SSL library is initialized only when curl_global_init() is called
            res = curl_global_init(CURL_GLOBAL_DEFAULT);
            if (res != CURLE_OK)
            {
                LogError("\n[ERROR] curl_global_init failed for CURL_GLOBAL_DEFAULT. Err: %d\n\n", res);
                nn::nifm::CancelNetworkRequest();
                return false;
            }
            res = curl_global_init(CURL_GLOBAL_SSL);
            if (res != CURLE_OK)
            {
                LogError("\n[ERROR] curl_global_init failed for CURL_GLOBAL_SSL. Err: %d\n\n", res);
                nn::nifm::CancelNetworkRequest();
                return false;
            }

            /* Setup for HTTPS. The SSL context will be allocated inside g_HttpsHelper.
            The allocated SSL context will be imported into libcurl by g_HttpsHelper.ImportSslContext
            below after CURL handle is created, and when URL contains https.
            The allocated SSL context will be freed in g_HttpsHelper.Finalize after all curl handles
            which imported this context are closed at the bottom.
            */
            if (m_HttpsHelper.Initialize() < 0)
            {
                LogError("[ERROR] Failed to initialize HTTPS helper util.\n");
                nn::nifm::CancelNetworkRequest();
                return false;
            }

            result = m_SslContext.Create(nn::ssl::Context::SslVersion_Auto);

            result = m_SslContext.ImportServerPki(
                &m_serverCertStoreId,
                TEST_ROOT_CA_CERT,
                sizeof(TEST_ROOT_CA_CERT),
                nn::ssl::CertificateFormat_Pem);

            if (result.IsFailure())
            {
                return -1; // error out here
            }

            return true;
        }   // NOLINT(impl/function_size)


        size_t NetworkTransferTask::CurlWriteFunction(char *data, size_t blobsize, size_t blobcount, void *userdata)
        {
            size_t count = blobsize*blobcount;
            return count;
        }

        size_t NetworkTransferTask::CurlSetSockOpts(void *pData, curl_socket_t curlfd, curlsocktype purpose)
        {
            int     rval = 0;
            Options *pOptions = static_cast<Options *>(pData);

            if (nullptr != pOptions)
            {
                if (-1 != pOptions->sndBufSize)
                {
                    rval = nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &pOptions->sndBufSize, sizeof(pOptions->sndBufSize));
                    if (rval != 0)
                    {
                        // LogInfo(" * nn::socket::SetSockOpt on So_SndBuf failed with error: %d\n\n", nn::socket::GetLastError());

                        return CURL_SOCKOPT_ERROR;
                    }
                    else
                    {
                        // LogInfo(" * nn::socket::SetSockOpt So_SndBuf set to: %d\n", pOptions->sndBufSize);
                    }
                }

                if (-1 != pOptions->rcvBufSize)
                {
                    rval = nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &pOptions->rcvBufSize, sizeof(pOptions->rcvBufSize));
                    if (rval != 0)
                    {
                        return CURL_SOCKOPT_ERROR;
                    }
                    else
                    {
                        // LogInfo(" * nn::socket::SetSockOpt So_RcvBuf set to: %d\n", pOptions->rcvBufSize);
                    }
                }

                rval = nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, &pOptions->disableTcpNagle, sizeof(pOptions->disableTcpNagle));
                if (rval != 0)
                {
                    // LogInfo(" * nn::socket::SetSockOpt on Tcp_NoDelay failed with error: %d\n\n", nn::socket::GetLastError());

                    return CURL_SOCKOPT_ERROR;
                }
            }

            return CURL_SOCKOPT_OK;
        }

        // get the file size from the URL
        bool NetworkTransferTask::GetFileSize(const Options &options, uint32_t* pSize)
        {
            bool    bRet = false;
            CURL*   curl = curl_easy_init();

            if (nullptr != curl)
            {
                char pErrorBuffer[CURL_ERROR_SIZE] = { 0 };

                LogInfo("url: %s\n", options.url.c_str());
                curl_easy_setopt(curl, CURLOPT_URL, options.url.c_str());
                curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
                curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
                curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, pErrorBuffer);
                curl_easy_setopt(curl, CURLOPT_SOCKOPTFUNCTION, CurlSetSockOpts);
                curl_easy_setopt(curl, CURLOPT_SOCKOPTDATA, &options);
                curl_easy_setopt(curl, CURLOPT_NOBODY, 1);
                curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, false);

                // check if the URL is using HTTPS
                if (HttpsHelperBase::IsUrlHttps(options.url.c_str()) == true)
                {
                    // Import the SSL context into libcurl. Because there's only one SSL context
                    // maintained in httpsHelper, same SSL context will be shared between CURL
                    // handles.

                    if (m_HttpsHelper.ImportSslContext(curl) < 0)
                    {
                        LogError("[ERROR] Failed to import the SSL context.\n");
                        goto cleanup;
                    }
                }

                CURLcode rval = curl_easy_perform(curl);
                if (rval == CURLE_OK)
                {
                    double dSize = 0.0;
                    if (curl_easy_getinfo(curl, CURLINFO_CONTENT_LENGTH_DOWNLOAD, &dSize) == CURLE_OK)
                    {
                        if (dSize < 0)
                        {
                            LogError(" * Unknown content length!\n");
                        }
                        else
                        {
                            *pSize = static_cast<uint32_t>(dSize);
                            LogInfo("\n*******Got File size! %u\n\n", *pSize);
                            bRet = true;
                        }
                    }
                    else
                    {
                        LogError("Failed to get content length\n");
                    }
                }
                else
                {
                    LogError("ERROR: Error fetching URL: %s. Error: %d\n", options.url.c_str(), rval);
                    LogInfo("Details: %s\n", pErrorBuffer);
                    if (HttpsHelperBase::IsUrlHttps(options.url.c_str()) == true)
                    {
                        m_HttpsHelper.PrintErrorMessage(curl, rval);
                    }
                }
            }
            else
            {
                LogError(" * Failed to create curl handle!\n");
            }

        cleanup:
            if (true == bRet)
            {
                curl_easy_cleanup(curl);
            }

            return bRet;
        }


        // download data
        float NetworkTransferTask::DownloadFile(uint32_t index, const Options &options, uint32_t fileSize)
        {
            float           fThroughput = 0.0f;
            nn::os::Tick    tickPrev = nn::os::Tick();
            nn::os::Tick    tickNow = nn::os::Tick();
            uint32_t        downloadSize = 0;
            uint32_t        chunk = 0;
            uint32_t        remainder = 0;
            uint32_t        previous = 0;
            float           fSeconds = 0.0f;
            int             iJitterValue = 0;

            uint32_t        curlEasyHandleCount = 0;
            CURLM*          curlMultiHandle = NULL;

            CURLMcode       status = CURLM_OK;
            int             stillRunning = 0;

            chunk = fileSize / options.numConnections;
            remainder = fileSize % options.numConnections;

            EasyData *curlEasyHandles = static_cast<EasyData *>(std::calloc(options.numConnections, sizeof(EasyData)));

            if (nullptr == curlEasyHandles)
            {
                LogError("[ERROR] Failed to allocate %d curl easy handles.\n", options.numConnections);
                return 0.0f;
            }

            for (curlEasyHandleCount = 0;
                curlEasyHandleCount < options.numConnections;
                curlEasyHandleCount++)
            {
                CURL *curlEasyHandle = curl_easy_init();
                if (nullptr == curlEasyHandle)
                {
                    LogError("ERROR: Could not create curl easy handle for url:%s", options.url.c_str());
                    goto cleanup;
                }

                curlEasyHandles[curlEasyHandleCount].easyHandle = curlEasyHandle;
                memset(curlEasyHandles[curlEasyHandleCount].pErrorBuffer, 0, CURL_ERROR_SIZE);

                curl_easy_setopt(curlEasyHandle, CURLOPT_URL, options.url.c_str());
                curl_easy_setopt(curlEasyHandle, CURLOPT_VERBOSE, 1L);
                curl_easy_setopt(curlEasyHandle, CURLOPT_FOLLOWLOCATION, 1);
                curl_easy_setopt(curlEasyHandle, CURLOPT_ERRORBUFFER, curlEasyHandles[curlEasyHandleCount].pErrorBuffer);
                curl_easy_setopt(curlEasyHandle, CURLOPT_WRITEFUNCTION, CurlWriteFunction);
                curl_easy_setopt(curlEasyHandle, CURLOPT_NOBODY, 0);
                curl_easy_setopt(curlEasyHandle, CURLOPT_SSL_VERIFYPEER, false);
                curl_easy_setopt(curlEasyHandle, CURLOPT_SOCKOPTFUNCTION, CurlSetSockOpts);
                curl_easy_setopt(curlEasyHandle, CURLOPT_SOCKOPTDATA, &options);

                // check if there is a jitter requirement
                iJitterValue = GetCurrentJitterValue();
                curl_easy_setopt(curlEasyHandle, CURLOPT_MAX_RECV_SPEED_LARGE, (curl_off_t)iJitterValue);
                LogInfo("Downloading %d bytes to %s with speed of %d\n", downloadSize, options.url.c_str(), iJitterValue);

                // check if URL is using HTTPS
                if (HttpsHelperBase::IsUrlHttps(options.url.c_str()) == true)
                {
                    // Import the SSL context into libcurl. Because there's only one SSL context
                    // maintained in g_HttpsHelper, same SSL context will be shared between CURL
                    // handles.
                    if (m_HttpsHelper.ImportSslContext(curlEasyHandle) < 0)
                    {
                        LogError("[ERROR] Failed to import the SSL context.\n");
                        goto cleanup;
                    }
                }

                if (options.numConnections > 1)
                {
                    char        pBuffer[64 + 1] = { '\0' };
                    uint32_t    increment = 0;

                    if (remainder > 0)
                    {
                        increment = chunk + 1;
                        --remainder;
                    }
                    else
                    {
                        increment = chunk;
                    }

                    // snprintf() isn't supported in Visual Studio 2013 so we can "safely" use sprintf() since we
                    // know the maximum length string that would ever be put into pBuffer using 32-bit values is 22
                    sprintf(pBuffer, "%u-%u", previous, previous + increment - 1);
                    curl_easy_setopt(curlEasyHandle, CURLOPT_RANGE, pBuffer);

                    previous += increment;

                    LogInfo("Chunk: %u  %s\n", chunk, pBuffer);
                }

            }

            // init a multi stack
            curlMultiHandle = curl_multi_init();
            if (nullptr == curlMultiHandle)
            {
                LogError("ERROR: Could not create libcurl multi handle.");
                goto cleanup;
            }

            // add the individual transfers
            for (uint32_t i = 0; i < curlEasyHandleCount; i++)
            {
                curl_multi_add_handle(curlMultiHandle, curlEasyHandles[i].easyHandle);
            }

            tickPrev = nn::os::GetSystemTick();
            LogInfo("Downloading...\n");
            do
            {
                status = curl_multi_perform(curlMultiHandle, &stillRunning);
            } while (status == CURLM_CALL_MULTI_PERFORM);

            LogInfo("Status: %u\n", status);

            while (stillRunning)
            {
                nn::socket::TimeVal timeout = { 0, 0 };

                nn::socket::FdSet  fdread = {};
                nn::socket::FdSet  fdwrite = {};
                nn::socket::FdSet  fdexcep = {};
                int     maxfd = -1;
                long    curl_timeo = -1;

                nn::socket::FdSetZero(&fdread);
                nn::socket::FdSetZero(&fdwrite);
                nn::socket::FdSetZero(&fdexcep);

                // set a suitable timeout to play around with
                timeout.tv_sec = 1;
                timeout.tv_usec = 0;

                curl_multi_timeout(curlMultiHandle, &curl_timeo);
                if (curl_timeo >= 0)
                {
                    timeout.tv_sec = curl_timeo / 1000;
                    if (timeout.tv_sec > 1)
                    {
                        timeout.tv_sec = 1;
                    }
                    else
                    {
                        timeout.tv_usec = (curl_timeo % 1000) * 1000;
                    }
                }

                //Get file descriptors from the transfers
                curl_multi_fdset(curlMultiHandle, &fdread, &fdwrite, &fdexcep, &maxfd);

                int rc = nn::socket::Select(maxfd + 1, &fdread, &fdwrite, &fdexcep, &timeout);

                switch (rc)
                {
                case nn::socket::SocketError:
                    // select error
                    stillRunning = 0;
                    LogError("ERROR: select() returned error %d\n", nn::socket::GetLastError());
                    break;
                case 0:
                default:
                    // timeout or readable/writable sockets
                    do
                    {
                        status = curl_multi_perform(curlMultiHandle, &stillRunning);
                    } while (status == CURLM_CALL_MULTI_PERFORM);
                    stillRunning = 0;
                    break;
                }
            }

            tickNow = nn::os::GetSystemTick();

            for (uint32_t iEasyHandle = 0; iEasyHandle < options.numConnections; ++iEasyHandle)
            {
                long    responseCode = 0;
                double  val = 0.0f;

                CURLcode ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_RESPONSE_CODE, &responseCode);
                if (ret != CURLE_OK)
                {
                    LogInfo("curl_easy_getinfo failed: %d\n", ret);
                }
                else
                {
                    LogInfo("CURLINFO_RESPONSE_CODE: %d\n", responseCode);
                }

                ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_SIZE_DOWNLOAD, &val);
                if (ret != CURLE_OK)
                {
                    LogInfo("curl_easy_getinfo failed: %d\n", ret);
                }
                else
                {
                    uint32_t numBytes = static_cast<uint32_t>(val);
                    downloadSize += numBytes;
                    LogInfo("CURLINFO_SIZE_DOWNLOAD: %u\n", numBytes);
                }

                /* check for total download time */
                ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_TOTAL_TIME, &val);
                if ((CURLE_OK == ret) && (val > 0))
                {
                    LogInfo("CURLINFO_TOTAL_TIME: %0.3f sec.\n", val);
                }

                /* check for average download speed */
                ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_SPEED_DOWNLOAD, &val);
                if ((CURLE_OK == ret) && (val > 0))
                {
                    LogInfo("CURLINFO_SPEED_DOWNLOAD: %0.3f kbyte/sec.\n", val / 1024);
                }

                /* check for name resolution time */
                ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_NAMELOOKUP_TIME, &val);
                if ((CURLE_OK == ret) && (val > 0))
                {
                    LogInfo("CURLINFO_NAMELOOKUP_TIME: %0.3f sec.\n", val);
                }

                /* check for connect time */
                ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_CONNECT_TIME, &val);
                if ((CURLE_OK == ret) && (val > 0))
                {
                    LogInfo("CURLINFO_CONNECT_TIME: %0.3f sec.\n", val);
                }

                LogInfo("Error Details %d: %s\n", iEasyHandle, curlEasyHandles[iEasyHandle].pErrorBuffer);
            }

            fSeconds = nn::os::ConvertToTimeSpan(tickNow - tickPrev).GetMilliSeconds() / 1000.0f;

            LogInfo("\n ** Finished download %u with Status: %d\n", index, status);

            LogInfo(" ** Duration: %.3f sec\n", fSeconds);

            if (downloadSize / kGigabyte >= 1.0f)
            {
                LogInfo(" ** Downloaded: %.3f GB\n", downloadSize / kGigabyte);
            }
            else if (downloadSize / kMegabyte >= 1.0f)
            {
                LogInfo(" ** Downloaded: %.3f MB\n", downloadSize / kMegabyte);
            }
            else if (downloadSize / kKilobyte >= 1.0f)
            {
                LogInfo(" ** Downloaded: %.3f KB\n", downloadSize / kKilobyte);
            }
            else
            {
                LogInfo(" ** Downloaded: %u Bytes\n", downloadSize);
            }

            /* bytes/ms -> Mb/s */
            fThroughput = downloadSize * 8.0f / fSeconds / kMegabyte;
            LogInfo(" ** Speed: %.3f Mbits/sec\n\n", fThroughput);

        cleanup:

            for (uint32_t i = 0; i < curlEasyHandleCount; i++)
            {
                if (curlMultiHandle)
                {
                    curl_multi_remove_handle(curlMultiHandle, curlEasyHandles[i].easyHandle);
                }

                curl_easy_cleanup(curlEasyHandles[i].easyHandle);
            }

            if (curlMultiHandle)
            {
                curl_multi_cleanup(curlMultiHandle);
            }

            free(curlEasyHandles);

            return fThroughput;
        }   // NOLINT(impl/function_size)


        // curl read
        size_t NetworkTransferTask::CurlReadFunction(void *ptr, size_t size, size_t nmemb, void *userp)
        {
            WriteData   *pData = (WriteData *)userp;
            size_t      count = size * nmemb;

            if (count < 1)
                return 0;

            if (pData->sizeLeft > 0)
            {
                size_t readsize = std::min(count, (size_t)pData->sizeLeft);
                memcpy(ptr, pData->pReadPos, readsize);
                pData->pReadPos += readsize;
                pData->sizeLeft -= readsize;
                return readsize;
            }

            return 0;                          /* no more data left to deliver */
        }


        // upload a file
        float NetworkTransferTask::UploadFile(const NetworkOptions &options)
        {
            float           fThroughput = 0.0f;
            nn::os::Tick    tickPrev = nn::os::Tick();
            nn::os::Tick    tickNow = nn::os::Tick();
            char            *pBuf = static_cast<char *>(std::calloc(options.uploadSize, 1));
            WriteData       writeData = { 0 };
            CURLcode        res = CURLE_OK;
            float           fSeconds = 0.0f;
            int             iJitterValue = 0;


            // fill buffer with randomish bytes
            memset(pBuf, rand(), options.uploadSize);

            writeData.pReadPos = pBuf;
            writeData.sizeLeft = options.uploadSize;

            if (nullptr == pBuf)
            {
                LogError("[ERROR] Failed to allocate %d bytes for upload.\n", options.uploadSize);
                return 0.0f;
            }

            CURL *curlEasyHandle = curl_easy_init();

            // check if the URL is using HTTPS
            if (HttpsHelperBase::IsUrlHttps(options.url.c_str()) == true)
            {
                // Import the SSL context into libcurl. Because there's only one SSL context
                // maintained in httpsHelper, same SSL context will be shared between CURL
                // handles.

                if (m_HttpsHelper.ImportSslContext(curlEasyHandle) < 0)
                {
                    LogError("[ERROR] Failed to import the SSL context.\n");
                    return 0;
                }
            }


            if (nullptr != curlEasyHandle)
            {
                // fill out the form for CURLFORM_STREAM
                struct curl_httppost* postForm = NULL;
                struct curl_httppost* lastForm = NULL;
                curl_formadd(&postForm, &lastForm,
                    CURLFORM_COPYNAME, UPLOAD_NAME,
                    CURLFORM_BUFFER, UPLOAD_FILENAME,
                    CURLFORM_BUFFERPTR, pBuf,
                    CURLFORM_BUFFERLENGTH, options.uploadSize,
                    CURLFORM_CONTENTTYPE, UPLOAD_CONTENTTYPE,
                    CURLFORM_END);

                curl_easy_setopt(curlEasyHandle, CURLOPT_URL, options.url.c_str());
                curl_easy_setopt(curlEasyHandle, CURLOPT_VERBOSE, 1L);
                curl_easy_setopt(curlEasyHandle, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_0);
                curl_easy_setopt(curlEasyHandle, CURLOPT_HTTPPOST, postForm);
                curl_easy_setopt(curlEasyHandle, CURLOPT_READFUNCTION, CurlReadFunction);
                curl_easy_setopt(curlEasyHandle, CURLOPT_POSTFIELDSIZE, options.uploadSize);
                curl_easy_setopt(curlEasyHandle, CURLOPT_SSL_VERIFYPEER, 0);
                curl_easy_setopt(curlEasyHandle, CURLOPT_SSL_VERIFYHOST, 0);

                // check if there is a jitter requirement
                iJitterValue = GetCurrentJitterValue();
                curl_easy_setopt(curlEasyHandle, CURLOPT_MAX_SEND_SPEED_LARGE, (curl_off_t)iJitterValue);
                LogInfo("Uploading %d bytes to %s with speed of %d\n", options.uploadSize, options.url.c_str(), iJitterValue);

                tickPrev = nn::os::GetSystemTick();

                res = curl_easy_perform(curlEasyHandle);
                if (CURLE_OK == res)
                {
                    long    responseCode = 0;
                    double  val = 0.0f;

                    tickNow = nn::os::GetSystemTick();

                    LogInfo("curl_easy_perform() succeeded!\n", res);

                    CURLcode ret = curl_easy_getinfo(curlEasyHandle, CURLINFO_RESPONSE_CODE, &responseCode);
                    if (ret != CURLE_OK)
                    {
                        LogInfo("curl_easy_getinfo failed: %d\n", ret);
                    }
                    else
                    {
                        LogInfo("CURLINFO_RESPONSE_CODE: %d\n", responseCode);
                    }

                    ret = curl_easy_getinfo(curlEasyHandle, CURLINFO_HTTP_CONNECTCODE, &responseCode);
                    if (ret != CURLE_OK)
                    {
                        LogInfo("curl_easy_getinfo failed: %d\n", ret);
                    }
                    else
                    {
                        LogInfo("CURLINFO_HTTP_CONNECTCODE: %d\n", responseCode);
                    }

                    /* check for total download time */
                    ret = curl_easy_getinfo(curlEasyHandle, CURLINFO_TOTAL_TIME, &val);
                    if ((CURLE_OK == ret) && (val > 0))
                    {
                        LogInfo("CURLINFO_TOTAL_TIME: %0.3f sec.\n", val);
                    }

                    fSeconds = nn::os::ConvertToTimeSpan(tickNow - tickPrev).GetMilliSeconds() / 1000.0f;

                    LogInfo("\n ** Finished upload\n");

                    LogInfo(" ** Duration: %.3f sec\n", fSeconds);

                    /* bytes/ms -> Mb/s */
                    fThroughput = options.uploadSize * 8.0f / fSeconds / kMegabyte;
                    LogInfo(" ** Speed: %.3f Mbits/sec\n\n", fThroughput);
                }
                else
                {
                    LogInfo("[ERROR] curl_easy_perform() failed! curl error: %d\n", res);
                }

                curl_formfree(postForm);
                curl_easy_cleanup(curlEasyHandle);
            }

            free(pBuf);

            return fThroughput;
        }   // NOLINT(impl/function_size)


        // return the delay value for this iteration of the download or upload
        int NetworkTransferTask::GetCurrentJitterValue()
        {
            if (m_jitterType == RANDOM)
            {
                m_jitterCurrentValue = rand() % MIN_JITTER + MAX_JITTER;
                return m_jitterCurrentValue;
            }
            else if (m_jitterType == CYCLE)
            {
                m_jitterCurrentValue = m_jitterCurrentValue - STEP_JITTER;
                if (m_jitterCurrentValue < MAX_JITTER)
                {
                    // start over again
                    m_jitterCurrentValue = MIN_JITTER;
                }
            }

            return 0;
        }
    }
}
