﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <Tasks/LogReaderTask.h>
#include <nn/nn_Log.h>

namespace nnt {
    namespace abuse {


        LogReaderTask::LogReaderTask(const String& typeName, const String& instanceName) : BaseTask(typeName, instanceName),
            m_readFilenames(),
            m_readFiles(),
            m_readDir(""),
            m_doneReadingLogs(false),
            m_doneProcessingFiles(false)
        {
            nn::os::InitializeMutex(&s_LogMutex, false, 0);
        }

        LogReaderTask::~LogReaderTask()
        {}

        InitStatus LogReaderTask::Initialize(const String& params)
        {
            (void)params;

            nn::os::LockMutex(&s_LogMutex);

            if (m_doneReadingLogs)
            {
                return INIT_OK; // No work to be done here.
            }

            m_readDir = Platform::GetLogDirectory();

            // The directory that we get is something like "Logs/TK1_19" , we need to go up a level.
            int i = m_readDir.length();
            while ( --i > 0 )
            {
                if (m_readDir[i] == '/')
                {
                    m_readDir[i] = '\0';
                    break;
                }
            }

            // Get a list of every file under 'Logs', recursively.
            Platform::GetFilesInDir(m_readFilenames, m_readDir.c_str(), true);
            OpenOptions readOptions;
            readOptions.read = true;
            // Attempt to open all of our files.
            if (!sOpenFiles(m_readFilenames, m_readFiles, readOptions))
            {
                LogWarning("LogReader Resource Denied.\n");
                return INIT_ERROR;
            }

            // Iterate through all of our files, read them and write them out to the log, most recent files first.
            for (int file = m_readFiles.size() - 1; file >= 0; --file)
            {
                // This constant 0x5a00000000000000 is an artifact of the way our filesystem tells us a file is invalid: it fills the filename string pointer with '5A' pattern. Sadly,
                // this pattern isn't even consistent - we get many files that have a string pointer of '5a5a5a5a00000020' and similar. They all start with '5a', however.
                if (nullptr == m_readFilenames[file].c_str() || 0x5a00000000000000 < (unsigned long long)m_readFilenames[file].c_str())
                    continue;

                if (nullptr != strstr(m_readFilenames[file].c_str(), "AbuseMain")) // Don't re-read the main log or else our log size will double with each run.
                    continue;

                // Emit starting marker
                LogVerbose("\nLogReader reading vvFILEvv[%s].\n", m_readFilenames[file].c_str());
                FlushLog(false /*crashing*/);
#undef min
                int size = Platform::GetFileSize(m_readFiles[file]);
                int chunk = (int)std::min((int64_t)size, (int64_t)LOG_READER_BUFFER_SIZE);
                for (int i = 0; i < size; i += chunk)
                {
                    Platform::FileRead(m_readFiles[file], i, m_buffer, chunk);
                    m_buffer[LOG_READER_BUFFER_SIZE] = '\0'; // Null terminate this buffer before we try to print.
                    LogInfo("%s", m_buffer);
                    FlushLog(false /*crashing*/);
                }
                memset(m_buffer, 0, (size_t)LOG_READER_BUFFER_SIZE); // Clear the buffer before reading in a new file.

                // Emit ending marker.
                LogVerbose("\nLogReader done reading ^^FILE^^[%s].\n", m_readFilenames[file].c_str());
                FlushLog(false /*crashing*/);
            }

            m_doneReadingLogs = true;
            nn::os::UnlockMutex(&s_LogMutex);
            return INIT_OK;
        }

        StartStatus LogReaderTask::Start()
        {
            return START_OK;
        }

        RunStatus LogReaderTask::Run()
        {
            return RUN_OK;
        }

        StopStatus LogReaderTask::Stop()
        {
            if (m_doneProcessingFiles) return STOP_OK;
            for (unsigned i = 0; i < m_readFiles.size(); ++i) {
                Platform::FileClose(m_readFiles[i]);
            }

            m_readFiles.clear();
            m_doneProcessingFiles = true;
            return STOP_OK;
        }

        ShutdownStatus LogReaderTask::Shutdown()
        {
            LogVerbose("LogReader Shutdown OK\n");
            return SHUTDOWN_OK;
        }

        const char* LogReaderTask::GetParamOptions()
        {
            return "";
        }

    }
}
