﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <vector>
#include <nn/fs.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_Debug.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <ctime>
#include <nn/mem/mem_StandardAllocator.h>
#include <cstdlib>
#include <nn/util/util_FormatString.h>
#include <Platform.h>
#include <AbuseString.h>
namespace nnt
{
    namespace abuse
    {
        uintptr_t sBlockAddress;
        static nn::os::Tick sPlatformStartTime;
        NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
       // nn::lmem::HeapHandle sHeap;

        int bytesAllocated = 0;

        struct FileTK1
        {
            nn::fs::FileHandle fileHandle;
            int64_t            fileOffset;
            String             fileName;
            int                fileOptions;
            bool               fileOpen;
        };

        //actual heap size is rounded to next multiple of nn::os::MemoryBlockUnitSize
        static const int HEAP_SIZE = 0x0C000000;
        static const int MAX_PATH_LENGTH  = 256;
        static const int FILE_SIZE = 1024;

        // scripts and logs are saved on the SD card
        const char SD_PREFIX[] = "sd";
        static char sWorkingDirectory[MAX_PATH_LENGTH + 1] = "/";
        static char sScriptDirectory[MAX_PATH_LENGTH + 1] = "Scripts/";
        static char sLogDirectory[MAX_PATH_LENGTH + 1] = "Logs/";
        static int  sLogDirectoryNum = 0;

        static nn::os::MutexType sFileGetFreeHandleMutex;
        static nn::os::MutexType sMemoryMutex;
        static nn::os::MutexType sFileOpenMutex;
        static std::vector<FileTK1, PlatformAllocator<FileTK1>> sFiles;

        static bool sSocketLibInitialized = false;
        int numOpenFiles = 0;

        static int sFindLogDirectoryNum()
        {
            nn::fs::DirectoryHandle dir;
            int dirNum = 0;
            char dirName[256];

            bool dirExists = true;
            while(dirExists)
            {
                nn::util::SNPrintf(dirName, 256, "%s%sMP_%d/", sWorkingDirectory, sLogDirectory, dirNum);
                nn::Result res = nn::fs::OpenDirectory(&dir, dirName, nn::fs::OpenDirectoryMode_Directory);

                dirExists = res.IsSuccess();

                if(dirExists)
                {
                    ++dirNum;
                    nn::fs::CloseDirectory(dir);
                }
            }

            return dirNum;
        }

        static inline size_t sRoundUp(size_t bytes, int roundTo)
        {
            return bytes + (roundTo - (bytes % roundTo));
        }

        void Platform::Initialize()
        {
            sPlatformStartTime = nn::os::GetSystemTick();

            //undefined reference to __nnOSClockGetTime (rynda 0.9.0)
            //time(&sPlatformStartTime);
            //sFormattedTime = *localtime(&sPlatformStartTime);

            nn::os::InitializeMutex(&sFileGetFreeHandleMutex, false, 0);
            nn::os::InitializeMutex(&sFileOpenMutex, false, 0);
            nn::os::InitializeMutex(&sMemoryMutex, false, 0);
            auto result = nn::os::SetMemoryHeapSize(sRoundUp(HEAP_SIZE, nn::os::MemoryBlockUnitSize));
            NN_LOG("Result: %d,%d\n", result.GetModule(), result.GetDescription());
           // nn::os::AllocateMemoryBlock(&sBlockAddress, sRoundUp(HEAP_SIZE, nn::os::MemoryBlockUnitSize));

           // memmove((void*)sBlockAddress, (void*)sBlockAddress, sRoundUp(HEAP_SIZE, nn::os::MemoryBlockUnitSize));

           // sHeap = nn::lmem::CreateExpHeap((void*)sBlockAddress, sRoundUp(HEAP_SIZE, nn::os::MemoryBlockUnitSize), -1);

            result = nn::fs::MountSdCardForDebug(SD_PREFIX);
            NN_ASSERT(result.IsSuccess());
        }

        void Platform::Finalize()
        {
            if(numOpenFiles > 0)
                NN_LOG("Warning: Closing with %d open files\n", numOpenFiles);
            for(size_t i = 0; i < sFiles.size(); ++i)
            {
                NN_LOG("closing %s\n", sFiles[i].fileName.c_str());
                FileClose((File)i);
            }

            nn::fs::Unmount(SD_PREFIX);

            nn::os::FinalizeMutex(&sFileGetFreeHandleMutex);
            nn::os::FinalizeMutex(&sFileOpenMutex);
            nn::os::FreeMemoryBlock(sBlockAddress, sRoundUp(HEAP_SIZE, nn::os::MemoryBlockUnitSize));

            //Can't finalize this mutex - a static nn::fs object calls Free after main exits.
            //nn::os::FinalizeMutex(&sMemoryMutex);
        }

        static inline bool sfileOpen(File file)
        {
            return file != FILE_INVALID_HANDLE &&  file < (int)sFiles.size() && sFiles[file].fileOpen;
        }


        File sgetFreeFile(const String& fileName, int fsOptions)
        {
            nn::os::LockMutex(&sFileGetFreeHandleMutex);

            File file = FILE_INVALID_HANDLE;
            for(size_t i = 0; i < sFiles.size(); ++i)
            {
                if(!sFiles[i].fileOpen)
                {
                    file =  i;
                    sFiles[file].fileName = fileName;
                    sFiles[file].fileOptions = fsOptions;
                    break;
                }
            }

            if(file == FILE_INVALID_HANDLE)
            {
                int oldCap = sFiles.capacity();
                sFiles.push_back(FileTK1());
                file = sFiles.size() - 1;
                sFiles[file].fileName = fileName;
                sFiles[file].fileOffset = 0;
                sFiles[file].fileOptions = fsOptions;

                NN_LOG("Adding %s, in sFiles %s\n", fileName.c_str(), sFiles[file].fileName.c_str());
                if(sFiles.capacity() > oldCap)
                {
                    for(int i = 0; i < sFiles.size(); ++i)
                        NN_LOG("Moved fileName: %s\n", sFiles[i].fileName.c_str());
                }

            }
            nn::os::UnlockMutex(&sFileGetFreeHandleMutex);
            return file;
        }

        const char* Platform::GetWorkingDirectory()
        {
            return sWorkingDirectory;
        }

        void Platform::SetWorkingDirectory(const char* directory)
        {
            nn::util::SNPrintf(sWorkingDirectory, 256, "%s:/%s/", SD_PREFIX, directory);
            sScriptDirectory[MAX_PATH_LENGTH] = '\0';
        }

        const char* Platform::GetScriptDirectory()
        {
            return sScriptDirectory;
        }

        void Platform::SetScriptDirectory(const char* scriptDirectory)
        {
            nn::util::SNPrintf(sScriptDirectory, 256, "%s/", scriptDirectory);
            sScriptDirectory[MAX_PATH_LENGTH] = '\0';
        }

        const char* Platform::GetLogDirectory()
        {
            return sLogDirectory;
        }

        void Platform::SetLogDirectory(const char* directory)
        {
            sLogDirectoryNum = sFindLogDirectoryNum();
            nn::util::SNPrintf(sLogDirectory, 256, "%s/MP_%d", directory, sLogDirectoryNum);
        }

        void Platform::GetFilesInDir(StringVector& vec, const char* dirName, bool includeSubDirs)
        {
            nn::fs::DirectoryHandle dirHandle;
            nn::Result result;

            char* dirPath = (char*)Platform::Allocate(MAX_PATH_LENGTH);
            nn::util::SNPrintf(dirPath, MAX_PATH_LENGTH, "%s%s", sWorkingDirectory, dirName);

            result = nn::fs::OpenDirectory(&dirHandle, dirPath, nn::fs::OpenDirectoryMode_All);

            if(result.IsSuccess())
            {
                int64_t numEntries = 1;
                nn::fs::DirectoryEntry* dirEntries= (nn::fs::DirectoryEntry*)Platform::Allocate(sizeof(nn::fs::DirectoryEntry) * 32);

                while(numEntries != 0)
                {
                    nn::fs::ReadDirectory(&numEntries, dirEntries, dirHandle, 32);

                    for(int i = 0; i < numEntries; ++i)
                    {
                        if(dirEntries[i].directoryEntryType == nn::fs::DirectoryEntryType_File)
                        {
                            nn::util::SNPrintf(dirPath, MAX_PATH_LENGTH, "%s/%s", dirName, dirEntries[i].name);
                            vec.push_back(dirPath);
                        }
                        else if(includeSubDirs)
                        {
                            nn::util::SNPrintf(dirPath, MAX_PATH_LENGTH, "%s/%s", dirName, dirEntries[i].name);
                            GetFilesInDir(vec, dirPath, true);
                        }
                    }
                }
                Platform::Free(dirEntries);
            }
            nn::fs::CloseDirectory(dirHandle);
            Platform::Free(dirPath);
        }

        File Platform::FileOpen(const char* fileName, OpenOptions openOptions, bool absolutePath)
        {
            nn::os::LockMutex(&sFileOpenMutex);

            int fsOptions = 0;

            if(openOptions.read)
                fsOptions |= nn::fs::OpenMode_Read;
            if(openOptions.write)
                fsOptions |= nn::fs::OpenMode_Write;
            fsOptions |= nn::fs::OpenMode_AllowAppend;

            char fullPath[MAX_PATH_LENGTH + 1];
            int length = 0;
            if(!absolutePath)
                length = nn::util::SNPrintf(fullPath, MAX_PATH_LENGTH, "%s%s", sWorkingDirectory, fileName);
            else
                length = nn::util::SNPrintf(fullPath, MAX_PATH_LENGTH, "%s:%s", SD_PREFIX, fileName);

            fullPath[length + 1] = '\0';
            File file = sgetFreeFile(fullPath, fsOptions);

            nn::Result result = nn::fs::OpenFile(&sFiles[file].fileHandle, fullPath,  fsOptions);

            if(result.IsSuccess())
            {
                nn::fs::GetFileSize(&sFiles[file].fileOffset, sFiles[file].fileHandle);
                nn::fs::CloseFile(sFiles[file].fileHandle);
                sFiles[file].fileOpen = true;
                ++numOpenFiles;
            }

            //attempt to create a file if open failed and create option is set
            if(result.IsFailure() && openOptions.create)
            {
                nn::fs::ResultPathNotFound pathNotFound;
                //File doesn't exist. Create parent directorues and file.
                if(result.GetDescription() == pathNotFound.GetDescription())
                {
                    int i = strlen(sWorkingDirectory);
                    char * buffer = (char*)Platform::Allocate(strlen(fullPath) + i);
                    memcpy(buffer, fullPath, i);

                    while(fullPath[i] != '\0')
                    {
                        while(fullPath[i] != '/' && fullPath[i] != '\0')
                        {
                            buffer[i] = fullPath[i];
                            ++i;
                        }

                        buffer[i] = 0;
                        if(fullPath[i] == '/')
                            result = nn::fs::CreateDirectory(buffer);
                        else
                        {
                            result = nn::fs::CreateFile(buffer, FILE_SIZE);
                            break;
                        }

                        buffer[i] = fullPath[i];
                        ++i;
                    }

                    if(result.IsSuccess())
                    {
                        nn::Result result2 = nn::fs::OpenFile(&sFiles[file].fileHandle, fullPath,  fsOptions);
                        if (result2.IsSuccess())
                        {
                            sFiles[file].fileOffset = 0;
                            sFiles[file].fileOpen = true;
                            ++numOpenFiles;
                        }
                    }
                    Platform::Free(buffer);
                }
            }

            nn::os::UnlockMutex(&sFileOpenMutex);
            return file;
        }

        void Platform::FileClose(File file)
        {
            nn::os::LockMutex(&sFileOpenMutex);
            if(sfileOpen(file))
            {
                sFiles[file].fileOpen = false;

                --numOpenFiles;
            }
            nn::os::UnlockMutex(&sFileOpenMutex);
        }

        void Platform::FileWrite(File file, const char* buffer, int bufferSize, bool append)
        {
            nn::os::LockMutex(&sFileOpenMutex);
            if(sfileOpen(file))
            {
                nn::fs::OpenFile(&sFiles[file].fileHandle, sFiles[file].fileName.c_str(),  sFiles[file].fileOptions);

                if(append)
                {
                    nn::fs::WriteFile(sFiles[file].fileHandle, sFiles[file].fileOffset, buffer, bufferSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                    sFiles[file].fileOffset += bufferSize;
                }
                else
                {
                    nn::fs::WriteFile(sFiles[file].fileHandle, 0, buffer, bufferSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                    if (bufferSize > sFiles[file].fileOffset)
                        sFiles[file].fileOffset = bufferSize;
                }

                nn::fs::CloseFile(sFiles[file].fileHandle);


                NN_LOG("Writing to %s bufferSize = %d\n", sFiles[file].fileName.c_str(), bufferSize);
            }
            nn::os::UnlockMutex(&sFileOpenMutex);
        }

        void Platform::FileRead(File file, int64_t offset, char* buffer, int bufferSize)
        {
            nn::os::LockMutex(&sFileOpenMutex);
            size_t length = 0;
            if(sfileOpen(file))
            {
                nn::fs::OpenFile(&sFiles[file].fileHandle, sFiles[file].fileName.c_str(),  sFiles[file].fileOptions);
                nn::fs::ReadFile(&length, sFiles[file].fileHandle, offset, (void*)buffer, bufferSize);
                nn::fs::CloseFile(sFiles[file].fileHandle);
            }
            nn::os::UnlockMutex(&sFileOpenMutex);
        }

        int64_t Platform::GetFileSize(File file)
        {
            if(sfileOpen(file))
            {
                return sFiles[file].fileOffset;
            }
            return 0;
        }

        const int Platform::GetMaxPathLength()
        {
            return MAX_PATH_LENGTH;
        }

        long Platform::GetAddrAny()
        {
            return nn::socket::InAddr_Any;
        }

        int Platform::InitializeSocketLib()
        {

            if(!sSocketLibInitialized)
            {
                nn::Result resultSocket;
                resultSocket = nn::socket::Initialize(
                                        g_SocketMemoryPoolBuffer,
                                        nn::socket::DefaultSocketMemoryPoolSize,
                                        nn::socket::MinSocketAllocatorSize,
                                        nn::socket::DefaultConcurrencyLimit);

                if(resultSocket.IsSuccess())
                    return 0;
                return -1;
            }
            return 0;
        }

        int Platform::FinalizeSocketLib()
        {
            if(sSocketLibInitialized)
            {
                sSocketLibInitialized = false;
                return nn::socket::Finalize().IsSuccess();
            }
            return 0;
        }

        int Platform::socket(nn::socket::Family family, nn::socket::Type type, nn::socket::Protocol protocol)
        {
            return nn::socket::Socket(family, type, protocol);
        }

      /*  int Platform::inet_pton(uint8_t family, const char* addressString, void* addressBuffer)
        {
            return ::inet_pton(family, addressString, addressBuffer);
        }*/

        int Platform::connect(int descriptor, const nn::socket::SockAddrIn* addr, int sizeBytes)
        {
            return nn::socket::Connect(descriptor, reinterpret_cast<const ::nn::socket::SockAddr*>(addr), sizeBytes);
        }
        nn::socket::Errno Platform::GetLastError()
        {
            return nn::socket::GetLastError();
        }

        int Platform::send(int descriptor, const void* buffer, size_t bufferByteSize, nn::socket::MsgFlag flags)
        {
            return nn::socket::Send(descriptor, buffer, bufferByteSize, flags);
        }

        int Platform::recv(int descriptor, void* buffer, size_t bufferByteSize, nn::socket::MsgFlag flags)
        {
            return nn::socket::Recv(descriptor, buffer, bufferByteSize, flags);
        }

        int Platform::sendto(int descriptor, void* buffer,size_t bufferByteSize, nn::socket::MsgFlag flags, const nn::socket::SockAddrIn* to, int toByteSize)
        {
            return nn::socket::SendTo(descriptor, buffer, bufferByteSize, flags, reinterpret_cast<const nn::socket::SockAddr*>(to), toByteSize);
        }

        int Platform::recvfrom(int descriptor, void* buffer,size_t bufferByteSize, nn::socket::MsgFlag flags, nn::socket::SockAddrIn* from, int* fromByteSize)
        {
            return nn::socket::RecvFrom(descriptor, buffer, bufferByteSize, flags, reinterpret_cast< nn::socket::SockAddr*>(from), (unsigned *)fromByteSize);
        }

        int Platform::shutdown(int descriptor, nn::socket::ShutdownMethod how)
        {
            return nn::socket::Shutdown(descriptor, how);
        }

        int Platform::close(int descriptor)
        {
            return nn::socket::Close(descriptor);
        }

        int Platform::poll(nn::socket::PollFd* descriptors, size_t fdCount, int timeout)
        {
            return nn::socket::Poll(descriptors, fdCount, timeout);
        }

        int Platform::bind(int descriptor, const nn::socket::SockAddr* addr, int addrSize)
        {
            return nn::socket::Bind(descriptor, addr, addrSize);
        }

        int Platform::listen(int descriptor, int backlogCount)
        {
            return nn::socket::Listen(descriptor, backlogCount);
        }

        int Platform::accept(int descriptor, nn::socket::SockAddr* addr, int* addrSize)
        {
            return nn::socket::Accept(descriptor, addr, (unsigned *)addrSize);
        }

        int Platform::SetNonBlocking(int descriptor, bool nonblocking)
        {
            if( nonblocking )
            {
                return nn::socket::Fcntl(descriptor, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock);
            }
            else
            {
                return nn::socket::Fcntl(descriptor, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::None);
            }
        }

        uint16_t Platform::htons(uint16_t hostshort)
        {
            return nn::socket::InetHtons(hostshort);
        }

        uint32_t Platform::htonl(uint32_t hostlong)
        {
            return nn::socket::InetHtonl(hostlong);
        }
        uint32_t Platform::ntohl(uint32_t netlong)
        {
            return nn::socket::InetNtohl(netlong);
        }

        uint64_t Platform::ntohll(uint64_t hostlonglong)
        {
            if(1 == nn::socket::InetNtohl(1))
                return hostlonglong;
            else
                return ((uint64_t)ntohl((hostlonglong) & 0xFFFFFFFF) << 32) | ntohl((hostlonglong) >> 32);
        }
        uint64_t Platform::htonll(uint64_t netlonglong)
        {
            if(1 == nn::socket::InetHtonl(1))
                return netlonglong;
            else
                return ((uint64_t)htonl((netlonglong) & 0xFFFFFFFF) << 32) | htonl((netlonglong) >> 32);
        }

        uint16_t Platform::PutNetworkBuffer(char* buffer, uint16_t start, unsigned data)
        {
            data = htonl(data);
            memcpy(buffer + start, &data, sizeof(unsigned));
            return start + sizeof(unsigned);
        }

        uint16_t Platform::PutNetworkBuffer(char* buffer, uint16_t start, uint64_t data)
        {
            data = htonll(data);
            memcpy(buffer + start, &data, sizeof(uint64_t));
            return start + sizeof(uint64_t);
        }

        static const int MEM_HEADER_SIZE = 2;

        void* Platform::AllocateAligned(size_t bytes, int alignment)
        {

            uintptr_t rawptr = (uintptr_t)malloc(bytes + alignment + MEM_HEADER_SIZE);
            rawptr += MEM_HEADER_SIZE;
            int mask = alignment - 1;
            int offset = alignment - (rawptr & (mask));

            uintptr_t alignedptr =  (rawptr + offset);
            *((short*)(alignedptr - 2)) = offset;
            //NN_LOG("alloc %p\n", alignedptr);
            return (void*)alignedptr;
        }

      /*  void* Platform::AllocateAligned(size_t bytes, int alignment)
        {
            nn::os::LockMutex(&sMemoryMutex);

            void* ptr = nn::lmem::AllocateFromExpHeap(sHeap, bytes, alignment);
            bytesAllocated += bytes;
            //NN_LOG("Alloc: BytesAllocated = %d\n", bytesAllocated);
            nn::os::UnlockMutex(&sMemoryMutex);
            return ptr;
        }*/

        void* Platform::Allocate(size_t bytes)
        {
            //nn::os::LockMutex(&sMemoryMutex);

            void* ptr =  AllocateAligned(bytes,4);
            //memset(ptr, 0, bytes);
            //NN_LOG("Alloc: BytesAllocated = %d\n", bytesAllocated);
            //nn::os::UnlockMutex(&sMemoryMutex);
            return ptr;
        }

        void*  Platform::Reallocate(void * ptr,size_t bytes)
        {

            if(bytes == 0)
            {
                if(ptr)
                {
                    Free(ptr);
                }
                return nullptr;
            }
            void * newBuffer = Allocate(bytes);

            if(!newBuffer)
                return nullptr;

            if(!ptr)
                return newBuffer;


            memcpy(newBuffer,ptr,bytes);

            Free(ptr);
            return newBuffer;
        }

        char* Platform::StringDuplicate(const char* string)
        {
           int buffsize = strlen(string) + 1;
           char * newString = (char *)Allocate(buffsize);

           if(!newString)
           {
               return nullptr;
           }

           memcpy(newString,string,buffsize);

           return newString;
        }

        void* Platform::ClearAllocate(size_t num,size_t size)
        {
            void * buff= Allocate(num*size);
            if(!buff)
            {
                return nullptr;
            }
            memset(buff,0,num*size);
            return buff;
        }

        void Platform::Free(void* ptr)
        {
           /* if(!ptr)
            {
                return;
            }

            nn::os::LockMutex(&sMemoryMutex);*/
            uintptr_t alignedptr = (uintptr_t)ptr;
            int offset = *(short*)(alignedptr - MEM_HEADER_SIZE);
            free((void*)(alignedptr - offset - MEM_HEADER_SIZE));
        /*    bytesAllocated = nn::lmem::GetTotalSize(sHeap) - nn::lmem::GetExpHeapTotalFreeSize(sHeap);
            //NN_LOG("Free:  BytesAllocated = %d\n", bytesAllocated);
            nn::os::UnlockMutex(&sMemoryMutex);*/
        }

        void Platform::Free(void* ptr, size_t size)
        {
            (void)size;
            Free(ptr);
        }

        size_t Platform::GetHeapSize()
        {
            return HEAP_SIZE;
        }

        int64_t Platform::GetStartTime()
        {
            return sPlatformStartTime.GetInt64Value();
        }

        int64_t Platform::GetSystemRunningTimeMilli()
        {
            return (GetPlatformTick() - sPlatformStartTime.GetInt64Value()) / (GetPlatformTickFrequency() / 1000);
        }

        int64_t Platform::GetPlatformTick()
        {
            return nn::os::GetSystemTick().GetInt64Value();
        }
        int64_t Platform::GetPlatformTickFrequency()
        {
            return nn::os::GetSystemTickFrequency();
        }
        double Platform::GetPlatformTimeSinceStart()
        {
            return (double)GetPlatformTick() / (double)GetPlatformTickFrequency();
        }
        const char* Platform::GetPlatformName()
        {
            return "MP";
        }

        const int Platform::GetNumCores()
        {
            return nnt::abuse::NUM_CORES - 1;
        }

    }
}
