﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include <FileUtility.h>

namespace nnt
{
    namespace abuse
    {
        void FileUtility::Initialize(const char* scriptDirectoryPath)
        {
            Platform::SetScriptDirectory(scriptDirectoryPath);
        }

        bool FileUtility::LoadScript(const String& scriptName, CommandVector& commands)
        {
            size_t pathLength = scriptName.length() + strlen(Platform::GetScriptDirectory()) + 2;
            char* scriptPath = (char*) Platform::Allocate( pathLength );
            int64_t offset = 0;
            int lineNum = 1;

            for(unsigned i = 0; i < commands.size(); ++i)
                Platform::Free(commands[i]);
            commands.clear();

            OpenOptions options;
            options.read = true;

            nn::util::SNPrintf(scriptPath, pathLength, "%s%s", Platform::GetScriptDirectory(), scriptName.c_str());

            bool parseSuccessful = false;
            File script = Platform::FileOpen(scriptPath, options);
            if(script != FILE_INVALID_HANDLE)
            {
                int fileSize = (int)Platform::GetFileSize(script);
                char *scriptBuffer = (char*)Platform::Allocate(fileSize);

                Platform::FileRead(script, offset, scriptBuffer, fileSize);

                parseSuccessful = parseBuffer(scriptBuffer, fileSize, lineNum, commands);

                Platform::FileClose(script);

                if(!parseSuccessful)
                {
                    for(unsigned i = 0; i < commands.size(); ++i)
                        Platform::Free(commands[i]);
                    commands.clear();
                }
                Platform::Free(scriptBuffer);
                Platform::Free(scriptPath);
            }
            return parseSuccessful;
        }

        BaseCommand* FileUtility::ParseCommand(const String& commandStr)
        {
            char* buffer = (char*)Platform::Allocate(commandStr.length() + 1);
            memcpy(buffer, commandStr.c_str(), commandStr.length() + 1);

            int index = IsValidCommand(commandStr.c_str());

            if(index != -1)
            {
                BaseCommand* command = validCommands[index].create();

                if(!command)
                    return nullptr;

                ArgVector args;
                size_t commandLength = strlen(validCommands[index].name);
                bool result = ParseArgs(buffer + commandLength + 1, (int)(commandStr.length() - (commandLength + 1)), args);

                CommandVector vec;
                command->ParseArgs(args, vec);

                if(!result)
                {
                    Platform::Free(command);
                    return nullptr;
                }
                return command;
            }
            return nullptr;
        }

        bool FileUtility::parseBuffer(char* buffer, int bufferSize, int& lineNum, CommandVector& commands)
        {
            int bytesParsed = 0;

            bufferSize = TrimBuffer(buffer, bufferSize);

            for(int i = 0; i <= bufferSize; ++i)
            {
                if(buffer[i] == '\n' || buffer[i] == '\0')
                {

                    buffer[i] = '\0';

                    ParseResult result = parseLine(buffer + bytesParsed, i - bytesParsed, lineNum, commands);

                    if(result == ParseResult::PARSE_UNMATCHED_COMMAND)
                        return false;

                    bytesParsed = i + 1;

                    if(lineNum != -1)
                        ++lineNum;
                }
            }

            //Make sure all STARTLOOPs have a matching ENDLOOP
            if(lineNum != -1)
            {
                for(BaseCommand* command : commands)
                {
                    if(command->GetCommandType() == COMMAND_START_LOOP)
                    {
                        CommandStartLoop* startLoop = (CommandStartLoop*) command;
                        if(startLoop->loopFinishedIndex == 0)
                            return false;
                    }
                }
            }
            return true;
        }

        int FileUtility::IsValidCommand(const char* line)
        {
            for(unsigned command = 0; command < GetNumRegisteredCommands(); ++command)
            {
                bool match = true;
                size_t commandLength = strlen(validCommands[command].name);

                for(size_t ch = 0; ch < commandLength; ++ch)
                {
                    if(line[ch] != validCommands[command].name[ch])
                    {
                        match = false;
                        break;
                    }
                }
                if(match)
                    return command;
            }

            return -1;
        }

        bool FileUtility::ParamsToArgs(const String& params, ArgVector& args)
        {
            args.clear();
            char* buffer = (char*)Platform::Allocate( params.length() + 1);
            strncpy(buffer, params.c_str(), params.length());
            buffer[params.length()] = '\0';

            int trimmedLength = FileUtility::TrimBuffer(buffer, params.length() + 1, true);
            return FileUtility::ParseArgs(buffer + 1, trimmedLength - 2, args);
        }

        ParseResult FileUtility::parseLine(char* line, int lineLength, int lineNum, CommandVector& commands)
        {
            //NN_LOG("%s\n", line);
            if(lineLength == 0)
                return ParseResult::PARSE_SUCCESS;

            int command = IsValidCommand(line);
            int commandLength = 0;

            if(command != -1)
                commandLength = (int)strlen(validCommands[command].name);

            if(command != -1 && (line[commandLength] == ' '|| line[commandLength] == '\0' || line[commandLength] == '\r' || line[commandLength] == '\n'))
            {
                ArgVector args;
                ParseArgs(line + commandLength + 1, lineLength - (commandLength + 1), args);

                BaseCommand* newCommand = validCommands[command].create();
                newCommand->m_commandIndex = command;
                newCommand->m_fileLine = lineNum;

                ParseResult result = newCommand->ParseArgs(args, commands);

                if(result == PARSE_SUCCESS)
                    commands.push_back(newCommand);

                else if(lineNum != -1)
                    NN_LOG("Error parsing args on line %d\n\t%s\n", lineNum, line);
                else
                    NN_LOG("Error parsing args from shell\n\t%s\n", line);

                return result;
            }

            if(lineNum != -1)
                NN_LOG("Warning: No matching command found on line %d:\n\t%s\n", lineNum, line);
            else
                NN_LOG("Warning: No matching command found for:\n\t%s\n", line);
            return ParseResult::PARSE_SUCCESS;

        }

        void sPutChar(char* buffer, char ch, int& index)
        {
            buffer[index] = ch;
            ++index;
        }

        int FileUtility::TrimBuffer(char* line, int lineLength, bool removeNewlines,bool removeQuotes)
        {
            int ch = 0;
            bool spaceFound = false;
            bool equalFound = false;
            bool quoteFound = false;
            bool commentFound = false;

            for(int i = 0; i < lineLength; ++i)
            {
                if(line[i] == '\n')
                {
                    if(!removeNewlines)
                        sPutChar(line, line[i], ch);
                    else
                        sPutChar(line, ' ', ch);
                    ++i;

                    //Ignore whitspace at beginning of line
                    while(line[i] == ' ' || line[i] == '\t')
                        ++i;

                    commentFound = false;
                }
                if(line[i] == '\r')
                {
                    continue;
                }
                if(commentFound)
                    continue;

                if(line[i] == '#')
                {
                    line[i] = '\0';
                    commentFound = true;
                    continue;
                }

                if(line[i] == '\"')
                {
                    equalFound = false;
                    spaceFound = false;
                    quoteFound = !quoteFound;

                    sPutChar(line, line[i], ch);
                    if(removeQuotes)
                    {
                        --ch;
                    }

                    continue;
                }
                if(quoteFound)
                {

                    sPutChar(line, line[i], ch);
                    continue;
                }

                if(line[i] == '\t')
                    line[i] = ' ';

                if(line[i] == '=')
                {
                    equalFound = true;

                    if(spaceFound)
                        --ch;

                    sPutChar(line, line[i], ch);
                    continue;
                }

                if(line[i] == ' ')
                {
                    if(!equalFound && !spaceFound)
                    {
                        spaceFound = true;
                        sPutChar(line, line[i], ch);
                    }
                }
                else
                {
                    spaceFound = false;
                    equalFound = false;
                    sPutChar(line, line[i], ch);
                }
            }

            line[ch] = '\0';
            return ch;
        }

        bool FileUtility::TryParseUnsigned(const ScriptArg& arg, unsigned min, unsigned max, unsigned* out)
        {
            unsigned tmp;
            bool parsed = ParseUnsigned(&tmp, arg.argValue);
            if(!parsed)
            {
                return false;
            }
            if(tmp < min || tmp > max)
            {
                return false;
            }
            *out = tmp;
            return true;
        }

        bool FileUtility::TryParseInt(const ScriptArg& arg, int min, int max, int* out)
        {
            int tmp;
            bool parsed = ParseInt(&tmp, arg.argValue);
            if(!parsed)
            {
                return false;
            }
            if(tmp < min || tmp> max)
            {
                return false;
            }
            *out=tmp;
            return true;
        }

        bool FileUtility::TryParseBool(const ScriptArg& arg, bool* out)
        {
            int tmp;
            bool result = TryParseInt(arg, 0, 1, &tmp);

            if(result)
                *out = tmp != 0;
            return result;
        }

        bool FileUtility::ParseArgs(char* line, int lineLength, ArgVector& args)
        {
            //By now, line has been trimmed and begins one character after the command word.
            //i.e. if the line in the file was "  WAIT  10 ", line is now "10"
            int anonCount = 0;
            int valueStartIndex = 0;
            int nameLength = 0;
            bool equalFound = false;
            bool quoteFound = false;
            ScriptArg curArg;

            if(lineLength <= 0)
                return true;

            for(int i = 0; i <= lineLength; ++i)
            {
                if(line[i] == '\"')
                {
                    quoteFound = true;
                    ++i;
                    while(line[i] != '\0' && line[i] != '"')
                    {
                        if(line[i] == '\0')
                            return false; //unmatched quote
                        ++i;
                    }

                    if(line[i] == '\"')
                        ++i;
                }

                if(line[i] == ' ' || line[i] == '\0' || quoteFound)
                {
                    line[i] = '\0';
                    curArg.argValue = line + valueStartIndex;

                    if(!equalFound)
                    {
                        char buffer[32];
                        int len = nn::util::SNPrintf(buffer, 32, "anonName_%d", anonCount);
                        buffer[len] ='\0';
                        curArg.argName = buffer;
                        ++anonCount;
                    }

                    if(equalFound || !(i == valueStartIndex) )
                        args.push_back(curArg);

                    equalFound = false;
                    quoteFound = false;
                    nameLength = 0;
                    valueStartIndex=i + 1;
                }
                else
                {
                    if(line[i] == '=')
                    {
                        line[i] = '\0';
                        curArg.argName = line + (i - nameLength);

                        equalFound = true;
                        valueStartIndex = i + 1;
                    }
                    else
                        ++nameLength;
                }
            }
            return true;
        }
    }
}
