﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include <Commands/CommandCommon.h>
#include <FileUtility.h>
namespace nnt
{
    namespace abuse
    {
        static unsigned* sGetIteratorPtr(const char* name, int nameLength, const CommandVector& commands)
        {
            if(commands.size() == 0)
                return nullptr;

            int j = (int)(commands.size() - 1);

            for(; j >= 0; --j)
            {
                BaseCommand* command = commands[j];
                if(command->GetCommandType() == COMMAND_START_LOOP)
                {
                    CommandStartLoop* startLoop = (CommandStartLoop*) command;
                    if(startLoop->itrName.length() == 0)
                        continue;

                    if(strncmp(name, startLoop->itrName.c_str(), nameLength) == 0)
                    {
                        return &startLoop->curItr;
                    }
                }
            }
            return nullptr;
        }

        const unsigned GetNumRegisteredCommands()
        {
            return sizeof(validCommands) / sizeof(RegisteredCommand);
        }

        bool ParseInt(int* parsedInt, const String& buffer)
        {
            char* end;
            *parsedInt = (int) strtol(buffer.c_str(), &end, 10);

            return end != buffer.c_str();
        }

        bool ParseUnsigned(unsigned* parsedInt, const String& buffer)
        {
            char* end;
            *parsedInt = (unsigned) strtol(buffer.c_str(), &end, 10);

            return end != buffer.c_str();
        }

        bool ParseIteratorValues(const String& name, const CommandVector& commands, IteratorVector& iterators, String& formattedName)
        {
            const char* cName = name.c_str();
            char* buffer = (char*)Platform::Allocate( name.length() );
            int labelStart = 0;
            bool parseError = false;
            int bufferIndex = 0;
            for(unsigned i = 0; i < name.length(); ++i)
            {
                if(cName[i] == '}')
                {
                    parseError = true;
                    break;
                }
                if(cName[i] != '{')
                {
                    buffer[bufferIndex] = cName[i];
                    ++bufferIndex;
                    continue;
                }

                labelStart = i + 1;

                while(cName[i] != '}')
                {
                    if(cName[i] == '\0')
                        parseError = true;
                    ++i;
                }

                //cName[i] == '}', cName[labelStart-1] = '{'
                buffer[bufferIndex] = '%';
                ++bufferIndex;

                if(commands.size() == 0)
                    return false;

                unsigned length = i - labelStart;

                unsigned* itrPtr = sGetIteratorPtr(cName + labelStart, length, commands);

                if(!itrPtr)
                {
                    parseError = true;
                    break;
                }
                iterators.push_back(itrPtr);
            }
            buffer[bufferIndex] = '\0';
            formattedName = buffer;
            Platform::Free(buffer);

            return !parseError;
        }

        unsigned* ParseIteratorValue(const String& name, const CommandVector& commands)
        {
            if(name[0] != '{' || name[name.length() - 1] != '}')
                return nullptr;

            return sGetIteratorPtr(name.c_str() + 1, name.length() - 2, commands);
        }

        void BuildFormattedName(String& formattedName, const IteratorVector& iterators, String& result)
        {
            size_t bufferSize = formattedName.length() + (iterators.size() * 12);
            char* buffer = (char*) Platform::Allocate(bufferSize);
            const char* name = formattedName.c_str();
            int bufferIndex = 0;
            int itrIndex = 0;
            for(unsigned i = 0; i < formattedName.length(); ++i)
            {
                if(name[i] != '%')
                {
                    buffer[bufferIndex] = name[i];
                    ++bufferIndex;
                }
                else
                {
                    bufferIndex += nn::util::SNPrintf(buffer + bufferIndex, bufferSize - bufferIndex, "%d", *iterators[itrIndex]);
                    ++itrIndex;
                }
            }
            buffer[bufferIndex] = '\0';
            result = buffer;
            Platform::Free(buffer);
        }
    }
}
