﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_FormatString.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <Abuse.h>
#include <Platform.h>
#include <Tasks/BaseTask.h>
#include <AbuseDriver.h>
#include <FileUtility.h>
#include <ShellServer.h>

namespace nnt
{
    namespace abuse
    {
        ListRegisteredTask        Abuse::s_registeredTasks;
        ListActiveTask            Abuse::s_activeTasks;
        CommandVector             Abuse::s_commands;
        Log*                      Abuse::s_log = nullptr;
        int*                      Abuse::s_tasksOnProcessor = nullptr;
        BaseTask*                 Abuse::s_resourceOwner[NUM_RESOURCES];
        int64_t*                  Abuse::s_heartbeats = nullptr;
        AbuseDriver*              Abuse::s_taskDrivers = nullptr;
        nn::os::MutexType         Abuse::s_taskMutex;
        nn::os::MutexType         Abuse::s_generalMutex;
        nn::os::MutexType         Abuse::s_coreMutex;
        nn::os::MutexType         Abuse::s_resourceMutex;
        nn::os::MutexType         Abuse::s_threadMutex;
        nn::os::BarrierType       Abuse::s_driverBarrier;
        nn::os::Tick              Abuse::s_startTime;
        String                    Abuse::s_activeScript;
        BaseCommand*              Abuse::s_shellCommand = nullptr;
        ShellServer*              Abuse::s_shell = nullptr;
        char*                      Abuse::s_scriptName = nullptr;
        void*                     Abuse::s_exceptionStack = nullptr;
        nn::os::UserExceptionInfo* Abuse::s_exceptionInfo = nullptr;
        int                       Abuse::s_numRunningDrivers = 0;
        unsigned                  Abuse::s_taskRuntimeMillis = 100;
        unsigned                  Abuse::s_timeoutMillis=10000;
        unsigned                  Abuse::s_commandIndex = 0;
        uint16_t                  Abuse::s_listenPort=8020;
        bool                      Abuse::s_isShellEnabled = false;
        bool                      Abuse::s_isRunning;
        bool                      Abuse::s_resourceOwned[NUM_RESOURCES];
        unsigned                  Abuse::s_cpuBusyCore = 3;
        int                       Abuse::s_cpuBusyThreadPriority = 19;
        unsigned                  Abuse::s_cpuBusyMs = 5000;
        unsigned                  Abuse::s_cpuBusyThreadDutyCyclePercent = 0;

        ScopedMutex::ScopedMutex(nn::os::MutexType* mutex)
            : m_mutex(mutex)
        {
            nn::os::LockMutex(m_mutex);
        }

        ScopedMutex::~ScopedMutex()
        {
            if(m_mutex)
                nn::os::UnlockMutex(m_mutex);
        }

        void hangFunc(void* arg)
        {
            while(arg);
        }

        void Abuse::Initialize(const String& configPath)
        {
            s_startTime = nn::os::GetSystemTick();
            nn::os::InitializeMutex(&s_taskMutex, false, 0);
            nn::os::InitializeMutex(&s_generalMutex,false,0);
            nn::os::InitializeMutex(&s_coreMutex, false, 0);
            nn::os::InitializeMutex(&s_resourceMutex, false, 0);
            nn::os::InitializeMutex(&s_threadMutex, false, 0);

            nn::os::InitializeBarrier(&s_driverBarrier, Platform::GetNumCores());

            s_exceptionStack = Platform::AllocateAligned(4096, 4096);
            s_exceptionInfo = (nn::os::UserExceptionInfo*) Platform::Allocate(sizeof(nn::os::UserExceptionInfo));
            //nn::os::SetUserExceptionHandler(exceptionHandler, (uintptr_t)s_exceptionStack, s_exceptionInfo);

            for(int i = 0; i < NUM_RESOURCES; ++i)
            {
                s_resourceOwned[i] = false;
                s_resourceOwner[i] = nullptr;
            }

            s_tasksOnProcessor = (int*)Platform::Allocate(sizeof(int) * Platform::GetNumCores());
            for (int i = 0; i <= Platform::GetNumCores(); i++)
            {
                s_tasksOnProcessor[i] = 0;
            }

            s_isRunning = true;

            s_scriptName = (char*)Platform::Allocate( Platform::GetMaxPathLength() + 1);
            s_scriptName[0] = '\0';

            s_heartbeats = (int64_t*) Platform::Allocate(nnt::abuse::NUM_CORES * sizeof(int64_t));
            s_timeoutMillis = 10000;

            OpenOptions options;
            options.read = true;
            File file = Platform::FileOpen(configPath.c_str(), options, true);

            if(file != FILE_INVALID_HANDLE)
            {
                int fileSize = (int)Platform::GetFileSize(file);
                char* buffer = (char*)Platform::Allocate( fileSize );
                Platform::FileRead(file, 0, buffer, fileSize);
                Platform::FileClose(file);
                fileSize = FileUtility::TrimBuffer(buffer, fileSize, true);

                ArgVector args;
                FileUtility::ParseArgs(buffer, fileSize, args);

                for(ScriptArg arg : args)
                {
                    if(arg.argName == "ScriptDir")
                        Platform::SetScriptDirectory(arg.argValue.c_str());
                    else if(arg.argName == "StartingScript")
                    {
                        strncpy(s_scriptName, arg.argValue.c_str(), arg.argValue.length());
                        s_scriptName[arg.argValue.length()] = '\0';
                    }
                    else if(arg.argName == "LogDir")
                        Platform::SetLogDirectory(arg.argValue.c_str());
                    else if(arg.argName == "WorkingDir")
                        Platform::SetWorkingDirectory(arg.argValue.c_str());
                    else if(arg.argName == "TimestampLogs")
                    {
                        unsigned enabled;
                        bool parsed = FileUtility::TryParseUnsigned(arg, 0, 1, &enabled);
                        if(parsed)
                            Log::ToggleGlobalTimesamps(enabled != 0);
                    }
                    else if(arg.argName == "DefaultVerbosity")
                    {
                        unsigned level;
                        bool parsed = FileUtility::TryParseUnsigned(arg, 0, NumVerbosityLevels - 1, &level);
                        if(parsed)
                        {
                            Log::SetGlobalVerbosity( (LogVerbosity)level);
                        }
                    }
                    else if(arg.argName=="ForwardToConsole")
                    {
                        unsigned forward;
                        bool parsed = FileUtility::TryParseUnsigned(arg, 0, 1, &forward);
                        if(parsed)
                            Log::SetGlobalForwardToConsole(forward != 0);
                    }
                    else if(arg.argName=="EnableShell")
                    {
                        unsigned enabled;
                        bool parsed = FileUtility::TryParseUnsigned(arg, 0, 1, &enabled);
                        if(parsed)
                            s_isShellEnabled= enabled != 0;
                    }
                    else if(arg.argName=="ListenPort")
                    {
                        unsigned port;
                        bool parsed = FileUtility::TryParseUnsigned(arg, 0, std::numeric_limits<uint16_t>::max(), &port);
                        if(parsed)
                            s_listenPort=(uint16_t)port;
                    }
                    else if(arg.argName=="CoreTimeoutMillis")
                        FileUtility::TryParseUnsigned(arg, 0, std::numeric_limits<unsigned>::max(), &s_timeoutMillis);
                    else if(arg.argName=="TaskRuntimeMillis")
                        FileUtility::TryParseUnsigned(arg, 1, std::numeric_limits<unsigned>::max(), &s_taskRuntimeMillis);
                    else if (arg.argName == "CpuBusyCore")
                    {
                        // simulate one CPU core is busy
                        FileUtility::TryParseUnsigned(arg, 0, std::numeric_limits<unsigned>::max(), &s_cpuBusyCore);
                    }
                    else if (arg.argName == "CpuBusyMs")
                    {
                        // simulate the amount busy time for one CPU core
                        FileUtility::TryParseUnsigned(arg, 0, std::numeric_limits<unsigned>::max(), &s_cpuBusyMs);
                    }
                    else if (arg.argName == "CpuBusyThreadPriority")
                    {
                        // simulate one CPU core is busy with this thread priority
                        FileUtility::TryParseInt(arg, std::numeric_limits<int>::min(), std::numeric_limits<int>::max(), &s_cpuBusyThreadPriority);
                    }
                    else if (arg.argName == "CpuBusyThreadDutyCyclePercent")
                    {
                        // simulate one CPU core is busy for this percent of time
                        FileUtility::TryParseUnsigned(arg, 0, 100, &s_cpuBusyThreadDutyCyclePercent);
                    }
                }
            }
            else
            {
                s_log = (Log*) Platform::Allocate(sizeof(Log));
                new (s_log) Log ("AbuseMain", -1);
                s_log->Write(VerbosityError, "Error opening %s\n", configPath.c_str());
            }

            if(s_log == nullptr)
            {
                s_log = (Log*) Platform::Allocate(sizeof(Log));
                new (s_log) Log ("AbuseMain", -1);
            }

            // write the system version information to the AbuseMain log file
            nn::settings::system::FirmwareVersion osVersion;
            nn::settings::system::GetFirmwareVersion(&osVersion);
            s_log->Write(VerbosityError, "-- Device OS version: %s    Revision: %s\n\n", osVersion.displayVersion, osVersion.revision);

            if(s_isShellEnabled)
            {
               s_shell = (ShellServer*) Platform::Allocate(sizeof(ShellServer));
               new (s_shell) ShellServer();
               s_shell->Initialize(s_listenPort);
            }

            s_activeScript = s_scriptName;
            FileUtility::LoadScript(s_activeScript, s_commands);

            OpenOptions o;
            o.create = true;
            o.write = true;

            char logPath[512];
            nn::util::SNPrintf(logPath, 512, "%s\\Log.txt", Platform::GetLogDirectory());

            Platform::FileClose(file);
        }  // NOLINT(impl/function_size)

        void Abuse::Finalize()
        {
            // kill the CPU core busy thread
            // nn::os::DestroyThread(&s_cpuBusyThread);

            if(s_tasksOnProcessor)
                Platform::Free(s_tasksOnProcessor);
            if(s_scriptName)
                Platform::Free(s_scriptName);
            for(ActiveTask& activeTask : s_activeTasks)
            {
                killTask(activeTask);
            }

            nn::os::SetUserExceptionHandler(nullptr, nn::os::HandlerStackUsesThreadStack, sizeof(nn::os::HandlerStackUsesThreadStack), nn::os::UserExceptionInfoUsesThreadStack);
            Platform::Free(s_exceptionStack);
            Platform::Free(s_exceptionInfo);

            nn::os::FinalizeMutex(&s_taskMutex);
            nn::os::FinalizeMutex(&s_generalMutex);
            nn::os::FinalizeMutex(&s_coreMutex);
            nn::os::FinalizeMutex(&s_resourceMutex);
            nn::os::FinalizeMutex(&s_threadMutex);
            nn::os::FinalizeBarrier(&s_driverBarrier);
        }

        bool Abuse::RegisterTask(const String& taskName, TaskCreateFunc createFunction)
        {
            //Tasknames must be unique
            for(RegisteredTask& task : s_registeredTasks)
            {
                if(taskName == task.typeName)
                    return false;
            }

            s_registeredTasks.push_back( RegisteredTask(taskName, createFunction));
            return true;
        }

        const char* Abuse::GetStartingScriptName()
        {
            return s_scriptName;
        }

        int Abuse::RandRange(int min, int max)
        {
            return (rand() % ( (max - min) + 1)) + min;
        }

        uint16_t Abuse::GetListenPort()
        {
            return s_listenPort;
        }

        bool Abuse::IsShellEnabled()
        {
            return s_isShellEnabled;
        }

        nn::os::Tick Abuse::GetStartTime()
        {
            return s_startTime;
        }

        int Abuse::GetLeastUsedProcessor()
        {
            ScopedMutex mutex(&s_coreMutex);
            if(!s_tasksOnProcessor)
                return 0;

            int minCore = 0;
            int numTasks = std::numeric_limits<int>::max();

            // cpu core 3 is reserved for the OS
            for(int i = 0; i < Platform::GetNumCores(); ++i)
            {
                if(s_tasksOnProcessor[i] < numTasks)
                {
                    numTasks = s_tasksOnProcessor[i];
                    minCore = i;
                }
            }
            return minCore;
        }

        void Abuse::ThreadRunningOnProcessor(int processor)
        {
            ScopedMutex mutex(&s_coreMutex);
            if(processor < Platform::GetNumCores())
                ++s_tasksOnProcessor[processor];
            // s_log->Write(VerbosityVerbose, "++ threads on %d to %d\n", processor, s_tasksOnProcessor[processor]);
        }
        void Abuse::ThreadStoppingOnProcessor(int processor)
        {
            ScopedMutex mutex(&s_coreMutex);
            if(processor < Platform::GetNumCores())
                --s_tasksOnProcessor[processor];
            // s_log->Write(VerbosityVerbose, "-- threads on %d to %d\n", processor, s_tasksOnProcessor[processor]);
        }

        bool Abuse::CreateTask(const TaskCreateArgs& createArgs)
        {
            ScopedMutex mutex(&s_taskMutex);
            s_log->Write(VerbosityInfo, "Creating task %s \"%s\"\n", createArgs.typeName.c_str(), createArgs.instanceName.c_str());
            //Having multiple anonymous tasks with the same name is allowed
            //It is not OK to have multiple tasks with the same instanceName, though - KILLTASK NAME=instanceName would be ambiguous.
            if(createArgs.instanceName != "\0")
            {
                //task names must be unique
                for(ActiveTask& task : s_activeTasks)
                {
                    if(createArgs.instanceName == task.task->GetName())
                    {
                        s_log->Write(VerbosityWarning, "Warning: Could not create task %s because task names must be unique\n", createArgs.instanceName.c_str());
                        return true;
                    }
                }
            }

            for(RegisteredTask& registeredTask : s_registeredTasks)
            {
                if(createArgs.typeName == registeredTask.typeName)
                {
                    BaseTask* task = registeredTask.Create(createArgs.typeName, createArgs.instanceName);

                    Log* log = task->GetLog();

                    log->ToggleTimestamps(createArgs.useTimestamps);
                    log->SetForwardToConsole(createArgs.forwardToConsole);
                    log->SetVerbosity(createArgs.verbosity);

                    if(task)
                    {
                        int processor = createArgs.processor;
                        if(processor == -1)
                            processor = GetLeastUsedProcessor();

                        ThreadRunningOnProcessor(processor);

                        if(createArgs.startSuspended)
                            task->SetSuspended(true);
                        task->SetProcessor(processor);
                        task->SetParams(createArgs.params);
                        task->SetCanRun(true);
                        task->SetPriority(createArgs.priority);
                        addTask(task, false);
                        return true;
                    }
                    else
                    {
                        return false;
                    }
                }

            }
            //taskname not found.
            s_log->Write(VerbosityWarning, "No task with name %s has been registered\n", createArgs.typeName.c_str());
            return false;
        }

        void Abuse::killTask(ActiveTask& activeTask)
        {

            s_log->Write(VerbosityInfo, "Killing task: %s\n", activeTask.task->GetName().c_str());

            activeTask.task->SetKilled();
            activeTask.task->SetCanRun(false);

            //Wait until task isn't running
            while(activeTask.task->IsRunning())
            {
                nn::os::YieldThread();
            }

            if(activeTask.task->IsInitialized())
                activeTask.task->Shutdown();

            for(int i = 0; i < NUM_RESOURCES; ++i)
            {
                if(s_resourceOwner[i] == activeTask.task)
                {
                    s_resourceOwner[i] = nullptr;
                    s_resourceOwned[i] = false;
                }
            }

            ThreadStoppingOnProcessor(activeTask.task->GetProcessor());
            activeTask.task->~BaseTask();
            Platform::Free(activeTask.task);
        }

        bool Abuse::KillTask(int id)
        {
            ScopedMutex mutex(&s_taskMutex);
            auto itr = s_activeTasks.begin();
            while(itr != s_activeTasks.end())
            {
                if(id == itr->task->GetId())
                {
                    killTask(*itr);
                    s_activeTasks.erase(itr);
                    return true;
                }
                ++itr;
            }
            return false;
        }

        bool Abuse::KillTask(const String& name)
        {
            if(name == "")
                return false;

            ScopedMutex mutex(&s_taskMutex);
            auto itr = s_activeTasks.begin();
            while(itr != s_activeTasks.end())
            {
                if(name == itr->task->GetName())
                {
                    killTask(*itr);
                    s_activeTasks.erase(itr);
                    return true;
                }
                ++itr;
            }
            return false;
        }

        void Abuse::KillAllTasks()
        {
            ScopedMutex mutex(&s_taskMutex);
            auto itr = s_activeTasks.begin();
            while(itr != s_activeTasks.end())
            {
                if(itr->task->GetTypeName() == "AbuseDriver")
                {
                    ++itr;
                    continue;
                }
                killTask(*itr);
                itr = s_activeTasks.erase(itr);
            }
            exit(0); // Exit gracefully so that we can get a PASS logged in CI
        }

        int Abuse::CleanupShutdownTasks()
        {
            ScopedMutex mutex(&s_taskMutex);
            int cleanedTasks = 0;
            auto itr = s_activeTasks.begin();
            while(itr != s_activeTasks.end())
            {
                if(itr->task->IsShutdown())
                {
                    killTask(*itr);
                    itr = s_activeTasks.erase(itr);
                    ++cleanedTasks;
                }
                else ++itr;
            }
            return cleanedTasks;
        }

        bool Abuse::setSuspended(const String& name, bool suspended)
        {
            if(name == "")
                return false;
            for(ActiveTask& task : s_activeTasks)
            {
                if(name == task.task->GetName())
                {
                    s_log->Write(VerbosityVerbose, "%s task: %s\n",suspended ? "Suspending" : "Resuming", name.c_str());
                    task.task->SetSuspended(suspended);
                    return true;
                }
            }
            return false;
        }

        bool Abuse::SuspendTask(const String& name)
        {
            return setSuspended(name, true);
        }

        void Abuse::SuspendAllTasks()
        {
            ScopedMutex mutex(&s_taskMutex);
            for(ActiveTask& task : s_activeTasks)
            {
                if(task.task->GetTypeName() == "AbuseDriver")
                    continue;
                s_log->Write(VerbosityInfo, "Suspending task %s\n", task.task->GetName().c_str());
                task.task->SetSuspended(true);
            }
        }

        bool Abuse::ResumeTask(const String& name)
        {
            return setSuspended(name, false);
        }

        void Abuse::ResumeAllTasks()
        {
            ScopedMutex mutex(&s_taskMutex);
            for(ActiveTask& task : s_activeTasks)
            {
                if(task.task->GetTypeName() == "AbuseDriver")
                    continue;
                s_log->Write(VerbosityInfo, "Resuming task %s\n", task.task->GetName().c_str());
                task.task->SetSuspended(false);
            }
        }

        void Abuse::GeneralMutexLock()
        {
            nn::os::LockMutex(&s_generalMutex);
        }

        void Abuse::GeneralMutexUnlock()
        {
            nn::os::UnlockMutex(&s_generalMutex);
        }

        void Abuse::GeneralMutexTryLock()
        {
            nn::os::TryLockMutex(&s_generalMutex);
        }

        ScopedMutex Abuse::GeneralMutexScopedLock()
        {
            return ScopedMutex(&s_generalMutex);
        }

        bool Abuse::IsRunning()
        {
            return s_isRunning;
        }

        void Abuse::StopRunning()
        {
            s_isRunning = false;
        }

        unsigned Abuse::GetTimeoutDuration()
        {
            return s_timeoutMillis;
        }

        void Abuse::Run()
        {
            CpuBusyThreadValues cpuBusyThreadArgs;
            nn::os::ThreadType cpuBusyThread;
            nn::Result result;
            void* cpuBusyThreadStack;

            //BaseTask* driver = BaseTask::CreateTask<AbuseDriver>("AbuseDriver", "");
            int numCores = Platform::GetNumCores();
            s_taskDrivers = (AbuseDriver*)Platform::Allocate(numCores * sizeof(AbuseDriver));

            nn::os::ThreadType s_driverThreads[NUM_CORES];
            void* s_driverStacks[NUM_CORES];

            // start a thread on each of the CPU cores to manage running tasks that are active on that core
            // core 3 is reserved for the OS, so don't use it
            for(int i = 0; i < numCores; i++)
            {
                new (&s_taskDrivers[i]) AbuseDriver(i, &s_activeTasks, s_heartbeats);
                s_driverStacks[i] = Platform::AllocateAligned(DRIVER_STACK_SIZE, nn::os::ThreadStackAlignment);
                nn::os::CreateThread(&s_driverThreads[i], driverWorkerThread, &s_taskDrivers[i], s_driverStacks[i], DRIVER_STACK_SIZE, nn::os::DefaultThreadPriority, i);
            }

            for(int i = 0; i < numCores; i++)
            {
                ++s_numRunningDrivers;
                nn::os::StartThread(&s_driverThreads[i]);
            }

            for(int i = 0; i < numCores; i++)
            {
                nn::os::WaitThread(&s_driverThreads[i]);
                nn::os::DestroyThread(&s_driverThreads[i]);

                Platform::Free(&s_driverThreads[i]);
                Platform::Free(&s_driverStacks[i]);
            }

            Platform::Free(s_taskDrivers);

            // start the parasitic CPU core busy thread if necessary
            if (s_cpuBusyThreadDutyCyclePercent > 0)
            {
                cpuBusyThreadArgs.cpuBusyMs = s_cpuBusyMs;
                cpuBusyThreadArgs.cpuBusyThreadDutyCyclePercent = s_cpuBusyThreadDutyCyclePercent;

                cpuBusyThreadStack = Platform::AllocateAligned(CPU_BUSY_STACK_SIZE, nn::os::ThreadStackAlignment);

                result = nn::os::CreateThread(&cpuBusyThread,
                    CpuBusyThreadFunc,
                    (void *)&cpuBusyThreadArgs,
                    cpuBusyThreadStack,
                    CPU_BUSY_STACK_SIZE,
                    s_cpuBusyThreadPriority,
                    s_cpuBusyCore);
                NN_ASSERT(result.IsSuccess(), "Cannot create thread for CPU core busy");

                nn::os::StartThread(&cpuBusyThread);
                nn::os::WaitThread(&cpuBusyThread);
                nn::os::DestroyThread(&cpuBusyThread);
                Platform::Free(cpuBusyThreadStack);
            }
        }

        bool Abuse::AcquireResource(BaseTask* task, AbuseResource resource)
        {
            ScopedMutex mutex(&s_resourceMutex);
            if(!task)
                return false;

            if(resource >= NUM_RESOURCES)
                return false;

            if(s_resourceOwned[resource])
                return false;

            s_resourceOwned[resource] = true;
            s_resourceOwner[resource] = task;
            return true;
        }

        void Abuse::ReleaseResource(BaseTask* task, AbuseResource resource)
        {
            ScopedMutex mutex(&s_resourceMutex);
            if(resource >= NUM_RESOURCES)
                return;

            if(s_resourceOwner[resource] != task)
                return;

            s_resourceOwner[resource] = nullptr;
            s_resourceOwned[resource] = false;
        }

        ListActiveTask* Abuse::GetActiveTasksList()
        {
            return &s_activeTasks;
        }

        void Abuse::driverWorkerThread(void* voidDriver)
        {
            while(s_numRunningDrivers != Platform::GetNumCores())
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

            AbuseDriver* driver = (AbuseDriver*)voidDriver;
            s_log->Write(VerbosityInfo, "Running driver %d\n", nn::os::GetCurrentCoreNumber());
            while(s_isRunning)
            {
                s_heartbeats[driver->GetCore()] = Platform::GetSystemRunningTimeMilli();
                if(driver->ShouldUpkeep())
                {
                    updateShell();
                    updateScripts();
                }

                nn::os::AwaitBarrier(&s_driverBarrier);

                nn::os::LockMutex(&s_taskMutex);
                BaseTask* task = driver->ChooseTask();
                nn::os::UnlockMutex(&s_taskMutex);

                if(!task || !task->CanRun())
                {
                    nn::os::YieldThread();
                    continue;
                }

                task->SetRunning(true);

                //s_log->Write(VerbosityInfo, "Core %d running task %d(%s)\n", nn::os::GetCurrentCoreNumber(), task->GetId(), task->GetName().c_str());

                nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), task->GetBasePriority());
                if(!task->IsInitialized())
                {
                    InitStatus status= task->Initialize(task->GetParams());
                    task->SetInitialized();
                    if(status != INIT_OK)
                    {
                        KillTask(task->GetId());
                        continue;
                    }
                }
                StartStatus startStatus = task->Start();

                RunStatus runStatus = RUN_OK;
                if(startStatus == START_OK)
                {
                    nn::os::Tick end = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(s_taskRuntimeMillis));
                    while(task->IsRunning() && end > nn::os::GetSystemTick())
                    {
                        runStatus = task->Run();
                        if(runStatus == RUN_YIELD)
                            break;
                        if(runStatus < 0)
                            task->SetCanRun(false);

                        s_heartbeats[driver->GetCore()] = Platform::GetSystemRunningTimeMilli();
                    }
                }

                StopStatus stopStatus = task->Stop();
                task->SetRunning(false);

                if(runStatus != RUN_YIELD)
                {
                    task->ResetPriority();
                }

                if(runStatus < 0 || startStatus < 0 || stopStatus < 0)
                    KillTask(task->GetId());

                nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), nn::os::HighestThreadPriority);

                //s_log->Write(VerbosityInfo, "Core %d stopping %s\n", nn::os::GetCurrentCoreNumber(), task->GetName().c_str());
                nn::os::YieldThread();
            }
        }

        void Abuse::addTask(BaseTask* task, bool acquireLock)
        {
            if(acquireLock)
                nn::os::LockMutex(&s_taskMutex);
            auto itr = s_activeTasks.begin();

            while(itr != s_activeTasks.end())
            {
                int priority = itr->task->GetCurrentPriority();
                if(priority > task->GetCurrentPriority())
                {
                    s_activeTasks.insert(itr, ActiveTask(task));
                    break;;
                }
                ++itr;
            }
            s_activeTasks.push_back(task);
            if(acquireLock)
                nn::os::LockMutex(&s_taskMutex);
        }

        void Abuse::exceptionHandler(nn::os::UserExceptionInfo* info)
        {
            s_log->Write(VerbosityError, "Triggered exception %x\n", info->exceptionType);
            for(ActiveTask task : s_activeTasks)
            {
                task.task->FlushLog(true);
            }
        }

        void Abuse::updateScripts()
        {
            if(s_shellCommand)
            {
                unsigned index = 0;
                bool result = s_shellCommand->Execute(index);

                if(index != 0)
                {
                    const ShellCommand& command = s_shell->PeekNextCommand();
                    if(result)
                        s_shell->SendCommandResult(command.socket, true, "Command \"%s\" Passed", command.args.c_str());
                    else
                        s_shell->SendCommandResult(command.socket, false, "Command \"%s\" Failed", command.args.c_str());
                    s_shell->PopNextCommand();
                    Platform::Free(s_shellCommand);
                    s_shellCommand = nullptr;
                }
                else
                {
                    const ShellCommand& command = s_shell->PeekNextCommand();
                    s_shell->SendReply(command.socket, ShellPacketType::PACKET_ACK_CONTINUE, "", 1);
                }
            }

            if(s_commands.size() == 0 || s_commandIndex == s_commands.size())
                return;

            else
            {
                bool result = s_commands[s_commandIndex]->Execute(s_commandIndex);

                if(s_commandIndex >= s_commands.size())
                    s_activeScript="";

                if(!result)
                {
                    clearScriptCommands();
                }
            }
        }

        void Abuse::updateShell()
        {
            if(!Abuse::IsShellEnabled())
                return;
            s_shell->UpdateServer();

            if(s_shell->IsCommandPending())
            {
                const ShellCommand& command = s_shell->PeekNextCommand();

                bool popCommand = true;
                switch(command.type)
                {
                case SHELL_SCRIPT_FILE:
                    {
                        if(s_activeScript != "")
                        {
                            s_shell->SendString(command.socket, "Could not start script because %s is already running\n", s_activeScript.c_str());
                            s_shell->SendString(command.socket, "Please run StopScript before running this script\n");
                            break;
                        }
                        OpenOptions options;
                        options.read = true;
                        File file = Platform::FileOpen(command.args.c_str(), options);

                        if(file != FILE_INVALID_HANDLE)
                        {
                            Platform::FileClose(file);

                            Abuse::KillAllTasks();
                            s_commandIndex = 0;
                            bool result = FileUtility::LoadScript(command.args, s_commands);

                            if(result)
                            {
                                s_activeScript = command.args;
                                s_shell->SendCommandResult(command.socket, true, "Successfully running script %s\n", command.args.c_str());
                            }
                            else
                            {
                                s_shell->SendCommandResult(command.socket, false, "Error loading script %s\n", command.args.c_str());
                            }
                        }
                        break;
                    }
                case SHELL_STOP_SCRIPT:
                    {
                        clearScriptCommands();
                        s_shell->SendCommandResult(command.socket, true, "Stopping script %s\n", s_activeScript.c_str());
                        s_activeScript = "";
                        Abuse::KillAllTasks();
                        break;
                    }
                case SHELL_RUN_SCRIPT_COMMAND:
                    {
                        if(s_activeScript != "")
                        {
                            s_shell->SendString(command.socket, "Could not docommand because %s is already running\n", s_activeScript.c_str());
                            s_shell->SendString(command.socket, "Please run StopScript before running this command\n");
                            break;
                        }
                        //Wait for this command to finish before getting the next one
                        if(s_shellCommand)
                        {
                            popCommand = false;
                            break;
                        }

                        s_shellCommand = FileUtility::ParseCommand(command.args);

                        if(s_shellCommand)
                        {
                            popCommand = false;
                        }
                        else
                            s_shell->SendCommandResult(command.socket, false, "Error parsing docommand\"%s\"\n", command.args.c_str());

                        break;
                    }
                case SHELL_EXIT_ABUSE:
                    s_shell->SendCommandResult(command.socket, true, "Quitting abuse\n", command.args.c_str());
                    s_isRunning = false;
                    break;
                case SHELL_GET_AVAILABLE_TASKS:
                {
                    for(RegisteredTask task : s_registeredTasks)
                    {
                        BaseTask* baseTask = task.Create(task.typeName, "");
                        s_shell->SendString(command.socket, "%s %s\n", task.typeName.c_str(), baseTask->GetParamOptions());
                        baseTask->~BaseTask();
                        Platform::Free(baseTask);
                    }
                    s_shell->SendReply(command.socket, ShellPacketType::PACKET_ACK_PASS, "", 1);
                    break;
                }
                case SHELL_INIT_MONITOR:
                    {
                        const int BUFFER_SIZE = 1010;
                        char workBuffer[BUFFER_SIZE + 1];
                        memset(workBuffer, 0, BUFFER_SIZE + 1);
                        workBuffer[0] = 0;
                        int size = Platform::PutNetworkBuffer(workBuffer, 1, (unsigned)Platform::GetNumCores());
                        for(int i = 0; i < Platform::GetNumCores(); ++i)
                            size = Platform::PutNetworkBuffer(workBuffer, (uint16_t)size, (uint64_t)0);
                        size = Platform::PutNetworkBuffer(workBuffer, (uint16_t)size, (uint64_t)Platform::GetSystemRunningTimeMilli());
                        size_t platformLength = strlen(Platform::GetPlatformName());
                        size = Platform::PutNetworkBuffer(workBuffer, (uint16_t)size, platformLength);
                        strncpy(workBuffer + size, Platform::GetPlatformName(), platformLength);
                        (workBuffer + size)[platformLength] = '\0';
                        size += (uint16_t)platformLength;

                        char* numTaskLoc = workBuffer + size;
                        size += 4;
                        unsigned numTasks = 0;
                        unsigned totalTasks = 0;
                        nn::os::LockMutex(&s_taskMutex);
                        for(ActiveTask& task : s_activeTasks)
                        {
                            int oldSize = size;
                            size = task.task->Serialize(workBuffer, size, BUFFER_SIZE);
                            if(size == -1)
                            {
                                workBuffer[oldSize]=1;
                                Platform::PutNetworkBuffer(numTaskLoc, 0, numTasks);
                                totalTasks += numTasks;
                                s_shell->SendReply(command.socket, PACKET_INIT_MONITOR, workBuffer, (uint16_t)oldSize + 1);
                                memset(workBuffer, 0, BUFFER_SIZE + 1);
                                workBuffer[0] = 1;
                                numTaskLoc = workBuffer + 1;
                                size = 5;
                                size = task.task->Serialize(workBuffer, (uint16_t)size, BUFFER_SIZE);
                                numTasks = 1;
                            }
                            else
                                ++numTasks;
                        }
                        nn::os::UnlockMutex(&s_taskMutex);
                        Platform::PutNetworkBuffer(numTaskLoc, 0, numTasks);
                        workBuffer[size]=(char)0x47;
                        ++size;
                        totalTasks += numTasks;
                        s_shell->SendReply(command.socket, PACKET_INIT_MONITOR, workBuffer, (uint16_t)size);
                        break;
                    }
                case SHELL_INVALID:
                default:
                    {
                        s_shell->SendString(command.socket, "Unrecognized command\n");
                        break;
                    }
                }

                if(popCommand)
                    s_shell->PopNextCommand();
            }
        }  // NOLINT(impl/function_size)

        void Abuse::clearScriptCommands()
        {
            for(unsigned i = 0; i < s_commands.size(); ++i)
            {
                Platform::Free(s_commands[i]);
            }
            s_commands.clear();
            s_commandIndex = 0;
        }

        void CpuBusyThreadFunc(void *args)
        {
            // max time to let thread run before yield
            const int thread_yield_ms = 10;
            CpuBusyThreadValues *cpuBusyThreadArgs = (CpuBusyThreadValues *)args;
            nn::TimeSpan stopTime = 0;
            nn::TimeSpan yieldTime = 0;
            bool fTerminate = false;
            nn::Result result;
            int i = 0;
            int off_ms = cpuBusyThreadArgs->cpuBusyMs * (100 - cpuBusyThreadArgs->cpuBusyThreadDutyCyclePercent) / cpuBusyThreadArgs->cpuBusyThreadDutyCyclePercent;

            while (!fTerminate)
            {
                // thread on cycle
                stopTime = nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromMilliSeconds(cpuBusyThreadArgs->cpuBusyMs);
                yieldTime = nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromMilliSeconds(thread_yield_ms);
                while (nn::os::GetSystemTick().ToTimeSpan() < stopTime)
                {
                    // busy wait loop
                    i++;
                    i--;

                    // since threads are run to completion on NX, a call to YieldThread
                    // is necessary in order to allow other threads with the same priority to run
                    // yield every duration of thread_yield_ms
                    if (nn::os::GetSystemTick().ToTimeSpan() >= yieldTime)
                    {
                        nn::os::YieldThread();
                        yieldTime = nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromMilliSeconds(thread_yield_ms);
                    }
                }

                if (!fTerminate)
                {
                    int slices_off_ms = off_ms;

                    // sleep during the thread off cycle in time slices of thread_yield_ms duration
                    while (slices_off_ms >= thread_yield_ms)
                    {
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(thread_yield_ms));
                        slices_off_ms = slices_off_ms - thread_yield_ms;

                        // since threads are run to completion on NX, a call to YieldThread
                        // is necessary in order to allow other threads with the same priority to run
                        nn::os::YieldThread();
                    }

                    // check for any left over time slice to sleep in ms
                    if (slices_off_ms > 0)
                    {
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(slices_off_ms));
                    }
                }
            }
        }

    }
}
