﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "fwk_Api.h"
#include "fwk_Print.h"

#include <nn/svc/svc_Base.h>

namespace nnt { namespace svc { namespace fwk {

static const int MaximumTestCaseCount = 256;
TestCase g_TestCases[MaximumTestCaseCount];
int      g_TestCaseCount = 0;

static const int MaximumTestCaseDependencyCount = 256; // Worst case: MaximumTestCaseCount ^ 2
TestCaseDependency g_TestCaseDependencies[MaximumTestCaseDependencyCount];
int                g_TestCaseDependencyCount = 0;

static const int MaximumTestCount = 1024;
Test   g_Tests[MaximumTestCount];
int    g_TestCount = 0;

int g_CurrentTest;
static const int MaximumSubTestDepth = 16;
const char* (g_SubTestPath[MaximumSubTestDepth]);
int g_SubTestPathDepth = 0;

struct Tracker
{
    const char* name;
    void (*printFunction) (void*);
    void* userdata;
};
static const int MaximumTrackerCount = 16;
Tracker g_Trackers[MaximumTrackerCount];
int g_TrackerCount = 0;

static bool StringEquals(const char* a, const char* b)
{
    while (*a == *b && *a != 0)
    {
        a++;
        b++;
    }

    return *a == 0 && *b == 0;
}

static const int IndexNotFound = -1;
static int FindTestCaseIndex(const char* testCaseName)
{
    for (int index = 0; index < g_TestCaseCount; index++)
    {
        TestCase& testCase = g_TestCases[index];

        if (StringEquals(testCase.name, testCaseName))
            return index;
    }

    return IndexNotFound;
}

static int FindTestCaseIndexOrCreate(const char* testCaseName)
{
    int index = FindTestCaseIndex(testCaseName);
    if (index == IndexNotFound)
    {
        TestCase& testCase = g_TestCases[g_TestCaseCount++];
        NNT_SVC_FWK_ABORT_UNLESS(g_TestCaseCount <= MaximumTestCaseCount);

        testCase.name = testCaseName;
        index = g_TestCaseCount - 1;
    }

    return index;
}

TestAdder::TestAdder(const char* testCaseName, const char* testName, void (*testFunction)())
{
    Test& test = g_Tests[g_TestCount++];
    NNT_SVC_FWK_ABORT_UNLESS(g_TestCount <= MaximumTestCount);

    test.testCaseIndex = FindTestCaseIndexOrCreate(testCaseName);
    test.name = testName;
    test.function = testFunction;
}

TestCaseDependencyAdder::TestCaseDependencyAdder(const char* testCaseName, const char* requiredTestCaseName)
{
    TestCaseDependency& testCaseDependency = g_TestCaseDependencies[g_TestCaseDependencyCount++];
    NNT_SVC_FWK_ABORT_UNLESS(g_TestCaseDependencyCount <= MaximumTestCaseDependencyCount);

    testCaseDependency.testCaseIndex = FindTestCaseIndexOrCreate(testCaseName);
    testCaseDependency.requiredTestCaseIndex = FindTestCaseIndexOrCreate(requiredTestCaseName);
}

ScopedSubTest::ScopedSubTest(const char* subTestName)
{
    NNT_SVC_FWK_ABORT_UNLESS(g_SubTestPathDepth < MaximumSubTestDepth);

    g_SubTestPath[g_SubTestPathDepth++] = subTestName;

    // TODO: buffer instead of doing many SvcOutputDebugString
    Test& test = g_Tests[g_CurrentTest];
    TestCase& testCase = g_TestCases[test.testCaseIndex];
    TaggedPrintf(Color_Green, "  RUNSUB  ", "%s.%s",
        testCase.name,
        test.name);

    for (int k = 0; k < g_SubTestPathDepth; k++)
        Printf(".%s", g_SubTestPath[k]);
    Printf("\n");
}

ScopedSubTest::~ScopedSubTest()
{
    // TODO: buffer instead of doing many SvcOutputDebugString
    Test& test = g_Tests[g_CurrentTest];
    TestCase& testCase = g_TestCases[test.testCaseIndex];
    TaggedPrintf(Color_Green, "   OKSUB  ", "%s.%s",
        testCase.name,
        test.name);

    for (int k = 0; k < g_SubTestPathDepth; k++)
        Printf(".%s", g_SubTestPath[k]);
    Printf("\n");

    NNT_SVC_FWK_ABORT_UNLESS(g_SubTestPathDepth > 0);
    g_SubTestPathDepth--;
}

void PushTracker(const char* name, void (*printFunction)(void*), void* userdata)
{
    NNT_SVC_FWK_ABORT_UNLESS(g_TrackerCount < MaximumTrackerCount);
    g_Trackers[g_TrackerCount].name = name;
    g_Trackers[g_TrackerCount].printFunction = printFunction;
    g_Trackers[g_TrackerCount].userdata = userdata;

    g_TrackerCount++;
}

void PopTracker()
{
    NNT_SVC_FWK_ABORT_UNLESS(g_TrackerCount > 0);
    g_TrackerCount--;
}

void DumpTrackers()
{
    Printf("Trackers: \n");
    for (int t = 0; t < g_TrackerCount; t++)
    {
        Tracker& tracker = g_Trackers[t];
        Printf("%s: ", tracker.name);
        tracker.printFunction(tracker.userdata);
        Printf("\n");
    }
}

static void VPrintf(const char* format, va_list va)
{
    char buffer[512];
    int length = TVSNPrintf(buffer, sizeof(buffer), format, va);
    NNT_SVC_FWK_ABORT_UNLESS(length >= 0);
    nn::svc::OutputDebugString(buffer, length);
}

void Printf(const char* format, ...)
{
    va_list va;
    va_start (va, format);
    VPrintf(format, va);
    va_end(va);
}

int SNPrintf(char* buffer, size_t bufferSize, const char* format, ...)
{
    va_list va;
    va_start (va, format);
    int length = TVSNPrintf(buffer, bufferSize, format, va);
    va_end(va);
    return length;
}

extern "C" void* memcpy(void* dest, const void* src, int size)
{
    const uint8_t* srcPtr = static_cast<const uint8_t*> (src);
    uint8_t* dstPtr = static_cast<uint8_t*> (dest);

    while (size--)
        *dstPtr++ = *srcPtr++;

    return dest;
}

extern "C" void* memset(void* s, int c, size_t n)
{
    uint8_t* ptr = static_cast<uint8_t*> (s);
    while (n--)
        *ptr++ = c;
    return s;
}

void TaggedPrintf(Color tagColor, const char* tag, const char* format, ...)
{
    int color =
        tagColor == Color_Normal ? 39 :
        tagColor == Color_Green ? 32 :
        tagColor == Color_Red ? 31 : -1;
    NNT_SVC_FWK_ABORT_UNLESS(color != -1);

    Printf("%c[%d;49m[%s]%c[0m ", 0x1b, color, tag, 0x1b);

    va_list va;
    va_start (va, format);
    VPrintf(format, va);
    va_end(va);
}

static bool IsTestCaseDone(int testCaseIndex)
{
    for (int testIndex = 0; testIndex < g_TestCount; testIndex++)
    {
        Test& test = g_Tests[testIndex];
        if (test.testCaseIndex == testCaseIndex && test.result == TestResult_NotRun)
        {
            return false;
        }
    }

    return true;
}

void RunTest(int testIndex)
{
    Test& test = g_Tests[testIndex];
    TestCase& testCase = g_TestCases[test.testCaseIndex];

    TaggedPrintf(Color_Green, " RUN      ", "%s.%s\n", testCase.name, test.name);
    g_CurrentTest = testIndex;

    test.function();
    test.result = TestResult_Success; // Test failure is an abort right now...
    TaggedPrintf(Color_Green, "       OK ", "%s.%s\n", testCase.name, test.name);
}

void RunAllTests()
{
    TaggedPrintf(Color_Normal, "==========", "Running %d tests\n", g_TestCount);

    for (;;)
    {
        bool hasRunTest = false;
        for (int testIndex = 0; testIndex < g_TestCount; testIndex++)
        {
            Test& test = g_Tests[testIndex];

            if (test.result != TestResult_NotRun)
            {
                continue;
            }

            // TODO: the time complexity is horrible right now...
            bool areDependenciesSatisfied = true;
            for (int dependencyIndex = 0; dependencyIndex < g_TestCaseDependencyCount; dependencyIndex++)
            {
                TestCaseDependency& dependency = g_TestCaseDependencies[dependencyIndex];
                if (dependency.testCaseIndex == test.testCaseIndex && ! IsTestCaseDone(dependency.requiredTestCaseIndex))
                {
                    areDependenciesSatisfied = false;
                }
            }

            if (areDependenciesSatisfied)
            {
                RunTest(testIndex);
                hasRunTest = true;
            }
        }

        if (! hasRunTest)
        {
            break;
        }
    }

    auto listTestsForResult = [] (TestResult result)
    {
        for (int testIndex = 0; testIndex < g_TestCount; testIndex++)
        {
            Test& test = g_Tests[testIndex];
            TestCase& testCase = g_TestCases[test.testCaseIndex];

            if (test.result == result)
            {
                TaggedPrintf(Color_Normal, "----------", " - %s.%s\n", testCase.name, test.name);
            }
        }
    };

    TaggedPrintf(Color_Normal, "==========", "Successful tests: \n");
    listTestsForResult(TestResult_Success);
    TaggedPrintf(Color_Normal, "==========", "Skipped tests: \n");
    listTestsForResult(TestResult_NotRun);
}

} } }

// From kern_Panic.cpp

#include <nn/diag/diag_LogTypes.h>
#include <nn/diag/detail/diag_DetailAssert.h>
#include <nn/diag/diag_AssertionFailureHandler.h>
namespace nn { namespace diag {
namespace {
const char* GetAssertionFailureMessageByType(
        AssertionType assertionType) NN_NOEXCEPT
{
    switch( assertionType )
    {
    case AssertionType_SdkAssert:
        return "SDK Assertion Failure";
    case AssertionType_SdkRequires:
        return "Precondition not met";
    case AssertionType_UserAssert:
        return "Assertion Failure";
    default:
        // 不明な AssertionType が渡された場合であっても、AssertionType が正しいことの Assert で停止するのではなく、
        // Assertion 失敗の情報を表示するべきです。
        return "Assertion Failure (Unknown Type)";
    }
}
void DefaultPrinter(
        const char* failureMessage,
        const char* condition,
        const char* functionName,
        const char* fileName,
        int lineNumber,
        const LogMessage& message) NN_NOEXCEPT
{
    // TORIAEZU: LogMessage オブジェクトのフォーマットがまだできないため、NN_SDK_LOG を使用せず直接デバッグ出力を行う
    NN_UNUSED(failureMessage);
    NN_UNUSED(condition);
    NN_UNUSED(functionName);
    NN_UNUSED(fileName);
    NN_UNUSED(lineNumber);
    NN_UNUSED(message);
    ::nnt::svc::fwk::Printf(
            "%s: \'%s\' at '%s':%d in %s\n",
            failureMessage, condition, fileName, lineNumber, functionName);
    ::nnt::svc::fwk::VPrintf(message.format, *message.args);
    ::nnt::svc::fwk::Printf("\n");
}

AssertionFailureOperation DefaultAssertionFailureHandler(
        const AssertionInfo &assertionInfo)
{
    DefaultPrinter(
            GetAssertionFailureMessageByType(assertionInfo.assertionType),
            assertionInfo.condition,
            assertionInfo.functionName,
            assertionInfo.fileName,
            assertionInfo.lineNumber,
            *assertionInfo.message);

    return AssertionFailureOperation_Abort;
}

}

void detail::OnAssertionFailure(
    AssertionType assertionType,
    const char* condition,
    const char* functionName,
    const char* fileName,
    int lineNumber,
    const char* format,
    ...)
{
    va_list list;
    va_start(list, format);

    const nn::diag::LogMessage message = {format, &list};
    const AssertionInfo assertionInfo = {assertionType, &message, condition, functionName, fileName, lineNumber};
    DefaultAssertionFailureHandler(assertionInfo);

    va_end(list);
}

void detail::OnAssertionFailure(
    AssertionType assertionType,
    const char* condition,
    const char* functionName,
    const char* fileName,
    int lineNumber)
{
    detail::OnAssertionFailure(assertionType, condition, functionName, fileName, lineNumber, "");
}
}}
