﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_TestSvc.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <cstring>
#include "sdmmc1/sdmmc_Api.h"

namespace
{
    const nn::sdmmc1::Port TestPort = nn::sdmmc1::Port_Mmc0;    // sdmmc3a
    //const nn::sdmmc1::Port TestPort = nn::sdmmc1::Port_SdCard0; // sdmmc4a

    const uint32_t TestSectorIndex = 0;
    const uint32_t TestSize = 32 * 1024;
    const uint32_t TestNumSectors = TestSize / nn::sdmmc1::SectorSize;
    NN_ALIGNAS(4096) uint8_t s_Buffer[TestSize];

    const char* GetPortName(nn::sdmmc1::Port port)
    {
        switch (port)
        {
        case nn::sdmmc1::Port_Mmc0:
            return "Port_Mmc0";
        case nn::sdmmc1::Port_SdCard0:
            return "Port_SdCard0";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

class Sdmmc
{
public:
    Sdmmc()
    {
        nn::sdmmc1::Initialize();
        NN_ABORT_UNLESS(nn::sdmmc1::Activate(TestPort).IsSuccess());
    }
    ~Sdmmc()
    {
        nn::sdmmc1::Deactivate(TestPort);
        nn::sdmmc1::Finalize();
    }
};

}
TEST(sMMU, Test0)
{
    nn::Result result;
    Sdmmc sdmmc;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, 0x80000000);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::MapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer),
            nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(result.IsSuccess());

    std::memset(s_Buffer, 0, sizeof(s_Buffer));
    result = nn::sdmmc1::Read(s_Buffer, sizeof(s_Buffer), TestPort, TestSectorIndex, TestNumSectors);
    ASSERT_TRUE(result.IsSuccess());

#if 0
    for (int i = 0; i < sizeof(s_Buffer); i += 16)
    {
        NN_LOG("%08x: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n", i,
                s_Buffer[i + 0], s_Buffer[i + 1], s_Buffer[i + 2], s_Buffer[i + 3], s_Buffer[i + 4], s_Buffer[i + 5],
                s_Buffer[i + 6], s_Buffer[i + 7], s_Buffer[i + 8], s_Buffer[i + 9], s_Buffer[i + 10], s_Buffer[i + 11],
                s_Buffer[i + 12], s_Buffer[i + 13], s_Buffer[i + 14], s_Buffer[i + 15]);
    }
#endif

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer));
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::CloseHandle(handle);
    ASSERT_TRUE(result.IsSuccess());
}

TEST(sMMU, Test1)
{
    nn::Result result;
    Sdmmc sdmmc;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, 0x80000000);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::SetMemoryAttribute(reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer),
            nn::svc::MemoryAttribute_Uncached, nn::svc::MemoryAttribute_Uncached);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::MapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer),
            nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer));
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::SetMemoryAttribute(reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer),
            nn::svc::MemoryAttribute_Uncached, 0);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::CloseHandle(handle);
    ASSERT_TRUE(result.IsSuccess());
}


TEST(sMMU, Test2)
{
    nn::Result result;
    Sdmmc sdmmc;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, 0x80000000);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::MapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer),
            nn::svc::MemoryPermission_Write, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(result.IsSuccess());

    std::memset(s_Buffer, 0, sizeof(s_Buffer));
    result = nn::sdmmc1::Read(s_Buffer, sizeof(s_Buffer), TestPort, TestSectorIndex, TestNumSectors);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer));
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::CloseHandle(handle);
    ASSERT_TRUE(result.IsSuccess());
}

TEST(sMMU, Test3)
{
    nn::Result result;
    Sdmmc sdmmc;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, 0x80000000);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::MapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer),
            nn::svc::MemoryPermission_Read, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(result.IsSuccess());

    NN_LOG("sMMU エラー割り込みが発生します for %p\n", s_Buffer);
    std::memset(s_Buffer, 0, sizeof(s_Buffer));
    result = nn::sdmmc1::Read(s_Buffer, sizeof(s_Buffer), TestPort, TestSectorIndex, TestNumSectors);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer));
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::CloseHandle(handle);
    ASSERT_TRUE(result.IsSuccess());
}

TEST(sMMU, Test4)
{
    nn::Result result;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, 0x80000000);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsFailure());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsFailure());

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer));
    ASSERT_TRUE(result.IsFailure());

    result = nn::svc::CloseHandle(handle);
    ASSERT_TRUE(result.IsSuccess());
}

TEST(sMMU, Test5)
{
    nn::Result result;
    nn::svc::Handle handle;

    for (int i = 0; i < 2u * 1024 * 1024 * 1024 / 4096; i++)
    {
        result = nn::svc::CreateDeviceAddressSpace(&handle, 0x80000000);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::MapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer),
                nn::svc::MemoryPermission_Read, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer));
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::CloseHandle(handle);
        ASSERT_TRUE(result.IsSuccess());
    }
}


TEST(sMMU, Test6)
{
    uintptr_t addr;
    size_t size;

    nn::Result result;
    Sdmmc sdmmc;
    nn::svc::Handle handle;

    size = 0x01000000;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_TRUE(result.IsSuccess());
    // 4MBアラインされるはず
    ASSERT_TRUE((addr & (0x00400000 - 1)) == 0);

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, addr);
    ASSERT_TRUE(result.IsSuccess());
    // 4MBアラインされるはず
    ASSERT_TRUE(((physInfo.physicalAddress + (addr - physInfo.virtualAddress)) & (0x00400000 - 1)) == 0);

    result = nn::svc::CreateDeviceAddressSpace(&handle, 0x80000000);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::MapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + 0x1000, size - 0x2000, addr + 0x1000,
            nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(result.IsSuccess());

    std::memset(reinterpret_cast<void*>(addr), 0, size);
    result = nn::sdmmc1::Read(reinterpret_cast<void*>(addr + 0x1000), size - 0x2000, TestPort, TestSectorIndex, TestNumSectors);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + 0x1000, size - 0x2000, addr + 0x1000);
    ASSERT_TRUE(result.IsSuccess());

    result = nn::svc::CloseHandle(handle);
    ASSERT_TRUE(result.IsSuccess());

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_TRUE(result.IsSuccess());
}

TEST(sMMU, Test7)
{
    uintptr_t addr;
    size_t size;

    nn::Result result;
    nn::svc::Handle handle;

    size = 0x01000000;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_TRUE(result.IsSuccess());
    // 4MBアラインされるはず
    ASSERT_TRUE((addr & (0x00400000 - 1)) == 0);

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, addr);
    ASSERT_TRUE(result.IsSuccess());
    // 4MBアラインされるはず
    ASSERT_TRUE(((physInfo.physicalAddress + (addr - physInfo.virtualAddress)) & (0x00400000 - 1)) == 0);

    for (int i = 0; i < 2u * 1024 * 1024 * 1024 / 4096; i++)
    {
        result = nn::svc::CreateDeviceAddressSpace(&handle, 0x80000000);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::AttachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::MapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr + 0x1000, size - 0x2000, addr + 0x1000,
                nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc3a, handle);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::DetachDeviceAddressSpace(nn::svc::DeviceName_Sdmmc4a, handle);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr + 0x1000, size - 0x2000, addr + 0x1000);
        ASSERT_TRUE(result.IsSuccess());

        result = nn::svc::CloseHandle(handle);
        ASSERT_TRUE(result.IsSuccess());
    }

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_TRUE(result.IsSuccess());
}
