﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_MmcDevice.h"
#include <nn/nn_Abort.h>
#include "sdmmc_Log.h"

namespace nn { namespace sdmmc1 {
namespace detail {

namespace
{
    // OCR Register ビット定義
    const uint32_t OcrCardPowerUpStatus = 0x1 << 31;
    const uint32_t OcrAccessModeShift = 29;
    const uint32_t OcrAccessModeMask = 0x3 << OcrAccessModeShift;
    const uint32_t OcrAccessModeSectorMode = 0x2 << OcrAccessModeShift;

    // RCA Register （0 以外の任意の値）
    const uint16_t Rca = 0x2;
}

Result MmcDevice::IssueCommandSendOpCond(uint32_t* pOutOcr, IHostController::BusPower busPower) const NN_NOEXCEPT
{
    // argument[31]    reserved
    // argument[30:29] Host support access mode, 10b : sector mode
    // argument[28:24] reserved
    // argument[23:15] 111 1111b : 2.7-3.6V
    // argument[14: 8] 1 1111 1111b : 2.0-2,6V
    // argument[7]     1b : 1.70-1.95V
    // argument[6:0]   reserved
    uint32_t argument = OcrAccessModeSectorMode;
    switch (busPower)
    {
    case IHostController::BusPower_1_8V:
        argument = argument | (0x1 << 7);
        break;
    case IHostController::BusPower_3_3V:
        argument = argument | (0x7F << 15);
        break;
    case IHostController::BusPower_Off:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    ResponseType responseType = ResponseType_R3;
    Command command(1, argument, responseType, false);

    IHostController* pHostController = BaseDevice::GetHostController();
    Result result = pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    pHostController->GetLastResponse(pOutOcr, sizeof(uint32_t), responseType);

    return ResultSuccess();
}

Result MmcDevice::IssueCommandSetRelativeAddr(uint16_t rca) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(rca > 0);

    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    // R1, このコマンドで状態遷移するため、state チェックは除外する
    uint32_t argument = static_cast<uint32_t>(rca) << 16;
    return BaseDevice::IssueCommandAndCheckR1(3, argument, false, DeviceState_Unknown);
}

Result MmcDevice::Activate() NN_NOEXCEPT
{
    // バス設定, クロック供給
    IHostController::BusPower busPower;
    BaseDevice::SetBusIdentificationMode(&busPower);

    // idle state へ遷移
    Result result = BaseDevice::IssueCommandGoIdleState();
    if (result.IsFailure())
    {
        return result;
    }

    // ready state へ遷移、High Capacity 有無取得
    bool isCompleted = false;
    while (true)
    {
        result = IssueCommandSendOpCond(&m_Ocr, busPower);
        if (result.IsFailure())
        {
            return result;
        }
        if ((m_Ocr & OcrCardPowerUpStatus) != 0)
        {
            isCompleted = true;
            if ((m_Ocr & OcrAccessModeMask) == OcrAccessModeSectorMode)
            {
                BaseDevice::SetHighCapacity(true);
            }
            else
            {
                BaseDevice::SetHighCapacity(false);
            }
            break;
        }
        // TODO: タイムアウト実装
    }

    // ident state へ遷移、CID 取得
    result = BaseDevice::IssueCommandAllSendCid(m_Cid, sizeof(m_Cid));
    if (result.IsFailure())
    {
        return result;
    }

    // RCA 設定、stby state へ遷移
    result = IssueCommandSetRelativeAddr(Rca);
    if (result.IsFailure())
    {
        return result;
    }
    BaseDevice::SetRca(Rca);

    // CSD 取得
    result = IssueCommandSendCsd(m_Csd, sizeof(m_Csd), Rca);
    if (result.IsFailure())
    {
        return result;
    }

    // Default Speed にクロックアップ
    BaseDevice::ChangeDeviceClockMode(IHostController::DeviceClockMode_SdCardDefaultSpeed);

    // tran state へ遷移
    result = BaseDevice::IssueCommandSelectCard(Rca);
    if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
