﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "sdmmc_Result.h"

namespace nn { namespace sdmmc1 {
namespace detail {

enum ResponseType {
    ResponseType_R0,
    ResponseType_R1,
    ResponseType_R2,
    ResponseType_R3,
    ResponseType_R6,
    ResponseType_R7
};

enum TransferDirection {
    TransferDirection_ReadFromDevice,
    TransferDirection_WriteToDevice
};

struct Command
{
    uint32_t m_CommandIndex;
    uint32_t m_CommandArgument;
    ResponseType m_ResponseType;
    bool m_IsBusy;

    Command(uint32_t commandIndex, uint32_t commandArgument, ResponseType responseType, bool isBusy) NN_NOEXCEPT : \
        m_CommandIndex(commandIndex), m_CommandArgument(commandArgument), m_ResponseType(responseType), m_IsBusy(isBusy) {}
};

struct TransferData
{
    void* m_pBuffer;
    size_t m_BlockSize;
    uint32_t m_NumBlocks;
    TransferDirection m_TransferDirection;
    bool m_IsMultiBlockTransfer;

    TransferData(void* pBuffer, size_t blockSize, uint32_t numBlocks, TransferDirection transferDirection, bool isMultiBlockTransfer) NN_NOEXCEPT : \
        m_pBuffer(pBuffer), m_BlockSize(blockSize), m_NumBlocks(numBlocks), m_TransferDirection(transferDirection), m_IsMultiBlockTransfer(isMultiBlockTransfer) {}
};

class IHostController
{
public:
    enum BusWidth {
        BusWidth_1Bit,
        BusWidth_4Bit,
        BusWidth_8Bit
    };

    enum BusPower {
        BusPower_Off,
        BusPower_1_8V,
        BusPower_3_3V
    };

    enum DeviceClockMode {
        DeviceClockMode_Identification,
        DeviceClockMode_SdCardDefaultSpeed,
        DeviceClockMode_MmcLegacySpeed,
        DeviceClockMode_SdCardHighSpeed,
        DeviceClockMode_MmcHighSpeed
    };

    virtual void Reset() NN_NOEXCEPT = 0;
    virtual void SetBusWidth(BusWidth busWidth) NN_NOEXCEPT = 0;
    virtual bool IsSupportBusPower(BusPower busPower) const NN_NOEXCEPT = 0;
    virtual void SetBusPower(BusPower busPower) NN_NOEXCEPT = 0;
    virtual void SetDeviceClock(uint32_t* pOutDeviceClockFrequencyKHz, DeviceClockMode deviceClockMode) NN_NOEXCEPT = 0;
    virtual void EnableDeviceClock() NN_NOEXCEPT = 0;
    virtual void DisableDeviceClock() NN_NOEXCEPT = 0;
    virtual Result IssueCommand(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT = 0;
    virtual void GetLastResponse(uint32_t* pOutResponse, size_t responseSize, ResponseType responseType) const NN_NOEXCEPT = 0;
    virtual void GetStopTransmissionResponse(uint32_t* pOutResponse, size_t responseSize) const NN_NOEXCEPT = 0;

    virtual Result IssueCommand(const Command* pCommand, TransferData* pTransferData) NN_NOEXCEPT
    {
        return IssueCommand(pCommand, pTransferData, nullptr);
    }

    virtual Result IssueCommand(const Command* pCommand) NN_NOEXCEPT
    {
        return IssueCommand(pCommand, nullptr, nullptr);
    }
};

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
