﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_ClockResetController.h"
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Abort.h>
#include <nn/svc/svc_Dd.h>

namespace nn { namespace sdmmc1 {
namespace detail {

namespace ClockResetController {

namespace
{
    struct ClockResetControllerRegisters
    {
        volatile uint32_t   rstSource0;         // 0x0
        volatile uint32_t   rstDevicesL0;       // 0x4
        volatile uint32_t   rstDevicesH0;       // 0x8
        volatile uint32_t   rstDevicesU0;       // 0xc
        volatile uint32_t   clkOutEnbL0;        // 0x10
        volatile uint32_t   clkOutEnbH0;        // 0x14
        volatile uint32_t   clkOutEnbU0;        // 0x18
        volatile uint8_t    dummy1[0x148];
        volatile uint32_t   clkSourceSdmmc4;    // 0x164
        volatile uint8_t    dummy2[0x54];
        volatile uint32_t   clkSourceSdmmc3;    // 0x1bc
    };

    #if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1))
        const uint64_t ClkRstPhysicalAddress = 0x60006000;
        ClockResetControllerRegisters* s_pRegisters = nullptr;

        // レジスタ CLK_RST_CONTROLLER_REG_CLK_OUT_ENB_U_0 ビット定義
        const uint32_t REG_CLK_OUT_ENB_U_0__SDMMC3 = 0x1 << 5;

        // レジスタ CLK_RST_CONTROLLER_REG_CLK_OUT_ENB_L_0 ビット定義
        const uint32_t REG_CLK_OUT_ENB_U_0__SDMMC4 = 0x1 << 15;

        class InitRegistesPointer
        {
        public:
            InitRegistesPointer()
            {
                uintptr_t clkRst;
                NN_ABORT_UNLESS(nn::svc::QueryIoMapping(&clkRst, ClkRstPhysicalAddress, 0x1000).IsSuccess());
                s_pRegisters = reinterpret_cast<ClockResetControllerRegisters*>(clkRst);
            }
        };
        InitRegistesPointer s_Init;
    #else
        ClockResetControllerRegisters* s_pRegisters = nullptr;
    #endif

}

bool IsEnableSourceClock(Module module) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

    switch (module)
    {
    #if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1))
        case Module_Sdmmc3:
            if ((s_pRegisters->clkOutEnbU0 & REG_CLK_OUT_ENB_U_0__SDMMC3) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        case Module_Sdmmc4:
            if ((s_pRegisters->clkOutEnbU0 & REG_CLK_OUT_ENB_U_0__SDMMC4) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
    #endif
    case Module_None:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void EnableSourceClock(Module module) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

    switch (module)
    {
    #if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1))
        case Module_Sdmmc3:
            s_pRegisters->clkOutEnbU0 |= REG_CLK_OUT_ENB_U_0__SDMMC3;
            return;
        case Module_Sdmmc4:
            s_pRegisters->clkOutEnbL0 |= REG_CLK_OUT_ENB_U_0__SDMMC4;
            return;
    #endif
    case Module_None:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void SetSourceClock(Module module) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

    switch (module)
    {
    #if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1))
        case Module_Sdmmc3:
            s_pRegisters->clkSourceSdmmc3 = 0x0U;   // PLLP 408MHz
            return;
        case Module_Sdmmc4:
            s_pRegisters->clkSourceSdmmc4 = 0x0U;   // PLLP 408MHz
            return;
    #endif
    case Module_None:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

} // namespace ClockResetController {

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
