﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "test_Common.h"
#include "sdmmc1/sdmmc_Common.h"
#include "sdmmc1/sdmmc_SdCard.h"
#include "sdmmc1/sdmmc_Mmc.h"
#include <nn/svc/svc_Dd.h>
#include <nn/nn_Abort.h>

namespace {
class SdmmcInitializer
{
public:
    SdmmcInitializer()
    {
#if defined (NN_BUILD_CONFIG_HARDWARE_NX)
        uintptr_t registersAddress;
        auto result = nn::svc::QueryIoMapping(&registersAddress, 0x70000000ull, 0x4000);
        NN_ABORT_UNLESS(result.IsSuccess());
        *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x3000) = 0x00002064;
        *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x3004) = 0x00002068;
        *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x3008) = 0x00002068;
        *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x300C) = 0x00002068;
        *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x3010) = 0x00002068;
        *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x3014) = 0x00002068;

        // PINMUX_AUX_DMIC3_CLK_0 出力設定
        uint32_t value = *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x30b4);
        value &= (~(0x1 << 6)); // E_INPUT Disable
        value &= (~(0x1 << 4)); // TRISTATE Disable
        value = (value & (~(0x3 << 2))) | (0x1 << 2); // PULL_DOWN
        *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x30b4) = value;
#endif
    }
};

class Sdmmc
{
public:
    Sdmmc(void* buffer, size_t size, uint64_t deviceAddressSpace, nn::sdmmc1::Port port)
        : m_Port(port), m_Activate(false), m_Buffer(buffer), m_Size(size)
    {
        static SdmmcInitializer sdmmcInit;
        nn::sdmmc1::BufferInfo info;
        info.pBuffer = m_Buffer;
        info.bufferSize = m_Size;
        info.bufferDeviceVirtualAddress = deviceAddressSpace;

        nn::sdmmc1::Initialize(port, &info, 1);
    }
    ~Sdmmc()
    {
        Deactivate();
        nn::sdmmc1::Finalize(m_Port);
    }

    bool Activate()
    {
        nn::Result result;
        switch(m_Port)
        {
        case nn::sdmmc1::Port_SdCard0:
            result = nn::sdmmc1::ActivateSdCard(m_Port, m_Buffer, m_Size);
            break;
        case nn::sdmmc1::Port_Mmc0:
            result = nn::sdmmc1::ActivateMmc(m_Port, m_Buffer, m_Size);
            break;
        default:
            return false;
        }
        m_Activate = result.IsSuccess();
        return m_Activate;
    }

    void Deactivate()
    {
        if (m_Activate)
        {
            nn::sdmmc1::Deactivate(m_Port);
        }
    }

private:
    nn::sdmmc1::Port m_Port;
    bool m_Activate;
    void* m_Buffer;
    size_t m_Size;
};

#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
const nn::svc::DeviceName DeviceName_SdCard = nn::svc::DeviceName_Sdmmc3a;
const nn::svc::DeviceName DeviceName_Mmc    = nn::svc::DeviceName_Sdmmc4a;
#else // (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || (NN_BUILD_CONFIG_HARDWARE_NX)
const nn::svc::DeviceName DeviceName_SdCard = nn::svc::DeviceName_Sdmmc1a;
const nn::svc::DeviceName DeviceName_Mmc    = nn::svc::DeviceName_Sdmmc4a;
#endif

}

