﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include "test_TestCondition.h"
#include "util_TestResource.h"
#include "test_TestSvc.h"
#include "util_TestAssert.h"
#include "util_TestProcess.h"
#include "util_TestIpc.h"
#include <cstring>

ConsumeHeapMemory::ConsumeHeapMemory()
    : m_HeapAddr(0)
    , m_HeapSize(0)
{
    Consume();
}

void ConsumeHeapMemory::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;
    for(m_HeapSize = 0;; m_HeapSize += HeapAlign)
    {
        result = nn::svc::SetHeapSize(&m_HeapAddr, m_HeapSize);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfMemory());
            m_HeapSize -= HeapAlign;
            result = nn::svc::SetHeapSize(&m_HeapAddr, m_HeapSize);
            m_IsConsumed = true;
            return;
        }
    }
    NN_ASSERT(false);
}

void ConsumeHeapMemory::Release()
{
    nn::Result result;
    result = nn::svc::SetHeapSize(&m_HeapAddr, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_HeapAddr = 0;
    m_HeapSize = 0;
    m_IsConsumed = false;
}

ConsumeHeapMemory::~ConsumeHeapMemory()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

void ConsumeHandleBase::Release()
{
    nn::Result result;

    for(int32_t i = 0; i < m_Count; i++)
    {
        if (m_pHandles[i] != nn::svc::INVALID_HANDLE_VALUE)
        {
            result = nn::svc::CloseHandle(m_pHandles[i]);
            NN_ASSERT_RESULT_SUCCESS(result);
        }
    }

    m_IsConsumed = false;
    m_Count = 0;
}

ConsumeHandle::ConsumeHandle()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}

void ConsumeHandle::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;

    uintptr_t pc = 0x1;
    uintptr_t sp = 0;
    uintptr_t param = 0;
    int32_t priority = nn::svc::LowestThreadPriority;
    int32_t idealCore = 0;

    m_Count = 0;

    for(int i = 0; m_Count < NumHandle && i < (NN_KERN_SLAB_OBJ_NUM_EVENT / 2); m_Count += 2, i++)
    {
        result = nn::svc::CreateEvent(&m_pHandles[m_Count], &m_pHandles[m_Count + 1]);
        if (result.IsFailure())
        {
            NN_ASSERT(result <= nn::svc::ResultMaxHandle());
            break;
        }
    }

    for (int i = 0; m_Count < NumHandle && i < (NN_KERN_SLAB_OBJ_NUM_SESSION / 2); m_Count += 2, i++)
    {
         result = nn::svc::CreateSession(&m_pHandles[m_Count], &m_pHandles[m_Count + 1], false, 0);
        if (result.IsFailure())
        {
            NN_ASSERT(result <= nn::svc::ResultMaxHandle());
            break;
        }
    }

    for (int i = 0; m_Count < NumHandle && i < (NN_KERN_SLAB_OBJ_NUM_THREAD / 2); m_Count++, i++)
    {
        result = nn::svc::CreateThread(&m_pHandles[m_Count], pc, param, sp, priority, idealCore);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

ConsumeHandle::~ConsumeHandle()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeThread::ConsumeThread()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}

void ConsumeThread::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;

    uintptr_t pc = 0x1;
    uintptr_t sp = 0;
    uintptr_t param = 0;
    int32_t priority = nn::svc::LowestThreadPriority;
    int32_t idealCore = 0;

    for(m_Count = 0; m_Count < NumHandle; m_Count++)
    {
        result = nn::svc::CreateThread(&m_pHandles[m_Count], pc, param, sp, priority, idealCore);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
}

ConsumeThread::~ConsumeThread()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeMemoryControlBlock::ConsumeMemoryControlBlock()
    : m_Process(nn::svc::INVALID_HANDLE_VALUE)
{
    Consume();
}

void ConsumeMemoryControlBlock::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;
    m_Count = 0;

    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);
    param.memoryNumPages = NN_KERN_SLAB_OBJ_NUM_SYS_MEMORY_BLOCK + 1;

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    result = nn::svc::CreateProcess(&m_Process, param, flags, DefaultCapabilityFlagNum);
    NN_ASSERT_RESULT_SUCCESS(result);

    const size_t Size = 0x1000;
    uintptr_t addr = param.memoryAddress;
    const uintptr_t EndAddr = param.memoryAddress + param.memoryNumPages * 0x1000;

    const nn::svc::MemoryPermission Permissions[] = {
        nn::svc::MemoryPermission_None,
        nn::svc::MemoryPermission_Read,
        nn::svc::MemoryPermission_ReadWrite,
        nn::svc::MemoryPermission_ReadExecute,
    };
    const int32_t PermSize = sizeof(Permissions) / sizeof(nn::svc::MemoryPermission);

    // 出来るだけメモリ管理をする
    while((addr + Size - 1) < EndAddr - 1)
    {
        result = nn::svc::SetProcessMemoryPermission(
                m_Process, addr, Size, Permissions[m_Count % PermSize]);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
        m_Count++;
        addr += Size;
    }

    Release();
    m_Count = -1;
}

void ConsumeMemoryControlBlock::Release()
{
    nn::Result result;
    result = nn::svc::CloseHandle(m_Process);
    NN_ASSERT_RESULT_SUCCESS(result);
    m_Count = 0;
    m_IsConsumed = false;
}

ConsumeMemoryControlBlock::~ConsumeMemoryControlBlock()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeEvent::ConsumeEvent()
    : m_Heap(NumHandle * (sizeof(nn::svc::Handle) + MsgSize))
{
    nn::Result result;
    m_MsgBuffer = m_Heap.GetAddress();
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress() + MsgSize * NumHandle);
    result = nn::svc::CreateSession(&m_ServerSession, &m_ClientSession, false, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    Consume();
}

void ConsumeEvent::Consume()
{
    NN_ASSERT(!m_IsConsumed);

    nn::Result result;

    for (m_Count = 0; m_Count < NumHandle; m_Count++)
    {
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &m_pHandles[m_Count], m_MsgBuffer + MsgSize * m_Count, MsgSize, m_ClientSession);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result ,nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

ConsumeEvent::~ConsumeEvent()
{
    if (m_IsConsumed)
    {
        Release();
    }

    nn::Result result;
    result = nn::svc::CloseHandle(m_ServerSession);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(m_ClientSession);
    NN_ASSERT_RESULT_SUCCESS(result);
}

ConsumeSharedMemory::ConsumeSharedMemory()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}

void ConsumeSharedMemory::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;

    const nn::svc::MemoryPermission Perm = nn::svc::MemoryPermission_ReadWrite;
    const size_t Size = 0x1000;

    for(m_Count = 0; m_Count < NumHandle; m_Count++)
    {
        result = nn::svc::CreateSharedMemory(&m_pHandles[m_Count], Size, Perm, Perm);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

ConsumeSharedMemory::~ConsumeSharedMemory()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeTransferMemory::ConsumeTransferMemory()
    : m_Heap(NumHandle * (AreaSize + sizeof(nn::svc::Handle)))
{
    m_Addr = m_Heap.GetAddress();
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress() + AreaSize * NumHandle);
    Consume();
}

void ConsumeTransferMemory::Consume()
{
    NN_ASSERT(!m_IsConsumed);

    nn::Result result;
    for(m_Count = 0; m_Count < NumHandle; m_Count++)
    {
        result = nn::svc::CreateTransferMemory(
                &m_pHandles[m_Count], m_Addr + AreaSize * m_Count, AreaSize, Perm);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

ConsumeTransferMemory::~ConsumeTransferMemory()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeDeviceAddressSpace::ConsumeDeviceAddressSpace()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}

void ConsumeDeviceAddressSpace::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;
    const uint64_t SpaceAddr = 0x80000000;
    const uint64_t Size = 0x80000000;

    for(m_Count = 0; m_Count < NumHandle; m_Count++)
    {
        result = nn::svc::CreateDeviceAddressSpace(&m_pHandles[m_Count], SpaceAddr, Size);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

ConsumeDeviceAddressSpace::~ConsumeDeviceAddressSpace()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeSession::ConsumeSession()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
    , m_NamedPort(PortName, NumHandle)
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}

void ConsumeSession::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;

    for(m_Count = 0; m_Count < NumHandle; m_Count++)
    {
        result = nn::svc::ConnectToNamedPort(&m_pHandles[m_Count], m_NamedPort.GetName());
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

void ConsumeSession::Release()
{
    nn::Result result;
    for (int32_t i = 0; i < m_Count; i++)
    {
        nn::svc::Handle tmpHandle;
        nn::svc::Handle portHandle = m_NamedPort.GetHandle();
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &portHandle, 1, 0);
        NN_ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::AcceptSession(&tmpHandle, portHandle);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(tmpHandle);
        NN_ASSERT_RESULT_SUCCESS(result);
    }

    ConsumeHandleBase::Release();
}

ConsumeSession::~ConsumeSession()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeProcess::ConsumeProcess()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}

void ConsumeProcess::Consume()
{
    NN_ASSERT(!m_IsConsumed);

    nn::Result result;
    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    for(m_Count = 0; m_Count < NumHandle; m_Count++)
    {
        result = nn::svc::CreateProcess(&m_pHandles[m_Count], param, flags, DefaultCapabilityFlagNum);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

ConsumeProcess::~ConsumeProcess()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumePort::ConsumePort()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}


void ConsumePort::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;

    for(m_Count = 0; 2 * m_Count < NumHandle; m_Count++)
    {
        result = nn::svc::CreatePort(&m_pHandles[2 * m_Count], &m_pHandles[2 * m_Count + 1], 1, false, 0);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

void ConsumePort::Release()
{
    m_Count *= 2;
    ConsumeHandleBase::Release();
}

ConsumePort::~ConsumePort()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeNamedPort::ConsumeNamedPort()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}

void ConsumeNamedPort::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;

    char name[3];
    name[2] = 0;

    for(m_Count = 0; m_Count < NumHandle; m_Count++)
    {
        name[1] = m_Count / CHAR_MAX + 1;
        name[0] = m_Count % (CHAR_MAX - 1) + 1;
        result = nn::svc::ManageNamedPort(&m_pHandles[m_Count], name, 1);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

void ConsumeNamedPort::Release()
{
    nn::Result result;
    char name[3];
    name[2] = 0;

    for(int32_t i = 0; i < m_Count; i++)
    {
        name[1] = i / CHAR_MAX + 1;
        name[0] = i % (CHAR_MAX - 1) + 1;

        nn::svc::Handle tmpHandle;
        result = nn::svc::ManageNamedPort(&tmpHandle, name, 0);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(m_pHandles[i]);
        NN_ASSERT_RESULT_SUCCESS(result);
    }
    m_IsConsumed = false;
    m_Count = 0;
}

ConsumeNamedPort::~ConsumeNamedPort()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeLightSession::ConsumeLightSession()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}

void ConsumeLightSession::Consume()
{
    NN_ASSERT(!m_IsConsumed);

    nn::Result result;

    for(m_Count = 0; m_Count * 2 < NumHandle; m_Count++)
    {
        result = nn::svc::CreateSession(&m_pHandles[m_Count * 2], &m_pHandles[m_Count * 2 + 1], true, 0);
        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

void ConsumeLightSession::Release()
{
    m_Count *= 2;
    ConsumeHandleBase::Release();
}

ConsumeLightSession::~ConsumeLightSession()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

ConsumeDebug::ConsumeDebug()
    : m_Heap(NumHandle * sizeof(nn::svc::Handle))
{
    m_pHandles = reinterpret_cast<nn::svc::Handle*>(m_Heap.GetAddress());
    Consume();
}

void ConsumeDebug::Consume()
{
    NN_ASSERT(!m_IsConsumed);
    nn::Result result;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 1 << 0); // EnableDebug

    for(m_Count = 0; m_Count * 2 < NumHandle; m_Count++)
    {
        result = nn::svc::CreateProcess(&m_pHandles[m_Count * 2 + 1], param, flags, DefaultCapabilityFlagNum + 1);
        NN_ASSERT_RESULT_SUCCESS(result);

        nn::Bit64 processId;
        result = nn::svc::GetProcessId(&processId, m_pHandles[m_Count * 2 + 1]);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::DebugActiveProcess(&m_pHandles[m_Count * 2], processId);

        if (result.IsFailure())
        {
            NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            result = nn::svc::CloseHandle(m_pHandles[m_Count * 2 + 1]);
            NN_ASSERT_RESULT_SUCCESS(result);
            m_IsConsumed = true;
            return;
        }
    }

    Release();
    m_Count = -1;
    return;
}

void ConsumeDebug::Release()
{
    m_Count *= 2;
    ConsumeHandleBase::Release();
}

ConsumeDebug::~ConsumeDebug()
{
    if (m_IsConsumed)
    {
        Release();
    }
}

