﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "util_TestResource.h"
#include <nn/svc/svc_Types.common.h>
#include <nn/util/util_IntrusiveList.h>

namespace {
const size_t TestMemorySize = 0x5000;

const nn::svc::MemoryState AllMemoryState[] = {
    nn::svc::MemoryState_Free,
    nn::svc::MemoryState_Io,
    nn::svc::MemoryState_Static,
    nn::svc::MemoryState_Code,
    nn::svc::MemoryState_CodeData,
    nn::svc::MemoryState_Normal,
    nn::svc::MemoryState_Shared,
    nn::svc::MemoryState_Alias,
    nn::svc::MemoryState_AliasCode,
#ifdef SUPPORT_ALIAS_CODE_DATA
    nn::svc::MemoryState_AliasCodeData,
#endif // SUPPORT_ALIAS_CODE_DATA
    nn::svc::MemoryState_Ipc,
    nn::svc::MemoryState_NonSecureIpc,
    nn::svc::MemoryState_NonDeviceIpc,
    nn::svc::MemoryState_Stack,
    nn::svc::MemoryState_ThreadLocal,
    nn::svc::MemoryState_Transfered,
    nn::svc::MemoryState_SharedTransfered,
    nn::svc::MemoryState_SharedCode,
    nn::svc::MemoryState_Inaccessible,
};

const int NumTestMemoryState = sizeof(AllMemoryState) / sizeof(nn::svc::MemoryState);

const uint32_t AllMemoryAttribute[] = {
    0,
    nn::svc::MemoryAttribute_Locked,
    nn::svc::MemoryAttribute_DeviceShared,
    nn::svc::MemoryAttribute_Uncached,
    nn::svc::MemoryAttribute_IpcLocked,
    nn::svc::MemoryAttribute_DeviceShared | nn::svc::MemoryAttribute_Uncached,
    nn::svc::MemoryAttribute_DeviceShared | nn::svc::MemoryAttribute_IpcLocked,

    // シーケンスとして作ることが出来ない
    // nn::svc::MemoryAttribute_Locked | *
    // nn::svc::MemoryAttribute_Uncached | nn::svc::MemoryAttribute_IpcLocked
};

const int NumTestMemoryAttribute = sizeof(AllMemoryAttribute) / sizeof(nn::svc::MemoryAttribute);

}

void CheckMemory(
        uintptr_t addr,
        nn::svc::MemoryState state,
        nn::svc::MemoryPermission permission,
        uint32_t attribute);

void CheckProcessMemory(
        nn::svc::Handle processHandle,
        uintptr_t addr,
        nn::svc::MemoryState state,
        nn::svc::MemoryPermission permission,
        uint32_t attribute);

void InitTestMemory();
bool GetFreeArea(uintptr_t* pOutAddr, size_t* pOutSize, size_t size);
bool GetProcessLargeFreeArea(uint64_t* pOutAddr, uint64_t size, nn::svc::Handle process, bool is64Bit);
bool GetLargeFreeArea(uint64_t* pOutAddr, uint64_t size);

bool GetReservedFreeMemoryArea(uintptr_t* pOutAddr, size_t* pOutSize);

class TestIpcMapArea
{
public:
    TestIpcMapArea()
    {
        Initialize(0, 0, false, 0);
    }
    ~TestIpcMapArea();

    void Initialize(const void* mapData, size_t mapSize, bool canWrite, int ipcType);

    const void* GetMapData() const { return m_pMapData; }
    uintptr_t GetMapAddress() const { return m_MapAddr; }
    size_t GetSize() const { return m_Size; }
    int GetType() const { return m_IpcType; }
    bool CanWrite() const { return m_CanWrite; }
    bool IsMapped() const { return m_IsMapped; }

    void Map();
    void Unmap();

private:
    const void* m_pMapData;
    uintptr_t m_MapAddr;
    size_t m_Size;
    int m_IpcType;
    bool m_CanWrite;
    bool m_IsMapped;

    nn::svc::Handle m_ServerSession;
    nn::svc::Handle m_ClientSession;
    nn::svc::Handle m_Thread;
};

class TestMemoryInfo
{
public:
    TestMemoryInfo() : m_End(true), m_HeapAddr(0), m_HeapSize(0), m_FreeAddr(g_FreeAreaBegin), m_FreeSize(g_FreeAreaEnd - g_FreeAreaBegin) {}
    virtual ~TestMemoryInfo() {}
    uintptr_t GetAddress() { return m_Addr; }
    size_t GetSize() { return m_Size; }
    nn::svc::MemoryState GetState() { return m_State; }
    nn::svc::MemoryPermission GetPermission() { return m_Permission; }
    uint32_t GetAttribute() { return m_Attribute; }
    bool CanWriteData() { return m_CanWriteData; }
    void CheckDefaultState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0) {}
    virtual void Close() {}
    void AddTest(TestMemoryInfo* test);
    int GetListNum() { return m_TestList.size(); }
    void SetSize(size_t size);
    void SetHeap(uintptr_t addr, size_t size);
    void SetFree(uintptr_t addr, size_t size);

    void GetTestListWithStates(TestMemoryInfo** pOut, const nn::svc::MemoryState* states, int numState);
    void GetTestListExceptStates(
            TestMemoryInfo** pOut, int numOut, const nn::svc::MemoryState* states, int numState);
    void GetTestListWithAttributes(TestMemoryInfo** pOut, const uint32_t* attrs, int numAttr);
    void GetTestListExceptAttributes(
            TestMemoryInfo** pOut, int numOut, const uint32_t* attrs, int numAttr);
private:
    nn::util::IntrusiveListNode m_Node;
    typedef nn::util::IntrusiveList<
        TestMemoryInfo,
        nn::util::IntrusiveListMemberNodeTraits<TestMemoryInfo, &TestMemoryInfo::m_Node> >
            TestList;
    TestList m_TestList;

protected:
    nn::svc::MemoryState m_State;
    nn::svc::MemoryPermission m_Permission;
    uint32_t m_Attribute;
    uintptr_t m_Addr;
    size_t m_Size;
    bool m_End;
    uintptr_t m_HeapAddr;
    size_t m_HeapSize;
    uintptr_t m_FreeAddr;
    size_t m_FreeSize;
    bool m_CanWriteData;
};

/*
   MemoryState
 */
class TestMemoryState : public TestMemoryInfo
{
public:
    virtual ~TestMemoryState() {}
    virtual void Initialize(const void* pData = nullptr, size_t size = 0) {}
    virtual void Close() {}
};

void GenerateMemoryStateList(TestMemoryInfo** pOut, uintptr_t bufferAddr, size_t bufferSize);
void GenerateMemoryAttributeList(TestMemoryInfo** pOut, uintptr_t bufferAddr, size_t bufferSize);

// Free
class TestFreeMemoryState : public TestMemoryState
{
public:
    TestFreeMemoryState();
    virtual ~TestFreeMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
};

// Io
class TestIoMemoryState : public TestMemoryState
{
public:
    TestIoMemoryState();
    virtual ~TestIoMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
};

// Static
class TestStaticMemoryState : public TestMemoryState
{
public:
    TestStaticMemoryState();
    virtual ~TestStaticMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
};

// Code
class TestCodeMemoryState : public TestMemoryState
{
public:
    TestCodeMemoryState();
    virtual ~TestCodeMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
};

// CodeData
class TestCodeDataMemoryState : public TestMemoryState
{
public:
    TestCodeDataMemoryState();
    virtual ~TestCodeDataMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
};

// Normal
class TestNormalMemoryState : public TestMemoryState
{
public:
    TestNormalMemoryState();
    virtual ~TestNormalMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
};

// Shared
class TestSharedMemoryState : public TestMemoryState
{
public:
    TestSharedMemoryState();
    virtual ~TestSharedMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    nn::svc::Handle handle;
};

// Alias
class TestAliasMemoryState : public TestMemoryState
{
public:
    TestAliasMemoryState();
    virtual ~TestAliasMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    uintptr_t m_FromAddr;
    uintptr_t m_ToAddr;
};

// AliasCode
class TestAliasCodeMemoryState : public TestMemoryState
{
public:
    TestAliasCodeMemoryState();
    virtual ~TestAliasCodeMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
    void SetPermission(nn::svc::MemoryPermission perm);
private:
    nn::svc::Handle m_Handle;
    nn::svc::Handle m_WritableEvent;
    nn::svc::Handle m_ReadableEvent;
};

// AliasCodeData
class TestAliasCodeDataMemoryState : public TestAliasCodeMemoryState
{
public:
    TestAliasCodeDataMemoryState();
    virtual ~TestAliasCodeDataMemoryState() {}
};

// Ipc
class TestIpcMemoryState : public TestMemoryState
{
public:
    TestIpcMemoryState();
    virtual ~TestIpcMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    TestIpcMapArea m_IpcMap;
};

// NonSecureIpc
class TestNonSecureIpcMemoryState : public TestMemoryState
{
public:
    TestNonSecureIpcMemoryState();
    virtual ~TestNonSecureIpcMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    TestIpcMapArea m_IpcMap;
};

// NonDeviceIpc
class TestNonDeviceIpcMemoryState: public TestMemoryState
{
public:
    TestNonDeviceIpcMemoryState();
    virtual ~TestNonDeviceIpcMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    TestIpcMapArea m_IpcMap;
};


// Stack
class TestStackMemoryState : public TestMemoryState
{
public:
    TestStackMemoryState();
    virtual ~TestStackMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    uintptr_t m_FromAddr;
    uintptr_t m_ToAddr;
};

// ThreadLocal
class TestThreadLocalMemoryState : public TestMemoryState
{
public:
    TestThreadLocalMemoryState();
    virtual ~TestThreadLocalMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
};

// Transfered
class TestTransferedMemoryState : public TestMemoryState
{
public:
    TestTransferedMemoryState();
    virtual ~TestTransferedMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    nn::svc::Handle m_Handle;
    uintptr_t m_FromAddr;
    uintptr_t m_ToAddr;
};

// SharedTransfered
class TestSharedTransferedMemoryState : public TestMemoryState
{
public:
    TestSharedTransferedMemoryState();
    virtual ~TestSharedTransferedMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    nn::svc::Handle m_Handle;
    uintptr_t m_FromAddr;
    uintptr_t m_ToAddr;
};

// SharedCode
class TestSharedCodeMemoryState : public TestMemoryState
{
public:
    TestSharedCodeMemoryState();
    virtual ~TestSharedCodeMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    TestProcess m_Process;
};

// Inaccessible
class TestInaccessibleMemoryState : public TestMemoryState
{
public:
    TestInaccessibleMemoryState();
    virtual ~TestInaccessibleMemoryState();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
};

class TestMemoryAttribute : public TestMemoryInfo
{
public:
    virtual ~TestMemoryAttribute() {}
    virtual void Initialize(const void* pData = nullptr, size_t size = 0) {}
    virtual void Close() {}
};

class TestNoneAttribute : public TestMemoryAttribute
{
public:
    TestNoneAttribute();
    virtual ~TestNoneAttribute();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();

    void SetNoneArea(uintptr_t addr, size_t size);
};

class TestLockedAttribute : public TestMemoryState
{
public:
    TestLockedAttribute();
    virtual ~TestLockedAttribute();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    nn::svc::Handle m_TransferHandle;
};

class TestDeviceSharedAttribute : public TestMemoryAttribute
{
public:
    TestDeviceSharedAttribute();
    virtual ~TestDeviceSharedAttribute();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    uint64_t m_AddressSpaceSize;
    uintptr_t m_DeviceAddress;
    nn::svc::Handle m_Process;
    nn::svc::Handle m_Device;
};

class TestUncachedAttribute : public TestMemoryAttribute
{
public:
    TestUncachedAttribute();
    virtual ~TestUncachedAttribute();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
};

class TestIpcLockedAttribute : public TestMemoryAttribute
{
public:
    TestIpcLockedAttribute();
    virtual ~TestIpcLockedAttribute();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    TestIpcMapArea m_IpcMap;
};

class TestDeviceSharedAndUncachedAttribute : public TestMemoryAttribute
{
public:
    TestDeviceSharedAndUncachedAttribute();
    virtual ~TestDeviceSharedAndUncachedAttribute();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    uint64_t m_AddressSpaceSize;
    uintptr_t m_DeviceAddress;
    nn::svc::Handle m_Process;
    nn::svc::Handle m_Device;
};

class TestDeviceSharedAndIpcLockedAttribute : public TestMemoryAttribute
{
public:
    TestDeviceSharedAndIpcLockedAttribute();
    virtual ~TestDeviceSharedAndIpcLockedAttribute();
    virtual void Initialize(const void* pData = nullptr, size_t size = 0);
    virtual void Close();
private:
    uint64_t m_AddressSpaceSize;
    uintptr_t m_DeviceAddress;
    nn::svc::Handle m_Process;
    nn::svc::Handle m_Device;

    TestIpcMapArea m_IpcMap;
};

