﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "util_TestLoader.h"
#include "test_Common.h"
#include "util_ProcessImageMemory.h"
#include "util_KipUtil.h"
#include "util_TestProcess.h"
#include <cstring>

TestLoader::TestLoader(const char* file, size_t size, uint64_t codeAddr)
    : mCodeAddress(codeAddr), mFilePtr(file), mFileSize(size), mCapability(nullptr), mNumCapability(0), mAdditionalCapability(nullptr), mNumAdditionalCapability(0)
{
    ReadKipHeader(&mKip, mFilePtr, mFileSize);
    size_t memSize = GetMemorySize(mKip);
    mPriority = mKip.GetPriority();
    mProcessorId = mKip.GetIdealProcessor();
    mStackSize = mKip.GetStackSize();
    mIs64Bit = mKip.Is64BitSpace();
    MakeProcessParameter(&mProcessHeader, mKip, mCodeAddress, memSize);
}

TestLoader::TestLoader(const char* file, size_t size)
    : mFilePtr(file), mFileSize(size), mCapability(nullptr), mNumCapability(0), mAdditionalCapability(nullptr), mNumAdditionalCapability(0)
{
    ReadKipHeader(&mKip, mFilePtr, mFileSize);
    size_t memSize = GetMemorySize(mKip);
    mPriority = mKip.GetPriority();
    mProcessorId = mKip.GetIdealProcessor();
    mStackSize = mKip.GetStackSize();
    mIs64Bit = mKip.Is64BitSpace();
    mCodeAddress = mIs64Bit? NN_SVC_ADDR_SMALL_MAP64_BEGIN: NN_SVC_ADDR_SMALL_MAP32_BEGIN;
    MakeProcessParameter(&mProcessHeader, mKip, mCodeAddress, memSize);
}


void TestLoader::SpawnProcess(nn::svc::Handle* pOutProcess)
{
    nn::Result result;

    ProcessImageMemory pim;
    nn::Bit32 flags[NumKipCapability + 2 + mNumAdditionalCapability];

    int numKipCapabilities = mKip.GetKernelCapabilitiesSize() / sizeof(nn::Bit32);
    if (!mCapability)
    {
        std::memcpy(flags, mKip.GetKernelCapabilities(), mKip.GetKernelCapabilitiesSize());
        MakeNo3Flag(&flags[numKipCapabilities], nn::svc::LowestThreadPriority, nn::svc::HighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[numKipCapabilities + 1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        numKipCapabilities += 2;
        if (mAdditionalCapability && mNumAdditionalCapability > 0)
        {
            std::memcpy(&flags[numKipCapabilities], mAdditionalCapability, mNumAdditionalCapability * sizeof(nn::Bit32));
            numKipCapabilities += mNumAdditionalCapability;
        }
    }
    const nn::Bit32* capability = mCapability ? mCapability : flags;
    int32_t  capabilityNum = mCapability ? mNumCapability : numKipCapabilities;

    result = nn::svc::CreateProcess(pOutProcess, mProcessHeader, capability, capabilityNum);
    NN_ASSERT_RESULT_SUCCESS(result);

    pim.LoadImage(*pOutProcess, mFilePtr, mFileSize, mKip, mCodeAddress);
    pim.SetPermission(*pOutProcess, mKip, mCodeAddress);
    pim.Unmap(*pOutProcess, mKip, mCodeAddress);
}

void TestLoader::StartProcess(nn::svc::Handle process)
{
    nn::Result result;

    result = nn::svc::StartProcess(process, mPriority, mProcessorId, mStackSize);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void TestLoader::SetAdditionalCapability(nn::Bit32* capability, int32_t num)
{
    mAdditionalCapability = capability;
    mNumAdditionalCapability = num;
}

