﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestIpc.h"
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <cstring>

namespace {

const int32_t MaxPointerDataNum = 13;
char g_Stack[2][DefaultStackSize] __attribute__((aligned(0x1000)));
nn::Bit64 g_Pid;

void TestDummyThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    nn::Result result;
    nn::Bit64 curPid;

    result = nn::svc::GetProcessId(&curPid, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    NN_ASSERT_RESULT_SUCCESS(result);

    NN_ASSERT(g_Pid == curPid);
}
}

void SendCopyHandle(nn::svc::Handle handle, nn::svc::Handle session)
{
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 1, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);

    ipcMsg.SetHandle(offset, handle);

    nn::Result result = nn::svc::SendSyncRequest(session);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void ReceiveCopyHandle(nn::svc::Handle* pOut, nn::svc::Handle session)
{
    nn::Result result;
    int32_t index;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::ReplyAndReceive(&index, &session, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    NN_ASSERT_RESULT_SUCCESS(result);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetTag() == 0x0001);
    NN_ASSERT(ipcHeader.GetPointerNum() == 0);
    NN_ASSERT(ipcHeader.GetSendNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
    NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
    NN_ASSERT(ipcHeader.GetRawNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(!ipcSpecial.GetProcessIdFlag());
    NN_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    *pOut = ipcMsg.GetHandle(offset);

    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(&index, &session, 0, session, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void SendMoveHandle(nn::svc::Handle handle, nn::svc::Handle session)
{
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 1, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);

    ipcMsg.SetHandle(offset, handle);

    nn::Result result = nn::svc::SendSyncRequest(session);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void ReceiveMoveHandle(nn::svc::Handle* pOut, nn::svc::Handle session)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &session, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetTag() == 0x0001);
    NN_ASSERT(ipcHeader.GetPointerNum() == 0);
    NN_ASSERT(ipcHeader.GetSendNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
    NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
    NN_ASSERT(ipcHeader.GetRawNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(!ipcSpecial.GetProcessIdFlag());
    NN_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    *pOut = ipcMsg.GetHandle(offset);

    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(&index, &session, 0, session, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void SendCopyHandleFromServer(nn::svc::Handle handle, nn::svc::Handle session)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 1, 0);

    ipcMsg.SetNull();
    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &session, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);

    ipcMsg.SetHandle(offset, handle);

    result = nn::svc::ReplyAndReceive(&index, &session, 0, session, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void ReceiveCopyHandleFromServer(nn::svc::Handle* pOut, nn::svc::Handle session)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(session);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetTag() == 0x0001);
    NN_ASSERT(ipcHeader.GetPointerNum() == 0);
    NN_ASSERT(ipcHeader.GetSendNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
    NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
    NN_ASSERT(ipcHeader.GetRawNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(!ipcSpecial.GetProcessIdFlag());
    NN_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    *pOut = ipcMsg.GetHandle(offset);
}

void SendMoveHandleFromServer(nn::svc::Handle handle, nn::svc::Handle session)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 0, 1);

    ipcMsg.SetNull();
    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &session, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);

    ipcMsg.SetHandle(offset, handle);

    result = nn::svc::ReplyAndReceive(&index, &session, 0, session, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void ReceiveMoveHandleFromServer(nn::svc::Handle* pOut, nn::svc::Handle session)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(session);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetTag() == 0x0001);
    NN_ASSERT(ipcHeader.GetPointerNum() == 0);
    NN_ASSERT(ipcHeader.GetSendNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
    NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
    NN_ASSERT(ipcHeader.GetRawNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(!ipcSpecial.GetProcessIdFlag());
    NN_ASSERT(ipcSpecial.GetCopyHandleNum() == 0);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 1);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    *pOut = ipcMsg.GetHandle(offset);
}


void SendServerProcessHandle(nn::svc::Handle session)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &session, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 1, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

    result = nn::svc::ReplyAndReceive(&index, &session, 0, session, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void ReceiveServerProcessHandle(nn::svc::Handle* pOut, nn::svc::Handle session)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(session);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetTag() == 0x0001);
    NN_ASSERT(ipcHeader.GetPointerNum() == 0);
    NN_ASSERT(ipcHeader.GetSendNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
    NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
    NN_ASSERT(ipcHeader.GetRawNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(!ipcSpecial.GetProcessIdFlag());
    NN_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    *pOut = ipcMsg.GetHandle(offset);
}

void SendClientProcessHandle(nn::svc::Handle session)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 1, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

    result = nn::svc::SendSyncRequest(session);
    NN_ASSERT_RESULT_SUCCESS(result);

}

void ReceiveClientProcessHandle(nn::svc::Handle* pOut, nn::svc::Handle session)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &session, 1, nn::svc::INVALID_HANDLE_VALUE, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetTag() == 0x0001);
    NN_ASSERT(ipcHeader.GetPointerNum() == 0);
    NN_ASSERT(ipcHeader.GetSendNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
    NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
    NN_ASSERT(ipcHeader.GetRawNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(!ipcSpecial.GetProcessIdFlag());
    NN_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    *pOut = ipcMsg.GetHandle(offset);

    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(&index, &session, 0, session, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}


void ServerSendData(nn::svc::Handle session, void* data, size_t size)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &session, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    int rawNum = (size + sizeof(nn::Bit32) - 1) / sizeof(nn::Bit32);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, rawNum, 0);

    int offset = ipcMsg.Set(ipcHeader);
    ipcMsg.SetRawArray(offset, data, size);

    result = nn::svc::ReplyAndReceive(&index, &session, 0, session, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void ClientReceiveData(void* buffer, size_t size, nn::svc::Handle session)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(session);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetTag() == 0x0001);
    NN_ASSERT(ipcHeader.GetPointerNum() == 0);
    NN_ASSERT(ipcHeader.GetSendNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
    NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    NN_ASSERT(ipcHeader.GetSpecialNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
    ::std::memcpy(buffer, &pMsgBuffer[offset], size);
}

void ServerReceiveData(void* buffer, size_t size, nn::svc::Handle session)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &session, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetTag() == 0x0001);
    NN_ASSERT(ipcHeader.GetPointerNum() == 0);
    NN_ASSERT(ipcHeader.GetSendNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
    NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    NN_ASSERT(ipcHeader.GetSpecialNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
    ::std::memcpy(buffer, &pMsgBuffer[offset], size);

    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(&index, &session, 0, session, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void ClientSendData(nn::svc::Handle session, void* data, size_t size)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    int rawNum = (size + sizeof(nn::Bit32) - 1) / sizeof(nn::Bit32);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, rawNum, 0);

    int offset = ipcMsg.Set(ipcHeader);
    ipcMsg.SetRawArray(offset, data, size);

    result = nn::svc::SendSyncRequest(session);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void CheckAsyncFailure(nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
    NN_ASSERT(ipcHeader.GetTag() == 0x0000);
    NN_ASSERT(ipcHeader.GetPointerNum() == 0x0);
    NN_ASSERT(ipcHeader.GetSendNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveNum() == 0);
    NN_ASSERT(ipcHeader.GetExchangeNum() == 0);
    NN_ASSERT(ipcHeader.GetRawNum() == 0);
    NN_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    NN_ASSERT(ipcHeader.GetSpecialNum() == 0);
}

nn::Bit16 GetIpcTag(nn::Bit32* pMsgBuffer)
{
    if (pMsgBuffer == nullptr)
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    }

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);

    return ipcHeader.GetTag();
}

void SetOnlyIpcTag(nn::Bit32* pMsgBuffer, nn::Bit16 ipcTag)
{
    if (pMsgBuffer == nullptr)
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    }
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcTag, 0, 0, 0, 0, 0, 0, 0);
    ipcMsg.Set(ipcHeader);
}

int AddProcessIdData(nn::svc::ipc::MessageBuffer* ipcMsg, int offset)
{
    return ipcMsg->SetProcessId(offset, 0);
}

int AddThreadHandleData(
        nn::svc::Handle* pOutThreadHandle, nn::svc::ipc::MessageBuffer* ipcMsg,
        int offset, uintptr_t sp)
{
    nn::Result result;
    uintptr_t pc = reinterpret_cast<uintptr_t>(TestDummyThread);
    uintptr_t param = 0;
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    result = nn::svc::GetProcessId(&g_Pid, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateThread(pOutThreadHandle, pc, param, sp, priority, idealCore);
    NN_ASSERT_RESULT_SUCCESS(result);


    return ipcMsg->SetHandle(offset, *pOutThreadHandle);
}

int AddPointerData(
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        const char** data, const size_t* dataSize, int32_t dataNum)
{
    for (int32_t i = 0; i < dataNum; i++)
    {
        nn::svc::ipc::MessageBuffer::PointerData pointData(data[i], dataSize[i], i);
        offset = ipcMsg->Set(offset, pointData);
    }
    return offset;
}

int AddMapData(
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        char* buffer, size_t bufferSize, uint8_t data)
{
    WriteMapData(buffer, bufferSize, data);
    return RegisterMapData(ipcMsg, offset, buffer, bufferSize);
}

int AddRawData(
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset, const char* buffer, size_t bufferSize)
{
    return ipcMsg->SetRawArray(offset, buffer, bufferSize);
}

int AddAllData(
        nn::svc::Handle* pOutCopyHandle, nn::svc::Handle* pOutMoveHandle,
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        char* sendDataBuffer, size_t sendDataSize, uint8_t sendDataChar,
        char* receiveDataBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        char* exchangeDataBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize)
{
    offset = AddProcessIdData(ipcMsg, offset);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    if (pOutCopyHandle)
    {
        offset = AddThreadHandleData(pOutCopyHandle, ipcMsg, offset, sp);
    }
    sp = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
    if (pOutMoveHandle)
    {
        offset = AddThreadHandleData(pOutMoveHandle, ipcMsg, offset, sp);
    }
    offset = AddPointerData(ipcMsg, offset, pointerData, pointerDataSizes, pointerDataNum);
    offset = AddMapData(ipcMsg, offset, sendDataBuffer, sendDataSize, sendDataChar);
    offset = AddMapData(ipcMsg, offset, receiveDataBuffer, receiveDataSize, receiveDataChar);
    offset = AddMapData(ipcMsg, offset, exchangeDataBuffer, exchangeDataSize, exchangeDataChar);
    offset = AddRawData(ipcMsg, offset, rawData, rawDataSize);
    return offset;
}

int AddAllData(
        nn::svc::Handle* pOutCopyHandle, nn::svc::Handle* pOutMoveHandle,
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        char* sendDataBuffer, size_t sendDataSize,
        char* receiveDataBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        char* exchangeDataBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize)
{
    offset = AddProcessIdData(ipcMsg, offset);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    if (pOutCopyHandle)
    {
        offset = AddThreadHandleData(pOutCopyHandle, ipcMsg, offset, sp);
    }
    sp = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
    if (pOutMoveHandle)
    {
        offset = AddThreadHandleData(pOutMoveHandle, ipcMsg, offset, sp);
    }
    offset = AddPointerData(ipcMsg, offset, pointerData, pointerDataSizes, pointerDataNum);
    if (sendDataBuffer != nullptr)
    {
        offset = RegisterMapData(ipcMsg, offset, sendDataBuffer, sendDataSize);
    }
    if (receiveDataBuffer != nullptr)
    {
        offset = AddMapData(ipcMsg, offset, receiveDataBuffer, receiveDataSize, receiveDataChar);
    }
    if (exchangeDataBuffer != nullptr)
    {
        offset = AddMapData(ipcMsg, offset, exchangeDataBuffer, exchangeDataSize, exchangeDataChar);
    }
    offset = AddRawData(ipcMsg, offset, rawData, rawDataSize);
    return offset;
}

int RegisterMapData(
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset,
        char* buffer, size_t bufferSize)
{
    nn::svc::ipc::MessageBuffer::MapData mapData(buffer, bufferSize);
    offset = ipcMsg->Set(offset, mapData);
    return offset;
}

void WriteMapData(char* buffer, size_t bufferSize, uint8_t data)
{
    if (bufferSize > 0)
    {
        uint8_t* ptr = reinterpret_cast<uint8_t*>(buffer);
        ptr[0] = data;
        ptr[bufferSize / 2] = data;
        ptr[bufferSize - 1] = data;
    }
}

void MakeOnlyProcessIdData(nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(true, 0, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);

    AddProcessIdData(&ipcMsg, offset);
}

void MakeOnlyCopyHandleData(nn::svc::Handle* pOutThreadHandle, nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 1, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);

    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    AddThreadHandleData(pOutThreadHandle, &ipcMsg, offset, sp);
}

void MakeOnlyMoveHandleData(nn::svc::Handle* pOutThreadHandle, nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 0, 1);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);

    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    AddThreadHandleData(pOutThreadHandle, &ipcMsg, offset, sp);
}

void MakeOnlyPointerWithIpcBufferData(
        nn::Bit32* pMsgBuffer, const char** data, const size_t* dataSize, int32_t dataNum)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, dataNum, 0, 0, 0, 0, 1);

    int offset = ipcMsg.Set(ipcHeader);
    AddPointerData(&ipcMsg, offset, data, dataSize, dataNum);
}

void MakeOnlyPointerWithOneUserBuffer(
        nn::Bit32* pMsgBuffer,
        const char** data, const size_t* dataSize, int32_t dataNum,
        char* buffer, size_t bufferSize)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, dataNum, 0, 0, 0, 0, 2);

    int offset = ipcMsg.Set(ipcHeader);
    offset = AddPointerData(&ipcMsg, offset, data, dataSize, dataNum);

    nn::svc::ipc::MessageBuffer::ReceiveListEntry recvEntry(buffer, bufferSize);
    offset = ipcMsg.Set(offset, recvEntry);
}

void MakeOnlyPointerWithMultiUserBuffer(
        nn::Bit32* pMsgBuffer,
        const char** data, const size_t* dataSize, int32_t dataNum,
        char** buffer, const size_t* bufferSize)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader
        ipcHeader(0x0001, 0, dataNum, 0, 0, 0, 0, 2 + dataNum);

    int offset = ipcMsg.Set(ipcHeader);
    offset = AddPointerData(&ipcMsg, offset, data, dataSize, dataNum);

    for (int32_t i = 0; i < dataNum; i++)
    {
        nn::svc::ipc::MessageBuffer::ReceiveListEntry recvEntry(buffer[i], bufferSize[i]);
        offset = ipcMsg.Set(offset, recvEntry);
    }
}

void MakeOnlyMapSendData(
        nn::Bit32* pMsgBuffer, char* buffer, size_t bufferSize, uint8_t data)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader
        ipcHeader(0x0001, 0, 0, 1, 0, 0, 0, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = AddMapData(&ipcMsg, offset, buffer, bufferSize, data);
}

void MakeOnlyMapReceiveData(
        nn::Bit32* pMsgBuffer, char* buffer, size_t bufferSize, uint8_t data)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader
        ipcHeader(0x0001, 0, 0, 0, 1, 0, 0, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = AddMapData(&ipcMsg, offset, buffer, bufferSize, data);
}

void MakeOnlyMapExchangeData(
        nn::Bit32* pMsgBuffer, char* buffer, size_t bufferSize, uint8_t data)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader
        ipcHeader(0x0001, 0, 0, 0, 0, 1, 0, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = AddMapData(&ipcMsg, offset, buffer, bufferSize, data);
}

void MakeOnlyRawData(nn::Bit32* pMsgBuffer, const char* buffer, size_t bufferSize)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    int32_t dataNum = (bufferSize + sizeof(nn::Bit32) - 1) / sizeof(nn::Bit32);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader
        ipcHeader(0x0001, 0, 0, 0, 0, 0, dataNum, 0);

    int offset = ipcMsg.Set(ipcHeader);
    offset = AddRawData(&ipcMsg, offset, buffer, bufferSize);
}

void MakeAllData(
        nn::svc::Handle* pOutCopyHandle, nn::svc::Handle* pOutMoveHandle,
        nn::Bit32* pMsgBuffer,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        char* sendDataBuffer, size_t sendDataSize, uint8_t sendDataChar,
        char* receiveDataBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        char* exchangeDataBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize,
        char** receivePointerDataBuffers, const size_t* receivePointerDataSizes, int32_t receiveListNum)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    int32_t rawDataNum = (rawDataSize + sizeof(nn::Bit32) - 1) / sizeof(nn::Bit32);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader
        ipcHeader(0x0001, 1, pointerDataNum, 1, 1, 1, rawDataNum, 2 + receiveListNum);
    int copyNum = (pOutCopyHandle) ? 1 : 0;
    int moveNum = (pOutMoveHandle) ? 1 : 0;
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(true, copyNum, moveNum);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = AddAllData(
                pOutCopyHandle, pOutMoveHandle,
                &ipcMsg, offset,
                pointerData, pointerDataSizes, pointerDataNum,
                sendDataBuffer, sendDataSize, sendDataChar,
                receiveDataBuffer, receiveDataSize, receiveDataChar,
                exchangeDataBuffer, exchangeDataSize, exchangeDataChar,
                rawData, rawDataSize);

    for (int32_t i = 0; i < receiveListNum; i++)
    {
        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvEntry(receivePointerDataBuffers[i], receivePointerDataSizes[i]);
        offset = ipcMsg.Set(offset, recvEntry);
    }
}

// 返信専用
void MakeAllData(
        nn::svc::Handle* pOutCopyHandle, nn::svc::Handle* pOutMoveHandle,
        nn::Bit32* pMsgBuffer,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        char* sendDataBuffer, size_t sendDataSize,
        char* receiveDataBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        char* exchangeDataBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize,
        char** receivePointerDataBuffers, const size_t* receivePointerDataSizes, int32_t receiveListNum)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    int32_t rawDataNum = (rawDataSize + sizeof(nn::Bit32) - 1) / sizeof(nn::Bit32);
    ipcMsg.SetNull();
    nn::svc::ipc::MessageBuffer::MessageHeader
        ipcHeader(0x0001, 1, pointerDataNum, 0, 0, 0, rawDataNum, 2 + receiveListNum);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(true, 1, 1);

    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = AddAllData(
                pOutCopyHandle, pOutMoveHandle,
                &ipcMsg, offset,
                pointerData, pointerDataSizes, pointerDataNum,
                nullptr, sendDataSize,
                nullptr, receiveDataSize, receiveDataChar,
                nullptr, exchangeDataSize, exchangeDataChar,
                rawData, rawDataSize);

    WriteMapData(receiveDataBuffer, receiveDataSize, receiveDataChar);
    WriteMapData(exchangeDataBuffer, exchangeDataSize, exchangeDataChar);

    for (int32_t i = 0; i < receiveListNum; i++)
    {
        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvEntry(receivePointerDataBuffers[i], receivePointerDataSizes[i]);
        offset = ipcMsg.Set(offset, recvEntry);
    }
}

#ifdef NO_GOOGLE_TEST
#define TEST_ASSERT(result) NN_ASSERT(result)
#define TEST_ASSERT_RESULT_SUCCESS(result) NN_ASSERT_RESULT_SUCCESS(result)
#else
#define TEST_ASSERT(result) ASSERT_TRUE(result)
#define TEST_ASSERT_RESULT_SUCCESS(result) ASSERT_RESULT_SUCCESS(result)
#endif

void CmpProcessIdData(nn::svc::Handle clientHandle, nn::Bit32* pMsgBuffer, int offset)
{
    nn::Bit64 pid;
    nn::Result result;

    result = nn::svc::GetProcessId(&pid, clientHandle);
    TEST_ASSERT_RESULT_SUCCESS(result);

    nn::Bit64* data = reinterpret_cast<nn::Bit64*>(&pMsgBuffer[offset]);
    TEST_ASSERT(*data == pid);
}

void CmpProcessIdData(nn::Bit64 pid, nn::Bit32* pMsgBuffer, int offset)
{
    nn::Result result;

    nn::Bit64* data = reinterpret_cast<nn::Bit64*>(&pMsgBuffer[offset]);
    TEST_ASSERT(*data == pid);
}

void CmpHandleData(const nn::svc::ipc::MessageBuffer* ipcMsg, int offset, bool isSuccess)
{
    nn::Result result;

    nn::svc::Handle handle = ipcMsg->GetHandle(offset);

    result = nn::svc::StartThread(handle);
    if (!isSuccess)
    {
        result = nn::svc::CloseHandle(handle);
        TEST_ASSERT_RESULT_SUCCESS(result);
        return;
    }
    TEST_ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    TEST_ASSERT_RESULT_SUCCESS(result);
    TEST_ASSERT(index == 0);

    result = nn::svc::CloseHandle(handle);
    TEST_ASSERT_RESULT_SUCCESS(result);
}

void CmpPointerData(
        const char** data, const size_t* dataSize, int32_t dataNum,
        const uintptr_t* bufferAddresses, const size_t* bufferSizes,
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset)
{
    for (int32_t i = 0; i < dataNum; i++)
    {
        nn::svc::ipc::MessageBuffer::PointerData pointData(offset + 2 * i, *ipcMsg);
        TEST_ASSERT(pointData.GetPointerIndex() == i);
        TEST_ASSERT(pointData.GetPointerSize() == dataSize[i]);
        TEST_ASSERT(bufferAddresses[i] <= pointData.GetPointerAddress());
        TEST_ASSERT(
                pointData.GetPointerAddress() + pointData.GetPointerSize()
                < bufferAddresses[i] + bufferSizes[i]);
        TEST_ASSERT(
                ::std::memcmp(
                    reinterpret_cast<void*>(pointData.GetPointerAddress()),
                    data[i],
                    dataSize[i])
                == 0);
    }
}

void CmpMap(const uint8_t* ptr, size_t bufferSize, uint8_t data)
{
    if (bufferSize == 0)
    {
        return;
    }
    TEST_ASSERT(ptr[0] == data);
    TEST_ASSERT(ptr[bufferSize / 2] == data);
    TEST_ASSERT(ptr[bufferSize - 1] == data);
}

void CmpMapData(
        uintptr_t* pOutAddr,
        size_t bufferSize, uint8_t data,
        nn::svc::ipc::MessageBuffer* ipcMsg, int offset)
{
    if (bufferSize == 0)
    {
        return;
    }

    nn::svc::ipc::MessageBuffer::MapData mapData(offset, *ipcMsg);
    TEST_ASSERT(mapData.GetDataSize() == bufferSize);

    uint8_t* ptr = reinterpret_cast<uint8_t*>(mapData.GetDataAddress());
    CmpMap(ptr, bufferSize, data);
    *pOutAddr = mapData.GetDataAddress();
}

void CmpRawData(const char* buffer, size_t bufferSize, nn::Bit32* pMsgBuffer, int offset)
{
    char* ptr = reinterpret_cast<char*>(&pMsgBuffer[offset]);
    TEST_ASSERT(::std::memcmp(ptr, buffer, bufferSize) == 0);
}

void CheckOnlyProcessIdData(nn::svc::Handle clientHandle, nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == 0);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 1);
    TEST_ASSERT(ipcSpecial.GetProcessIdFlag());
    TEST_ASSERT(ipcSpecial.GetCopyHandleNum() == 0);
    TEST_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    CmpProcessIdData(clientHandle, pMsgBuffer, offset);
}

void CheckOnlyProcessIdData(nn::Bit64 pid, nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == 0);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 1);
    TEST_ASSERT(ipcSpecial.GetProcessIdFlag());
    TEST_ASSERT(ipcSpecial.GetCopyHandleNum() == 0);
    TEST_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    CmpProcessIdData(pid, pMsgBuffer, offset);
}

void CheckOnlyCopyHandleData(nn::Bit32* pMsgBuffer, bool isSuccess)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == 0);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 1);
    TEST_ASSERT(!ipcSpecial.GetProcessIdFlag());
    TEST_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);
    TEST_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    CmpHandleData(&ipcMsg, offset, isSuccess);
}

void CheckOnlyMoveHandleData(nn::Bit32* pMsgBuffer, bool isSuccess)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == 0);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 1);
    TEST_ASSERT(!ipcSpecial.GetProcessIdFlag());
    TEST_ASSERT(ipcSpecial.GetCopyHandleNum() == 0);
    TEST_ASSERT(ipcSpecial.GetMoveHandleNum() == 1);

    int offset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    CmpHandleData(&ipcMsg, offset, isSuccess);
}

void CheckOnlyPointerWithIpcBufferData(
        const char** data, const size_t* dataSize, int32_t dataNum,
        nn::Bit32* pMsgBuffer, size_t bufferSize)
{
    uintptr_t bufferAddresses[MaxPointerDataNum];
    size_t bufferSizes[MaxPointerDataNum];

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == dataNum);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 1);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 0);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);

    for (int32_t i = 0; i < dataNum; i++)
    {
        bufferAddresses[i] = reinterpret_cast<uintptr_t>(pMsgBuffer);
        bufferSizes[i] = bufferSize;
    }
    CmpPointerData(data, dataSize, dataNum, bufferAddresses, bufferSizes, &ipcMsg, offset);
}

void CheckOnlyPointerWithOneUserBuffer(
        uintptr_t bufferAddr, size_t bufferSize,
        const char** data, const size_t* dataSize, int32_t dataNum,
        nn::Bit32* pMsgBuffer)
{
    uintptr_t bufferAddresses[MaxPointerDataNum];
    size_t bufferSizes[MaxPointerDataNum];
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == dataNum);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 2);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 0);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);

    for (int32_t i = 0; i < dataNum; i++)
    {
        bufferAddresses[i] = bufferAddr;
        bufferSizes[i] = bufferSize;
    }
    CmpPointerData(data, dataSize, dataNum, bufferAddresses, bufferSizes, &ipcMsg, offset);
}

void CheckOnlyPointerWithMultiUserBuffer(
        const uintptr_t* bufferAddr, const size_t* bufferSize,
        const char** data, const size_t* dataSize, int32_t dataNum,
        nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == dataNum);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == dataNum + 2);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 0);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);

    CmpPointerData(data, dataSize, dataNum, bufferAddr, bufferSize, &ipcMsg, offset);
}

void CheckOnlyMapSendData(
        uintptr_t* pOutAddr, size_t bufferSize, uint8_t data, nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == 0);
    TEST_ASSERT(ipcHeader.GetSendNum() == 1);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
    CmpMapData(pOutAddr, bufferSize, data, &ipcMsg, offset);
}

void CheckOnlyMapReceiveData(
        uintptr_t* pOutAddr, size_t bufferSize, uint8_t data, nn::Bit32* pMsgBuffer)
{
    NN_UNUSED(data);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == 0);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 1);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
    nn::svc::ipc::MessageBuffer::MapData mapData(offset, ipcMsg);
    TEST_ASSERT(mapData.GetDataSize() == bufferSize);
    *pOutAddr = mapData.GetDataAddress();
}

void CheckOnlyMapExchangeData(
        uintptr_t* pOutAddr, size_t bufferSize, uint8_t data, nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == 0);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 1);
    TEST_ASSERT(ipcHeader.GetRawNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
    CmpMapData(pOutAddr, bufferSize, data, &ipcMsg, offset);
}

void CheckOnlyRawData(const char* buffer, size_t bufferSize, nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
    int32_t dataNum = (bufferSize + sizeof(nn::Bit32) - 1) / sizeof(nn::Bit32);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == 0);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == dataNum);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 0);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 0);

    int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
    CmpRawData(buffer, bufferSize, pMsgBuffer, offset);
}


void CheckAllData(
        uintptr_t* pOutSendMapAddr, uintptr_t* pOutReceiveMapAddr, uintptr_t* pOutExchangeMapAddr,
        nn::svc::Handle clientHandle, bool isHandleAlive, bool doCheckMoveHandle,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        const uintptr_t* pointerBuffers, const size_t* pointerBufferSizes,
        size_t sendDataSize, uint8_t sendDataChar,
        size_t receiveDataSize, uint8_t receiveDataChar,
        size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize,
        nn::Bit32* pMsgBuffer)
{
    NN_UNUSED(receiveDataChar);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
    int32_t rawDataNum = (rawDataSize + sizeof(nn::Bit32) - 1) / sizeof(nn::Bit32);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == pointerDataNum);
    TEST_ASSERT(ipcHeader.GetSendNum() == 1);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 1);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 1);
    TEST_ASSERT(ipcHeader.GetRawNum() == rawDataNum);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == 2 + pointerDataNum);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 1);
    TEST_ASSERT(ipcSpecial.GetProcessIdFlag());
    TEST_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);
    if (doCheckMoveHandle)
    {
        TEST_ASSERT(ipcSpecial.GetMoveHandleNum() == 1);
    }
    else
    {
        TEST_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);
    }

    int specialOffset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);
    int mapOffset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
    int mapDelta =  nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32);

    int rawOffset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);

    // PID
    int tmpOffset = 0;
    CmpProcessIdData(clientHandle, pMsgBuffer, specialOffset);
    tmpOffset = 2;

    // Copy Handle
    CmpHandleData(&ipcMsg, specialOffset + tmpOffset, isHandleAlive); // Copy
    tmpOffset++;

    // Move Handle
    if (doCheckMoveHandle)
    {
        CmpHandleData(&ipcMsg, specialOffset + tmpOffset, isHandleAlive); // Copy
        tmpOffset++;
    }

    // Pointer
    CmpPointerData(
            pointerData, pointerDataSizes, pointerDataNum,
            pointerBuffers, pointerBufferSizes,
            &ipcMsg, specialOffset + tmpOffset);

    if (sendDataSize > 0)
    {
    // Send
        CmpMapData(pOutSendMapAddr, sendDataSize, sendDataChar, &ipcMsg, mapOffset);
    }
    else
    {
        pOutSendMapAddr = nullptr;
    }

    if (receiveDataSize > 0)
    {
        // Receive
        nn::svc::ipc::MessageBuffer::MapData receiveMapData(mapOffset + mapDelta, ipcMsg);
        TEST_ASSERT(receiveMapData.GetDataSize() == receiveDataSize);
        *pOutReceiveMapAddr= receiveMapData.GetDataAddress();
    }
    else
    {
        pOutReceiveMapAddr = nullptr;
    }

    if (exchangeDataSize > 0)
    {
        // Exchange
        CmpMapData(
                pOutExchangeMapAddr, exchangeDataSize, exchangeDataChar,
                &ipcMsg, mapOffset + mapDelta * 2);
    }
    else
    {
        pOutExchangeMapAddr = nullptr;
    }

    // Raw
    CmpRawData(rawData, rawDataSize, pMsgBuffer, rawOffset);
}

void CheckAllDataFromClient(
        nn::Bit64 pid, bool doCheckMoveHandle,
        const char** pointerData, const size_t* pointerDataSizes, int32_t pointerDataNum,
        const uintptr_t* pointerBuffers, const size_t* pointerBufferSizes,
        const uint8_t* sendMapBuffer, size_t sendDataSize, uint8_t sendDataChar,
        const uint8_t* recvMapBuffer, size_t receiveDataSize, uint8_t receiveDataChar,
        const uint8_t* exchMapBuffer, size_t exchangeDataSize, uint8_t exchangeDataChar,
        const char* rawData, size_t rawDataSize,
        nn::Bit32* pMsgBuffer)
{
    // Send はチェックしない
    NN_UNUSED(sendMapBuffer);
    NN_UNUSED(sendDataSize);
    NN_UNUSED(sendDataChar);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
    int32_t rawDataNum = (rawDataSize + sizeof(nn::Bit32) - 1) / sizeof(nn::Bit32);

    TEST_ASSERT(ipcHeader.GetTag() == 0x0001);
    TEST_ASSERT(ipcHeader.GetPointerNum() == pointerDataNum);
    TEST_ASSERT(ipcHeader.GetSendNum() == 0);
    TEST_ASSERT(ipcHeader.GetReceiveNum() == 0);
    TEST_ASSERT(ipcHeader.GetExchangeNum() == 0);
    TEST_ASSERT(ipcHeader.GetRawNum() == rawDataNum);
    TEST_ASSERT(ipcHeader.GetReceiveListNum() == pointerDataNum + 2);
    TEST_ASSERT(ipcHeader.GetSpecialNum() == 1);
    TEST_ASSERT(ipcSpecial.GetProcessIdFlag());
    TEST_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);
    if (doCheckMoveHandle)
    {
        TEST_ASSERT(ipcSpecial.GetMoveHandleNum() == 1);
    }
    else
    {
        TEST_ASSERT(ipcSpecial.GetMoveHandleNum() == 0);
    }

    int specialOffset = nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial);

    int rawOffset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);

    int tmpOffset = 0;
    // PID
    CmpProcessIdData(pid, pMsgBuffer, specialOffset);
    tmpOffset = 2;

    // Copy Handle
    CmpHandleData(&ipcMsg, specialOffset + tmpOffset, false); // Copy
    tmpOffset++;

    // Move Handle
    if (doCheckMoveHandle)
    {
        CmpHandleData(&ipcMsg, specialOffset + tmpOffset, false); // Move
        tmpOffset++;
    }

    // Pointer
    CmpPointerData(
            pointerData, pointerDataSizes, pointerDataNum,
            pointerBuffers, pointerBufferSizes,
            &ipcMsg, specialOffset + tmpOffset);

    // Send
    // 特になし

    // Receive
    CmpMap(recvMapBuffer, receiveDataSize, receiveDataChar);

    // Exchange
    CmpMap(exchMapBuffer, exchangeDataSize, exchangeDataChar);

    // Raw
    CmpRawData(rawData, rawDataSize, pMsgBuffer, rawOffset);
}

void GetMapAddr(uintptr_t* pOutAddr, nn::svc::ipc::MessageBuffer* ipcMsg, int offset)
{
    nn::svc::ipc::MessageBuffer::MapData mapData(offset, *ipcMsg);
    *pOutAddr = mapData.GetDataAddress();
}

#ifdef ENABLE_SHOW_IPC
void ShowIpcHeader(nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_LOG("+++ IPC Header +++\n");
    NN_LOG("Tag: 0x%x\n", ipcHeader.GetTag());
    NN_LOG("PointerNum: 0x%x\n", ipcHeader.GetPointerNum());
    NN_LOG("SendNum: 0x%x\n", ipcHeader.GetSendNum());
    NN_LOG("ReceiveNum: 0x%x\n", ipcHeader.GetReceiveNum());
    NN_LOG("ExchangeNum: 0x%x\n", ipcHeader.GetExchangeNum());
    NN_LOG("RawNum: 0x%x\n", ipcHeader.GetRawNum());
    NN_LOG("ReceiveListNum: 0x%x\n", ipcHeader.GetReceiveListNum());
    NN_LOG("SpecialNum: 0x%x\n", ipcHeader.GetSpecialNum());

    if (ipcHeader.GetSpecialNum() > 0)
    {
        NN_LOG("ProcessIdFlag: 0x%x\n", ipcSpecial.GetProcessIdFlag());
        NN_LOG("CopyHandleNum: 0x%x\n", ipcSpecial.GetCopyHandleNum());
        NN_LOG("MoveHandleNum: 0x%x\n", ipcSpecial.GetMoveHandleNum());
    }
    NN_LOG("--- IPC Header ---\n");
}
#endif

