﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "test_TestSvc.h"
#include <nn/svc/svc_Result.h>
#ifndef NO_GOOGLE_TEST
#include <nnt/gtest/gtest.h>
#endif

void TestLog(const char *pFormat, ...);

#ifdef NN_LOG
#undef NN_LOG
#define NN_LOG(...) do { TestLog(__VA_ARGS__); } while(false)
#endif

void TestAssertDump(const char* filename, int lineNo, nn::Result* result);
void TestAssertDump(const char* filename, int lineNo, nn::Bit64 val1, nn::Bit64 val2);

inline void DebugResult(nn::Result result)
{
    NN_LOG("[Result]: Module=%d, Description=%d\n", result.GetModule(), result.GetDescription());
}

#define ENABLE_ASSERT_LOG
#define ENABLE_ASSERT_LOOP
#define ENABLE_ASSERT_DETAIL_LOG

#ifdef ENABLE_ASSERT_LOOP
#define END_TEST() do { asm volatile ("1: b 1b"); } while (false)
#else
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
#define END_TEST() do { asm volatile ("bkpt 1"); } while (false)
#elif defined NN_BUILD_CONFIG_CPU_ARM_V8A
#if defined NN_BUILD_CONFIG_ABI_ILP32
#define END_TEST() do { asm volatile ("bkpt 1"); } while (false)
#elif defined NN_BUILD_CONFIG_ABI_LP64
#define END_TEST() do { asm volatile ("hlt 0"); } while (false)
#endif // NN_BUILD_CONFIG_ABI
#endif // NN_BUILD_CONFIG_CPU
#endif // ENABLE_ASSERT_LOOP

#ifdef ENABLE_ASSERT_LOG
#ifdef ADD_FAILURE
#define SPAWN_GTEST_ASSERT_ERROR() do \
{\
    nn::svc::MemoryInfo blockInfo; \
    nn::svc::PageInfo pageInfo; \
    nn::Result result = nn::svc::QueryMemory(&blockInfo, &pageInfo, reinterpret_cast<uintptr_t>(&blockInfo)); \
    if (result.IsFailure()) \
    {\
        DebugResult(result); \
        END_TEST(); \
    }\
    if (blockInfo.state == nn::svc::MemoryState_Stack) \
    {\
        ADD_FAILURE(); \
    }\
    else \
    {\
        END_TEST(); \
    }\
} while (false)
#else
#define SPAWN_GTEST_ASSERT_ERROR() do { return; } while (false)
#endif

#define TEST_ASSERT_RESULT_LOG(result) do \
{\
    TestAssertDump(__FILE__, __LINE__, result); \
} while (false)

#define TEST_ASSERT_VAL_LOG(val1, val2) do \
{\
    TestAssertDump(__FILE__, __LINE__, val1, val2); \
} while (false)

#else
#define TEST_ASSERT_RESULT_LOG(result) do {} while (false)
#define TEST_ASSERT_VAL_LOG(val1, val2) do {} while (false)
#endif

class TestIpcAssertObject
{
public:
    TestIpcAssertObject(nn::svc::Handle handle, nn::Bit32* pMsgBuffer, size_t size)
        : m_Log(true)
        , m_Handle(handle)
        , m_pMsgBuffer(pMsgBuffer)
        , m_Size(size)
    {
    }

    ~TestIpcAssertObject()
    {
        if (m_Log)
        {
            OutputLog();
            END_TEST();
        }
    }

    void Cancel()
    {
        m_Log = false;
    }

    void OutputLog();

private:
    bool m_Log;
    nn::svc::Handle m_Handle;
    nn::Bit32* m_pMsgBuffer;
    size_t m_Size;
};

class TestMemoryAreaObject
{
public:
    TestMemoryAreaObject(uint64_t addr, nn::svc::Handle handle = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS)
        : m_Log(true), m_Address(addr), m_Handle(handle)
    {
    }

    ~TestMemoryAreaObject()
    {
        if (m_Log)
        {
            OutputLog();
            END_TEST();
        }
    }

    void Cancel()
    {
        m_Log = false;
    }

    void OutputLog();

private:
    bool m_Log;
    uint64_t m_Address;
    nn::svc::Handle m_Handle;
};

#ifdef ASSERT_TRUE
#undef ASSERT_TRUE
#endif

#ifdef ASSERT_EQ
#undef ASSERT_EQ
#endif

#ifdef ASSERT_NE
#undef ASSERT_NE
#endif

#ifdef ASSERT_LT
#undef ASSERT_LT
#endif

#ifdef ASSERT_LE
#undef ASSERT_LE
#endif

#ifdef ASSERT_GT
#undef ASSERT_GT
#endif

#ifdef ASSERT_GE
#undef ASSERT_GE
#endif

#ifdef NN_ASSERT
#undef NN_ASSERT
#endif

#ifdef NN_ASSERT_EQUAL
#undef NN_ASSERT_EQUAL
#endif

#ifdef NN_ASSERT_NOT_EQUAL
#undef NN_ASSERT_NOT_EQUAL
#endif

#ifdef NN_ASSERT_LESS
#undef NN_ASSERT_LESS
#endif

#ifdef NN_ASSERT_LESS_EQUAL
#undef NN_ASSERT_LESS_EQUAL
#endif

#ifdef NN_ASSERT_GREATER
#undef NN_ASSERT_GREATER
#endif

#ifdef NN_ASSERT_GREATER_EQUAL
#undef NN_ASSERT_GREATER_EQUAL
#endif



#define ASSERT_TRUE(condition) do \
{\
    if (!(condition)) \
    {\
        TEST_ASSERT_RESULT_LOG(nullptr); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    }\
} while (false)

#define ASSERT_EQ(val1, val2) do \
{\
    if ( ! ((val1) == (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    }\
} while (false)

#define ASSERT_NE(val1, val2) do \
{\
    if ( ! ((val1) != (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    }\
} while (false)

#define ASSERT_LT(val1, val2) do \
{\
    if ( ! ((val1) < (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    }\
} while (false)

#define ASSERT_LE(val1, val2) do \
{\
    if ( ! ((val1) <= (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    }\
} while (false)

#define ASSERT_GT(val1, val2) do \
{\
    if ( ! ((val1) > (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    }\
} while (false)

#define ASSERT_GE(val1, val2) do \
{\
    if ( ! ((val1) >= (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    }\
} while (false)

#define ASSERT_RESULT_SUCCESS(result) do \
{\
    if (result.IsFailure()) \
    { \
        TEST_ASSERT_RESULT_LOG(&result); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    } \
} while (false)

#define ASSERT_RESULT_FAILURE(result) do \
{\
    if (!result.IsFailure()) \
    { \
        TEST_ASSERT_RESULT_LOG(&result); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    } \
} while (false)

#define ASSERT_RESULT_FAILURE_VALUE(result, expectResult) do \
{\
    ASSERT_RESULT_FAILURE(result); \
    if (!(result <= expectResult)) \
    { \
        TEST_ASSERT_RESULT_LOG(&result); \
        SPAWN_GTEST_ASSERT_ERROR(); \
    } \
} while (false)

#define NN_ASSERT(condition, ...) do \
{\
    if (!(condition)) \
    {\
        TEST_ASSERT_RESULT_LOG(nullptr); \
        END_TEST(); \
    }\
} while (false)

#define NN_ASSERT_EQUAL(val1, val2) do \
{\
    if ( ! ((val1) == (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        END_TEST(); \
    }\
} while (false)

#define NN_ASSERT_NOT_EQUAL(val1, val2) do \
{\
    if ( ! ((val1) != (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        END_TEST(); \
    }\
} while (false)

#define NN_ASSERT_LESS(val1, val2) do \
{\
    if ( ! ((val1) < (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        END_TEST(); \
    }\
} while (false)

#define NN_ASSERT_LESS_EQUAL(val1, val2) do \
{\
    if ( ! ((val1) <= (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        END_TEST(); \
    }\
} while (false)

#define NN_ASSERT_GREATER(val1, val2) do \
{\
    if ( ! ((val1) > (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        END_TEST(); \
    }\
} while (false)

#define NN_ASSERT_GREATER_EQUAL(val1, val2) do \
{\
    if ( ! ((val1) >= (val2)) ) \
    {\
        TEST_ASSERT_VAL_LOG((val1), (val2)); \
        END_TEST(); \
    }\
} while (false)

#define NN_ASSERT_RESULT_SUCCESS(result) do \
{\
    if (result.IsFailure()) \
    { \
        TEST_ASSERT_RESULT_LOG(&result); \
        END_TEST(); \
    } \
} while (false)

#define NN_ASSERT_RESULT_FAILURE(result) do \
{\
    if (!result.IsFailure()) \
    { \
        TEST_ASSERT_RESULT_LOG(&result); \
        END_TEST(); \
    } \
} while (false)

#define NN_ASSERT_RESULT_FAILURE_VALUE(result, expectResult) do \
{\
    if (!(result <= expectResult)) \
    { \
        TEST_ASSERT_RESULT_LOG(&result); \
        END_TEST(); \
    } \
} while (false)

#define ASSERT_NO_FATAL_FAILURE(func) do \
{\
    func; \
    if (HasFatalFailure()) \
    {\
        return; \
    }\
} while (false)

#ifdef ENABLE_DEBUG_PRINT
#define DEBUG_PRINT(...) NN_LOG(__VA_ARGS__)
#else
#define DEBUG_PRINT(...) do {} while(false)
#endif

extern nn::Bit32 g_ClientBeforeData[DefaultUserBufferSize / sizeof(nn::Bit32)];
extern nn::Bit32 g_ClientAfterData[DefaultUserBufferSize / sizeof(nn::Bit32)];
extern nn::Bit32 g_ServerBeforeData[DefaultUserBufferSize / sizeof(nn::Bit32)];
extern nn::Bit32 g_ServerAfterData[DefaultUserBufferSize / sizeof(nn::Bit32)];

#define CopyIpcBuffer(buffer, pMsgBuffer, size) do \
{\
    std::memcpy(buffer, pMsgBuffer, size); \
} while (false)

#define BeforeClientIpc(pMsgBuffer, size) do \
{\
    size_t bufSize = sizeof(g_ClientBeforeData); \
    std::memset(g_ClientBeforeData, 0, bufSize); \
    CopyIpcBuffer(g_ClientBeforeData, pMsgBuffer, bufSize > size ? size : bufSize); \
} while (false)

#define AfterClientIpc(pMsgBuffer, size) do \
{\
    size_t bufSize = sizeof(g_ClientAfterData); \
    std::memset(g_ClientAfterData, 0, bufSize); \
    CopyIpcBuffer(g_ClientAfterData, pMsgBuffer, bufSize > size ? size : bufSize); \
} while (false)

#define BeforeServerIpc(pMsgBuffer, size) do \
{\
    size_t bufSize = sizeof(g_ServerBeforeData); \
    std::memset(g_ServerBeforeData, 0, bufSize); \
    CopyIpcBuffer(g_ServerBeforeData, pMsgBuffer, bufSize > size ? size : bufSize); \
} while (false)

#define AfterServerIpc(pMsgBuffer, size) do \
{\
    size_t bufSize = sizeof(g_ServerAfterData); \
    std::memset(g_ServerAfterData, 0, bufSize); \
    CopyIpcBuffer(g_ServerAfterData, pMsgBuffer, bufSize > size ? size : bufSize); \
} while (false)

