﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_Tcb.h>

namespace {

char g_Buffer[2][DefaultStackSize] __attribute__((aligned(0x1000)));

void SignalAndCancelThread(nn::svc::Handle* readableEvent)
{
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t index;
    nn::svc::Handle handles[1] = { *readableEvent };

    result = nn::svc::WaitSynchronization(&index, handles, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    result = nn::svc::ClearEvent(handles[0]);
    ASSERT_RESULT_SUCCESS(result);

    // Cancel とタイムアウトでは、タイムアウトが優先されるため、ns を-1にする
    result = nn::svc::WaitSynchronization(&index, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
}

void SignalAndTimeoutThread(nn::svc::Handle* readableEvent)
{
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t index;
    nn::svc::Handle handles[1] = { *readableEvent };

    result = nn::svc::WaitSynchronization(&index, handles, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    result = nn::svc::ClearEvent(handles[0]);
    ASSERT_RESULT_SUCCESS(result);
}

void CancelAndTimeoutThread(nn::svc::Handle* readableEvent)
{
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t index;
    nn::svc::Handle handles[1] = { *readableEvent };

    result = nn::svc::WaitSynchronization(&index, handles, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // Cancel とタイムアウトでは、タイムアウトが優先されるため、ns を-1にする
    result = nn::svc::WaitSynchronization(&index, handles, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
}

void WaitTwoThread(uintptr_t addr)
{
    AutoThreadExit autoExit;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(addr);

    int32_t index;
    nn::Result result = nn::svc::WaitSynchronization(&index, handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace

TEST(WaitSynchronization, OrderTestWithSignalAndCancel)
{
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    uintptr_t pc = reinterpret_cast<uintptr_t>(SignalAndCancelThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer[0] + sizeof(g_Buffer[0]));
    uintptr_t param = reinterpret_cast<uintptr_t>(&readableEvent);
    int32_t currentPriority;

    result = nn::svc::GetThreadPriority(
            &currentPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetThreadPriority(
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, TestHighestThreadPriority);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-38 (同じコア), 21-39 (違うコア)
    // 待ち状態で同期オブジェクトのシグナルとキャンセルが重なった時に、
    // 同期オブジェクトのシグナルが優先される。
    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority + 1;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            result = nn::svc::CreateThread(&handle, pc, param, sp,
                    priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CancelSynchronization(handle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::SignalEvent(writableEvent);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, currentPriority);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, OrderTestWithSignalAndTimeout)
{
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    uintptr_t pc = reinterpret_cast<uintptr_t>(SignalAndTimeoutThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer[0] + sizeof(g_Buffer[0]));
    uintptr_t param = reinterpret_cast<uintptr_t>(&readableEvent);
    int32_t currentPriority;

    result = nn::svc::GetThreadPriority(
            &currentPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetThreadPriority(
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, TestHighestThreadPriority);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-40 (同じコア), 21-41 (違うコア)
    // 待ち状態で同期オブジェクトのシグナルとタイムアウトが重なった時に、
    // 同期オブジェクトのシグナルが優先される。
    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority + 1;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            result = nn::svc::CreateThread(&handle, pc, param, sp,
                    priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::SignalEvent(writableEvent);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, currentPriority);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, OrderTestWithCancelAndTimeout)
{
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    uintptr_t pc = reinterpret_cast<uintptr_t>(CancelAndTimeoutThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer[0] + sizeof(g_Buffer[0]));
    uintptr_t param = reinterpret_cast<uintptr_t>(&readableEvent);
    int32_t currentPriority;

    result = nn::svc::GetThreadPriority(
            &currentPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetThreadPriority(
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, TestHighestThreadPriority);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 21-42(同じコア), 21-43(違うコア)
    // 待ち状態でCancelSynchronization とタイムアウトが重なった時に、
    // タイムアウトが優先される。
    for (int32_t priority = TestHighestThreadPriority + 1;
            priority <= TestLowestThreadPriority;
            priority++)
    {
        result = nn::svc::CreateThread(&handle, pc, param, sp,
                priority, nn::svc::IdealCoreUseProcessValue);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CancelSynchronization(handle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(handle);
        ASSERT_RESULT_SUCCESS(result);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, currentPriority);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(WaitSynchronization, WaitMultiThreadTest)
{
    int32_t middlePriority = (TestLowestThreadPriority - TestHighestThreadPriority) / 2 + TestHighestThreadPriority;
    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority; priority <= TestLowestThreadPriority; priority++)
        {
            nn::Result result;
            nn::svc::Handle handle;
            nn::svc::Handle writableEvent;
            nn::svc::Handle readableEvent;

            result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
            ASSERT_RESULT_SUCCESS(result);

            AutoHandleClose readerCloser(readableEvent);
            AutoHandleClose writerCloser(writableEvent);

            uintptr_t pc = reinterpret_cast<uintptr_t>(WaitTwoThread);
            uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer[0] + sizeof(g_Buffer[0]));
            uintptr_t param = reinterpret_cast<uintptr_t>(&readableEvent);

            TestThread client1(pc, param, sp, priority, idealCore);
            client1.Start();

            sp = reinterpret_cast<uintptr_t>(g_Buffer[1] + sizeof(g_Buffer[1]));
            TestThread client2(pc, param, sp, middlePriority, 0);
            client2.Start();

            result = nn::svc::SignalEvent(writableEvent);
            ASSERT_RESULT_SUCCESS(result);

            int32_t index;
            handle = client1.GetHandle();
            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            handle = client2.GetHandle();
            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}
