﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/os.h>

extern "C" void nnMain();

namespace {
const uintptr_t WaitKey = 0;
const nn::Bit32 InvalidValue = 0xdeadbeaf;

nn::Bit32 g_Value;
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));

void SignalThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;

    NN_ASSERT(g_Value == 0);
    g_Value = InvalidValue;
    nn::svc::SignalProcessWideKey(WaitKey, 1);
}

void LockThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Value);
    nn::Bit32 newValue = 1;

    g_Value = arg | nn::svc::Handle::WaitMask;
    result = nn::svc::ArbitrateLock(nn::svc::Handle(arg), addr, newValue);
    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(g_Value == newValue);
}

} // namespace

TEST(WaitProcessWideKeyAtomic, ReleaseWithInvalidValue)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Value);
    nn::Bit32 newValue = 1;

    // TEST 138-17
    // 待機が解除されたときに、addr で指定されるアドレスに、
    // 0 でも Thread 値でもない値が格納されていたら、ResultInvalidState を返す
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(SignalThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = 0;
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        result = nn::svc::WaitProcessWideKeyAtomic(addr, WaitKey, newValue, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());
        ASSERT_TRUE(g_Value == InvalidValue);
    }
}

TEST(WaitProcessWideKeyAtomic, ArbitrateUnlockTest)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Value);
    nn::Bit32 newValue = 2;

    // TEST 138-18
    // カレントスレッドの待機列にスレッドが存在する状態で呼び出すとき、
    // ArbitrateUnlock の振る舞いと同じ振る舞いをする
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(LockThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = nn::os::GetCurrentThread()->_handle;
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        nn::svc::SleepThread(100 * 1000 * 1000);

        result = nn::svc::WaitProcessWideKeyAtomic(addr, WaitKey, newValue, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
}
