﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Dd.h>

namespace {
const nn::svc::MemoryPermission ValidPerm[] = {
    nn::svc::MemoryPermission_ReadWrite,
    nn::svc::MemoryPermission_Read,
    nn::svc::MemoryPermission_None,
};

} // namespace

TEST(UnmapTransferMemory, HandleTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        nn::svc::MemoryPermission perm = ValidPerm[i];
        TestHeap heap(HeapAlign);
        size_t size = heap.GetSize();
        uintptr_t heapAddr = heap.GetAddress();
        uintptr_t addr = g_FreeAreaBegin;

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::CreateTransferMemory(&handle, heapAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser(handle);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::MapTransferMemory(handle, addr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        // TEST 128-1
        // CreateTransferMemory で作成したハンドルを受け付ける
        // test_MapTransferMemory_Arguments.cpp にてテスト

        // TEST 128-2
        // INVALID_HANDLE_VALUE を受け付けない
        result = nn::svc::UnmapTransferMemory(nn::svc::INVALID_HANDLE_VALUE, addr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        GetMemoryInfo(&blockInfo, addr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        // TEST 128-3
        // スレッドの擬似ハンドルを受け付けない
        result = nn::svc::UnmapTransferMemory(
                nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, addr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        GetMemoryInfo(&blockInfo, addr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        // TEST 128-4
        // プロセスの擬似ハンドルを受け付けない
        result = nn::svc::UnmapTransferMemory(
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, addr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        GetMemoryInfo(&blockInfo, addr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        {
            nn::svc::Handle sharedHandle;
            nn::svc::MemoryPermission sharedPerm = nn::svc::MemoryPermission_ReadWrite;
            result = nn::svc::CreateSharedMemory(
                    &sharedHandle, size, sharedPerm, sharedPerm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(sharedHandle);

            TestSharedMemory sharedMem(
                    sharedHandle, addr + size, size, sharedPerm);
            GetMemoryInfo(&blockInfo, addr + size);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(blockInfo.permission == sharedPerm);
            ASSERT_TRUE(blockInfo.attribute == 0);

            // TEST 128-5
            // 共有メモリのハンドルは受け付けない
            result = nn::svc::UnmapTransferMemory(sharedHandle, addr + size, size);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

            GetMemoryInfo(&blockInfo, addr + size);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(blockInfo.permission == sharedPerm);
            ASSERT_TRUE(blockInfo.attribute == 0);

            // TEST 128-32
            // CreateSharedMemory で作った共有領域を解除できない
            result = nn::svc::UnmapTransferMemory(handle, addr + size, size);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

            GetMemoryInfo(&blockInfo, addr + size);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(blockInfo.permission == sharedPerm);
            ASSERT_TRUE(blockInfo.attribute == 0);

            // 一度クローズするためにマップを解除する
            result = nn::svc::UnmapTransferMemory(handle, addr, size);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&blockInfo, addr);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
            ASSERT_TRUE(blockInfo.attribute == 0);

            GetMemoryInfo(&blockInfo, addr + size);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(blockInfo.permission == sharedPerm);
            ASSERT_TRUE(blockInfo.attribute == 0);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 128-6
            // Close した CreateTransferMemory のハンドルを受け付けない
            result = nn::svc::UnmapTransferMemory(handle, addr + size, size);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        }

    }
} // NOLINT(readability/fn_size)

TEST(UnmapTransferMemory, AddrTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        nn::svc::MemoryPermission perm = ValidPerm[i];
        uintptr_t heapAddr;
        size_t heapSize = g_FreeAreaEnd - g_FreeAreaBegin;
        size_t size = 0x2000;
        TestHeap heap(heapSize);
        heapAddr = heap.GetAddress();
        ASSERT_TRUE(heapSize == heap.GetSize());
        ASSERT_TRUE((heapAddr & (0xfff)) == 0);

        result = nn::svc::CreateTransferMemory(&handle, heapAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser(handle);

        // TEST 128-6
        // 4KB にアライメントされている値を受け付ける
        // test_MapTransferMemory_Arguments.cpp にてテスト

        // TEST 128-7
        // 4KB にアライメントされていないと失敗する
        uintptr_t toAddr = g_FreeAreaBegin;
        result = nn::svc::MapTransferMemory(handle, toAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        for (uintptr_t i = 1; i < 0x1000; i++)
        {
            result = nn::svc::UnmapTransferMemory(handle, toAddr + i, size);
            ASSERT_TRUE(result <= nn::svc::ResultInvalidAddress() ||
                    result <= nn::svc::ResultInvalidRegion());
        }

        result =  nn::svc::UnmapTransferMemory(handle, toAddr, size);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(UnmapTransferMemory, SizeTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        nn::svc::MemoryPermission perm = ValidPerm[i];
        uintptr_t heapAddr;
        size_t heapSize = g_FreeAreaEnd - g_FreeAreaBegin;
        TestHeap heap(heapSize);
        heapAddr = heap.GetAddress();
        ASSERT_TRUE(heapSize == heap.GetSize());
        ASSERT_TRUE((heapAddr & (0xfff)) == 0);

        size_t baseSize = 0x2000;
        result = nn::svc::CreateTransferMemory(&handle, heapAddr, baseSize, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser(handle);

        // TEST 128-9
        // CreateTransferMemory と一致していると成功する
        // test_MapTransferMemory_Arguments.cpp にてテスト

        uintptr_t toAddr = g_FreeAreaBegin;
        result = nn::svc::MapTransferMemory(handle, toAddr, baseSize, perm);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 128-8
        // CreateTransferMemory と一致していないと失敗する
        for (size_t size = 1; size < baseSize; size++)
        {
            result = nn::svc::UnmapTransferMemory(handle, toAddr, size);
            ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
                    result <= nn::svc::ResultInvalidRegion());
        }

        // TEST 128-33
        // 0 を受け付けない
        result = nn::svc::UnmapTransferMemory(handle, toAddr, 0);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
                result <= nn::svc::ResultInvalidRegion());

        // TEST 128-34
        // MapTransferMemory でマップした領域の一部分だけを解除することはできない
        result =  nn::svc::UnmapTransferMemory(handle, toAddr, 0x1000);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
                result <= nn::svc::ResultInvalidRegion());

        result =  nn::svc::UnmapTransferMemory(handle, toAddr, baseSize);
        ASSERT_RESULT_SUCCESS(result);

    }
}

TEST(UnmapTransferMemory, MemoryAreaTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidPerm) / sizeof(nn::svc::MemoryPermission)); i++)
    {
        nn::svc::MemoryPermission perm = ValidPerm[i];
        uintptr_t addr = g_FreeAreaBegin;
        size_t size = 0x1000;

        uintptr_t heapAddr;
        TestHeap heap(HeapAlign);
        heapAddr = heap.GetAddress();
        ASSERT_TRUE((heapAddr & (0xfff)) == 0);

        result = nn::svc::CreateTransferMemory(&handle, heapAddr, size, perm);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose transferCloser(handle);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::MapTransferMemory(handle, addr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 128-27
        // オーバーフローする組み合わせを指定できない
        GetMemoryInfo(&blockInfo, addr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::UnmapTransferMemory(
                handle, addr, static_cast<size_t>(-1 * size));
        ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
                result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

        GetMemoryInfo(&blockInfo, addr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        GetMemoryInfo(&blockInfo, addr);
        if (perm == nn::svc::MemoryPermission_None)
        {
            // TEST 128-36
            // MemoryState_Transfered の領域を受け付ける
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);
        }
        else
        {
            // TEST 128-37
            // MemoryState_SharedTransfered の領域を受け付ける
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);
        }
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::UnmapTransferMemory(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

}

