﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <cstring>

extern "C" void nnMain();

TEST(UnmapSharedMemory, InvalidHandleTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size = 0x1000;

    // TEST 18-7
    result = nn::svc::UnmapSharedMemory(nn::svc::INVALID_HANDLE_VALUE, g_FreeAreaBegin, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 18-8
    result =  nn::svc::UnmapSharedMemory(
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, g_FreeAreaBegin, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 18-9
    result =  nn::svc::UnmapSharedMemory(
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, g_FreeAreaBegin, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(UnmapSharedMemory, Rw_InvalidAddressTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    // 前準備
    size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size, nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 18-10
    // アライメントされていないアドレスを指定すると失敗する
    for (int i = 1; i < 0x1000; i++)
    {
        result = nn::svc::UnmapSharedMemory(handle, i, size);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidAddress() ||
                result <= nn::svc::ResultInvalidRegion());
    }

    // 終了処理
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}


TEST(UnmapSharedMemory, Rw_InvalidMemoryStateTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    // 前準備
    size = 0x2000;
    result = nn::svc::CreateSharedMemory(
            &handle, size,
            nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // メモリ状態が一様でないと失敗する
    // 前方
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin - (size / 2), size);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // 後方
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin + (size / 2), size);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // TEST 18-24
    // addr, size がオーバーフローする組み合わせは受け付けない
    result = nn::svc::UnmapSharedMemory(
            handle, g_FreeAreaBegin + size, static_cast<uintptr_t>(-size));
    ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
            result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

    // 終了処理
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapSharedMemory, Ro_InvalidAddressTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size, nn::svc::MemoryPermission_Read, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 18-17
    // アライメントされていないアドレスを指定すると失敗する
    for (int i = 1; i < 0x1000; i++)
    {
        result = nn::svc::UnmapSharedMemory(handle, i, size);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidAddress() ||
                result <= nn::svc::ResultInvalidRegion());
    }

    // 終了処理
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}


TEST(UnmapSharedMemory, Ro_InvalidMemoryStateTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    // 前準備
    size = 0x2000;
    result = nn::svc::CreateSharedMemory(
            &handle, size, nn::svc::MemoryPermission_Read, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    // メモリ状態が一様でないと失敗する
    // 前方
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin - (size / 2), size);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // 後方
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin + (size / 2), size);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // TEST 18-24
    // addr, size がオーバーフローする組み合わせは受け付けない
    result = nn::svc::UnmapSharedMemory(
            handle, g_FreeAreaBegin + size, static_cast<uintptr_t>(-size));
    ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
            result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

    // 終了処理
    result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

