﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <cstring>

extern "C" void nnMain();

namespace {
} // namespace

TEST(UnmapProcessCodeMemory, HandleTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-2
    // 実行中のプロセスのハンドルを受け付ける
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    WaitProcess(processHandle);

    stateProcess.SetDead();

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-3
    // 実行後のプロセスのハンドルを受け付ける
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-4
    // プロセスの擬似ハンドルを受け付けない
    result = nn::svc::UnmapProcessCodeMemory(
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 120-5
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::UnmapProcessCodeMemory(
            nn::svc::INVALID_HANDLE_VALUE, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 120-6
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::UnmapProcessCodeMemory(
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-7
    // Close したプロセスのハンドルを受け付けない
    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(UnmapProcessCodeMemory, AddrTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-8
    // toAddr のアラインメントが4KBに揃っていると成功する
    // TEST 120-10
    // fromAddr のアラインメントが4KBに揃っていると成功する
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-9
    // toAddr のアラインメントが4KBに揃っていないと失敗する
    for (uint64_t i = toAddr + 1; i < toAddr + 0x1000; i++)
    {
        result = nn::svc::UnmapProcessCodeMemory(processHandle, i, fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    // TEST 120-11
    // fromAddr のアラインメントが4KBに揃っていないと失敗する
    for (uint64_t i = fromAddr + 1; i < fromAddr + 0x1000; i++)
    {
        result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, i, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapProcessCodeMemory, SizeTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-12
    // size のアラインメントが4KBに揃っていると成功する
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 120-13
    // size のアラインメントが4KBに揃っていないと失敗する
    for (uint64_t i = 1; i < 0x1000; i++)
    {
        result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, i);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    // TEST 120-14
    // size は 0 を受け付けない
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapProcessCodeMemory, ToAddrAreaTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();
    nn::svc::MemoryState state;
    uint64_t addr;
    uint64_t size;
    uint64_t memSize = 0x2000;

    // 許可されていないメモリ状態
    uint64_t toAddr = g_FreeAreaEnd - memSize;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, memSize);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, memSize);
    ASSERT_RESULT_SUCCESS(result);

    for (int i = 0; i < NumTestMemoryState; i++)
    {
        state = AllMemoryState[i];
        stateProcess.GetMemoryArea(&addr, &size, state);
        if (size < memSize)
        {
            stateProcess.FreeMemoryArea();
            continue;
        }

        result = nn::svc::UnmapProcessCodeMemory(processHandle, addr, fromAddr, memSize);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
                result <= nn::svc::ResultInvalidRegion());

        stateProcess.FreeMemoryArea();
    }

    // TEST 120-30
    // toAddr/size で示す領域 のメモリ状態が一様でないと失敗する (前方)
    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr - (memSize / 2), fromAddr, memSize);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // TEST 120-31
    // toAddr/size で示す領域 のメモリ状態が一様でないと失敗する (後方)
    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr + (memSize / 2), fromAddr, memSize);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // TEST 120-32
    // オーバーフローする組み合わせを指定できない
    size = static_cast<uint64_t>(-0x1000);
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // 終了処理
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, memSize);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapProcessCodeMemory, FromAddrAreaTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();
    nn::svc::MemoryState state;
    uint64_t addr;
    uint64_t size;
    uint64_t memSize = 0x2000;

    // 許可されていないメモリ状態
    uint64_t toAddr = g_FreeAreaEnd - memSize;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, memSize);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, memSize);
    ASSERT_RESULT_SUCCESS(result);

    for (int i = 0; i < NumTestMemoryState; i++)
    {
        state = AllMemoryState[i];
        stateProcess.GetMemoryArea(&addr, &size, state);
        if (size < memSize)
        {
            stateProcess.FreeMemoryArea();
            continue;
        }

        result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, addr, memSize);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
                result <= nn::svc::ResultInvalidRegion());

        stateProcess.FreeMemoryArea();
    }

    // TEST 120-48
    // fromAddr/size で示す領域 のメモリ状態が一様でないと失敗する (前方)
    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr, fromAddr - (memSize / 2), memSize);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // TEST 120-49
    // fromAddr/size で示す領域 のメモリ状態が一様でないと失敗する (後方)
    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr, fromAddr + (memSize / 2), memSize);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // TEST 120-50
    // オーバーフローする組み合わせを指定できない
    size = static_cast<uint64_t>(-0x1000);
    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // 終了処理
    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, memSize);
    ASSERT_RESULT_SUCCESS(result);
}

