﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Dd.h>

namespace {
char g_Buffer[0x2000] __attribute__((aligned(0x1000)));
const uint64_t SpaceAddr = 0;
} // namespace

TEST(UnmapDeviceAddressSpace, CombinationTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uint64_t align = 0x400000;
    uint64_t addr = reinterpret_cast<uint64_t>(g_Buffer);
    uint64_t mapSize = 0x1000;
    uint64_t deviceAddress = addr & (align - 1);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + mapSize, mapSize, deviceAddress + mapSize,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // 133-41
    // processAddress/size と deviceAddress/size の組み合わせがマップ時と一致していないと失敗する
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress + mapSize);
    ASSERT_RESULT_FAILURE(result);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + mapSize, mapSize, deviceAddress);
    ASSERT_RESULT_FAILURE(result);

    // 終了処理
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + mapSize, mapSize, deviceAddress + mapSize);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapDeviceAddressSpace, ReleaseMergedBlockTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uint64_t align = 0x400000;
    uint64_t addr = reinterpret_cast<uint64_t>(g_Buffer);
    uint64_t mapSize = 0x1000;
    uint64_t deviceAddress = addr & (align - 1);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + mapSize, mapSize, deviceAddress + mapSize,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.size == mapSize * 2);

    // 133-43
    // MapDeviceAddressSpaceAligned を複数回呼ぶことでマージされたブロックを一括して解放することが出来る
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize * 2, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);
    ASSERT_TRUE(blockInfo.baseAddress <= addr);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= addr + mapSize * 2);
}

TEST(UnmapDeviceAddressSpace, PartialReleaseTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uint64_t align = 0x400000;
    uint64_t addr = reinterpret_cast<uint64_t>(g_Buffer);
    uint64_t mapSize = 0x1000;
    uint64_t deviceAddress = addr & (align - 1);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize * 2, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.size == mapSize * 2);

    // 133-44
    // 領域の一部を解放することが出来る
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + mapSize, mapSize, deviceAddress + mapSize);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.size == mapSize);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);
}
