﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>

namespace {

char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));
int32_t g_SavedPriority;

struct TestData
{
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
};

void TestPriority(TestData* pData)
{
    AutoThreadExit autoExit;

    for (int i = TestHighestThreadPriority; i <= TestLowestThreadPriority; i++)
    {
        nn::Result result;
        int32_t index;

        result = nn::svc::WaitSynchronization(&index, &pData->readableEvent, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);
        result = nn::svc::ResetSignal(pData->readableEvent);
        ASSERT_RESULT_SUCCESS(result);

        nnHandle handle;
        handle.value = 0xFFFF8000;
        // TEST 11-8 (同じコア) or TEST 11-11 (違うコア)
        // 作成元のスレッドで SetThreadPriority を呼び、作成されたスレッドで設定の確認をする
        {
            int32_t tmp;
            result = nn::svc::GetThreadPriority(&tmp, static_cast<nn::svc::Handle>(handle));
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(tmp == g_SavedPriority);
        }

        result = nn::svc::SignalEvent(pData->writableEvent);
        ASSERT_RESULT_SUCCESS(result);
    }
}
}

/* TEST 11-3 */
/* 自スレッドの優先度を変更することが出来ることを確認する */
TEST(SetThreadPriority, Test0)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nnHandle handle;
    handle.value = 0xFFFF8000;
    int32_t curPriority;

    result = nn::svc::GetThreadPriority(&curPriority, static_cast<nn::svc::Handle>(handle));
    ASSERT_RESULT_SUCCESS(result);

    for (int32_t i = TestHighestThreadPriority; i <= TestLowestThreadPriority; i++)
    {
        result = nn::svc::SetThreadPriority(static_cast<nn::svc::Handle>(handle), i);
        ASSERT_RESULT_SUCCESS(result);
        int32_t tmp;
        result = nn::svc::GetThreadPriority(&tmp, static_cast<nn::svc::Handle>(handle));
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(tmp == i);
    }
    result = nn::svc::SetThreadPriority(static_cast<nn::svc::Handle>(handle), curPriority);
    ASSERT_RESULT_SUCCESS(result);
}


/* TEST 11-1, 11-8, 11-10, 11-11 */
/* 自スレッド以外のスレッドに対して、SetThreadPriority の設定が反映されていることを確認する */
TEST(SetThreadPriority, Test1)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle writableEvent0;
    nn::svc::Handle readableEvent0;
    result = nn::svc::CreateEvent(&writableEvent0, &readableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    nn::svc::Handle writableEvent1;
    nn::svc::Handle readableEvent1;
    result = nn::svc::CreateEvent(&writableEvent1, &readableEvent1);
    ASSERT_RESULT_SUCCESS(result);
    TestData data;
    data.readableEvent = readableEvent0;
    data.writableEvent = writableEvent1;

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(TestPriority);
        nn::svc::Handle handle;
        int32_t index;

        result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&data), sp, 32, idealCore);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(handle);
        ASSERT_RESULT_SUCCESS(result);

        for (int32_t i = TestHighestThreadPriority; i <= TestLowestThreadPriority; i++)
        {
            // TEST 11-1 (同じコア) or TEST 11-10 (違うコア)
            // 作成もとで、SetThreadPriority を呼び、その結果を確認する
            g_SavedPriority = i;
            nn::Result result = nn::svc::SetThreadPriority(handle, i);
            ASSERT_RESULT_SUCCESS(result);

            int32_t tmp;
            result = nn::svc::GetThreadPriority(&tmp, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(tmp == i);

            result = nn::svc::SignalEvent(writableEvent0);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::WaitSynchronization(&index, &readableEvent1, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);
            result = nn::svc::ResetSignal(readableEvent1);
            ASSERT_RESULT_SUCCESS(result);
        }

        result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
    result = nn::svc::CloseHandle(writableEvent0);
    result = nn::svc::CloseHandle(readableEvent0);
    result = nn::svc::CloseHandle(writableEvent1);
    result = nn::svc::CloseHandle(readableEvent1);
}

