﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <cstring>

namespace {

} // namespace

TEST(StartProcess, HandleTest)
{
    TestProcessLeak leakTest;
    nn::Result result;

    // TEST 122-2
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::StartProcess(
            nn::svc::INVALID_HANDLE_VALUE, TestLowestThreadPriority, 0,
            DefaultStackSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 122-3
    // プロセスの擬似ハンドルを受け付けない
    result = nn::svc::StartProcess(
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, TestLowestThreadPriority,
            g_ProcessIdealCore, DefaultStackSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

    // TEST 122-4
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::StartProcess(
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, TestLowestThreadPriority, 0,
            DefaultStackSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 122-5
    // Close したプロセスを受け付けない
    {
        TestProcess process(1);
        nn::svc::Handle handle = process.GetHandle();

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartProcess(
                handle, TestLowestThreadPriority, 0, DefaultStackSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    {
        nn::Bit32 flag[DefaultCapabilityFlagNum];
        SetDefaultCapability(flag, DefaultCapabilityFlagNum);

        TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);
        nn::svc::Handle handle = process.GetHandle();
        AssignExitCode(handle, process.GetCodeAddress(), process.GetCodeAreaSize());

        // 同じコアで動作させることで、スタートさせたプロセスに遷移させない
        result = nn::svc::StartProcess(
                handle, TestLowestThreadPriority, g_ProcessIdealCore,
                DefaultStackSize);
        ASSERT_RESULT_SUCCESS(result);
        nn::svc::ResetSignal(handle);

        // TEST 122-6
        // 実行中のプロセスを受け付けない
        result = nn::svc::StartProcess(
                handle, TestLowestThreadPriority, g_ProcessIdealCore,
                DefaultStackSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        WaitProcess(handle);

        // TEST 122-23
        // 終了したプロセスを受け付けない
        result = nn::svc::StartProcess(
                handle, TestLowestThreadPriority, g_ProcessIdealCore,
                DefaultStackSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());
    }
}

TEST(StartProcess, PriorityTest)
{
    TestProcessLeak leakTest;
    nn::Result result;

    nn::Bit32 flag[DefaultCapabilityFlagNum];
    SetDefaultCapability(flag, DefaultCapabilityFlagNum);

    TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);
    nn::svc::Handle handle = process.GetHandle();
    AssignExitCode(handle, process.GetCodeAddress(), process.GetCodeAreaSize());

    // TEST 122-8
    // 優先度の最高優先度よりも小さい値を受け付けない
    result = nn::svc::StartProcess(
            handle, nn::svc::HighestThreadPriority - 1, 0, DefaultStackSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPriority());

    // TEST 122-9
    // 優先度の最低優先度よりも大きい値を受け付けない
    result = nn::svc::StartProcess(
            handle, nn::svc::LowestThreadPriority + 1, 0, DefaultStackSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPriority());
}

TEST(StartProcess, CoreNoTest)
{
    TestProcessLeak leakTest;
    nn::Result result;

    nn::Bit32 flag[DefaultCapabilityFlagNum];
    SetDefaultCapability(flag, DefaultCapabilityFlagNum);

    TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);
    nn::svc::Handle handle = process.GetHandle();
    AssignExitCode(handle, process.GetCodeAddress(), process.GetCodeAreaSize());

    int32_t priority = TestLowestThreadPriority;

    // TEST 122-11
    // 存在しないコアは指定できない
    result = nn::svc::StartProcess(handle, priority, NumCore, DefaultStackSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCoreNumber());

    // TEST 122-19
    // IdealCoreUseProcessValue を指定することが出来ない
    result = nn::svc::StartProcess(
            handle, priority, nn::svc::IdealCoreUseProcessValue, DefaultStackSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCoreNumber());

    // TEST 122-20
    // IdealCoreDontCare を指定することが出来ない
    result = nn::svc::StartProcess(
            handle, priority, nn::svc::IdealCoreDontCare, DefaultStackSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCoreNumber());

    // TEST 122-21
    // IdealCoreNoUpdate を指定することが出来ない
    result = nn::svc::StartProcess(
            handle, priority, nn::svc::IdealCoreNoUpdate, DefaultStackSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCoreNumber());
}

TEST(StartProcess, StackSizeTest)
{
    TestProcessLeak leakTest;
    nn::Result result;

    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    {
        nn::Bit32 flag[DefaultCapabilityFlagNum];
        SetDefaultCapability(flag, DefaultCapabilityFlagNum);

        TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);
        nn::svc::Handle handle = process.GetHandle();
        AssignExitCode(handle, process.GetCodeAddress(), process.GetCodeAreaSize());

        // TEST 122-13
        // 0 を指定することが出来る
        result = nn::svc::StartProcess(handle, priority, idealCore, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    {
        nn::Bit32 flag[DefaultCapabilityFlagNum];
        SetDefaultCapability(flag, DefaultCapabilityFlagNum);

        TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);
        nn::svc::Handle handle = process.GetHandle();
        AssignExitCode(handle, process.GetCodeAddress(), process.GetCodeAreaSize());

        // TEST 122-14
        // 大きすぎる値を渡すと失敗する
        result = nn::svc::StartProcess(handle, priority, idealCore, SIZE_MAX);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfMemory());
    }
}

