﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Result.h>

namespace {
char g_Buffer[2][0x2000] __attribute__((aligned(0x1000)));
const bool HighSender = true;
const bool LowSender = false;

struct TestData
{
    bool sender;
    nn::svc::Handle eventHandle;
};

void SignalTest(TestData *data)
{
    AutoThreadExit autoExit;
    nn::Result result;
    if (data->sender)
    {
        result = nn::svc::SignalEvent(data->eventHandle);
        ASSERT_RESULT_SUCCESS(result);
    }
    else
    {
        int index;
        result = nn::svc::WaitSynchronization(&index, &data->eventHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
    }
}

void CreateSignalTestThread(bool sender, int32_t diffPriority)
{
    nn::Result result;
    nn::svc::Handle handles[2];
    nn::svc::Handle readableEvent;
    nn::svc::Handle writableEvent;
    TestData data[2];
    uintptr_t sp[2];
    uintptr_t pc;
    uintptr_t param[2];
    int32_t index;

    ASSERT_TRUE(diffPriority >= 0);

    sp[0] = reinterpret_cast<uintptr_t>(g_Buffer[0] + sizeof(g_Buffer[0]));
    sp[1] = reinterpret_cast<uintptr_t>(g_Buffer[1] + sizeof(g_Buffer[1]));
    pc = reinterpret_cast<uintptr_t>(SignalTest);

    param[0] = reinterpret_cast<uintptr_t>(&data[0]);
    param[1] = reinterpret_cast<uintptr_t>(&data[1]);

    if (sender == HighSender)
    {
        data[0].sender = true;
        data[1].sender = false;
    }
    else
    {
        data[0].sender = false;
        data[1].sender = true;
    }

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t otherCore = 0; otherCore < NumCore; otherCore++)
        {
            for (int32_t priority = TestHighestThreadPriority;
                priority + diffPriority <= TestLowestThreadPriority; priority++)
            {
                result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
                ASSERT_RESULT_SUCCESS(result);
                if (sender == HighSender)
                {
                    data[0].eventHandle = writableEvent;
                    data[1].eventHandle = readableEvent;
                }
                else
                {
                    data[0].eventHandle = readableEvent;
                    data[1].eventHandle = writableEvent;
                }

                result = nn::svc::CreateThread(&handles[0], pc, param[0], sp[0],
                    priority, idealCore);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CreateThread(&handles[1], pc, param[1], sp[1],
                    priority + diffPriority, otherCore);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::StartThread(handles[0]);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::StartThread(handles[1]);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::WaitSynchronization(&index, &handles[0], 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &handles[1], 1, -1);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::CloseHandle(handles[0]);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(handles[1]);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::CloseHandle(writableEvent);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(readableEvent);
                ASSERT_RESULT_SUCCESS(result);
            }
        }
    }

}

}

TEST(SignalEventTest, HighPrioritySendsSignalTest)
{
    TestEventLeak leakTest;
    /* TEST 15-20 (違うコア), 15-22 (同じコア) */
    /* 高い優先度のスレッドから低い優先度のスレッドへシグナルを送信できる */
    CreateSignalTestThread(HighSender, 1);
}

TEST(SignalEventTest, LowPrioritySendsSignalTest)
{
    TestEventLeak leakTest;
    /* TEST 15-21 (違うコア), 15-23 (同じコア) */
    /* 低い優先度のスレッドから高い優先度のスレッドへシグナルを送信できる */
    CreateSignalTestThread(LowSender, 1);
}

TEST(SignalEventTest, SamePrioritySendsSignalTest)
{
    TestEventLeak leakTest;
    /* TEST 15-24(違うコア), 15-25(同じコア) */
    /* 同じ優先度のスレッドへシグナルを送信できる */
    CreateSignalTestThread(HighSender, 0);
}

TEST(SignalEventTest, MultipleSendSignal)
{
    TestEventLeak leakTest;
    nn::Result result;

    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableEvent);
    AutoHandleClose rEventCloser(readableEvent);

    // TEST 15-26
    // シグナルを受信する前に再度シグナルを送信することが出来る
    result = nn::svc::SignalEvent(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SignalEvent(writableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 15-27
    // 複数回シグナルされても、一度リセットされると、すべてクリアされる
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ClearEvent(readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}
