﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_HardwareParamsSelect.h>
#include <nn/svc/svc_Result.h>

namespace {

const int32_t NumPriority = TestLowestThreadPriority - TestHighestThreadPriority + 1;

char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));
int32_t g_SavedPriority;

struct TestData
{
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
};

void TestPriorityFunction0(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
}

/*
   自スレッドの優先度を変更する
 */
void TestPriorityFunction1(TestData* pData)
{
    AutoThreadExit autoExit;

    for (int32_t priority = TestHighestThreadPriority;
        priority <= TestLowestThreadPriority; priority++)
    {
        nn::Result result;

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &pData->readableEvent, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::ClearEvent(pData->readableEvent);
        ASSERT_RESULT_SUCCESS(result);

        {
            result = nn::svc::SetThreadPriority(
                        nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, g_SavedPriority);
            ASSERT_RESULT_SUCCESS(result);
            int32_t tmp;
            result = nn::svc::GetThreadPriority(&tmp, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(tmp == g_SavedPriority);
        }

        result = nn::svc::SignalEvent(pData->writableEvent);
        ASSERT_RESULT_SUCCESS(result);
    }
}

/* TEST 11-15, 11-16 */
/*
   他スレッドがスレッドの実行前に優先度を変更し、起動したスレッドで優先度が変更されたかを確認する
 */
void TestPriorityFunction2(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t tmp;

    result = nn::svc::GetThreadPriority(&tmp, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(tmp == g_SavedPriority);
}

} //namespace {

/* TEST 11-2 */
/* 不正な優先度が渡されたときにエラーを返すことを確認する */
TEST(SetThreadPriority, InvalidPriorityTest)
{
    TestThreadLeak leakTest;
    nn::Result result;

    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
            nn::svc::HighestThreadPriority - 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPriority());
    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
            nn::svc::LowestThreadPriority + 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPriority());
}

/* TEST 11-4, 11-5, 11-6 */
/* 不正なハンドルが渡されたときにエラーを返すことを確認する */
TEST(SetThreadPriority, InvalidHandleTest)
{
    TestThreadLeak leakTest;
    nn::svc::Handle closedHandle;
    nn::Result result;
    int32_t priority = TestLowestThreadPriority;
    uintptr_t sp;
    uintptr_t pc;
    int32_t curPriority;

    result = nn::svc::GetThreadPriority(&curPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);

    sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    pc = reinterpret_cast<uintptr_t>(TestPriorityFunction0);

    // TEST 11-4
    result = nn::svc::SetThreadPriority(
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, priority);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 11-5
    result = nn::svc::SetThreadPriority(nn::svc::INVALID_HANDLE_VALUE, priority);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    result = nn::svc::CreateThread(&closedHandle, pc, 0, sp, 32, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(closedHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 11-6
    result = nn::svc::SetThreadPriority(closedHandle, priority);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, curPriority);
    ASSERT_RESULT_SUCCESS(result);
}

/* TEST 11-9, 11-12 */
/* 自スレッド以外の実行中スレッドに対して、SetThreadPriority の設定が反映されていることを確認する */
TEST(SetThreadPriority, SetPriorityTest0)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle writableEvent0;
    nn::svc::Handle readableEvent0;
    nn::svc::Handle writableEvent1;
    nn::svc::Handle readableEvent1;
    TestData data;
    int32_t index;

    result = nn::svc::CreateEvent(&writableEvent0, &readableEvent0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateEvent(&writableEvent1, &readableEvent1);
    ASSERT_RESULT_SUCCESS(result);

    data.readableEvent = readableEvent0;
    data.writableEvent = writableEvent1;


    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(TestPriorityFunction1);
        nn::svc::Handle handle;

        result = nn::svc::CreateThread(
                &handle, pc, reinterpret_cast<uintptr_t>(&data), sp, 32, idealCore);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(handle);
        ASSERT_RESULT_SUCCESS(result);

        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority; priority++)
        {
            g_SavedPriority = priority;
            result = nn::svc::SignalEvent(writableEvent0);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::WaitSynchronization(&index, &readableEvent1, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::ClearEvent(readableEvent1);
            ASSERT_RESULT_SUCCESS(result);

            // 11-9 (同じコア) or 11-12 (違うコア)
            // 作成したスレッドの中で優先度を変更し、それが他のスレッドにも伝わることを確認
            int32_t tmp;
            result = nn::svc::GetThreadPriority(&tmp, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(tmp == g_SavedPriority);
        }

        result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
    result = nn::svc::CloseHandle(writableEvent0);
    result = nn::svc::CloseHandle(readableEvent0);
    result = nn::svc::CloseHandle(writableEvent1);
    result = nn::svc::CloseHandle(readableEvent1);
}

/*
   自スレッド以外の実行前のスレッドに対して、
   SetThreadPriority の設定が反映されていることを確認する
 */
TEST(SetThreadPriority, SetPriorityTest1)
{
    TestThreadLeak leakTest;
    nn::Result result;
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    uintptr_t pc = reinterpret_cast<uintptr_t>(TestPriorityFunction2);
    int32_t tmp;
    int32_t index;

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority; priority++)
        {
            nn::svc::Handle handle;

            result = nn::svc::CreateThread(&handle, pc, 0, sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            // 11-13 (同じコア) or 11-14 (違うコア)
            // スレッドの実行前に優先度を変更し、作成したスレッドで変更を確認する
            g_SavedPriority = (priority + 1) % NumPriority + TestHighestThreadPriority;
            result = nn::svc::SetThreadPriority(handle, g_SavedPriority);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::GetThreadPriority(&tmp, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(tmp == g_SavedPriority);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            // 11-17 (同じコア) or 11-18 (違うコア)
            // スレッドの実行後に優先度を変更し、作成したスレッドで変更を確認する
            result = nn::svc::SetThreadPriority(handle, priority);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::GetThreadPriority(&tmp, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(tmp == priority);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}

