﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Dd.h>

namespace {
char g_Buffer[0x2000] __attribute__((aligned(0x1000)));
}
extern "C" void nnMain();

TEST(SetMemoryPermissionTest, HeapTest)
{
    uintptr_t addr;
    size_t size;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.size == size);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

    // TEST 2-36
    // RW -> RW
    for (size_t i = 0; i < size; i += 0x1000)
    {
        result = nn::svc::SetMemoryPermission(addr + i, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    }

    // TEST 2-37
    // RW -> R
    for (size_t i = 0; i < size; i += 0x1000)
    {
        result = nn::svc::SetMemoryPermission(addr + i, 0x1000, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == (i + 0x1000));
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        if (size != (i + 0x1000))
        {
            result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + i + 0x1000);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(blockInfo.baseAddress == addr + i + 0x1000);
            ASSERT_TRUE(blockInfo.size == size - (i + 0x1000));
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        }
    }

    // TEST 2-38
    // R -> N
    for (size_t i = 0; i < size; i += 0x1000)
    {
        result = nn::svc::SetMemoryPermission(addr + i, 0x1000, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == (i + 0x1000));
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        if (size != (i + 0x1000))
        {
            result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + i + 0x1000);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(blockInfo.baseAddress == addr + i + 0x1000);
            ASSERT_TRUE(blockInfo.size == size - (i + 0x1000));
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        }
    }

    // TEST 2-39
    // N -> RW && size > 0x1000
    {
        result = nn::svc::SetMemoryPermission(addr, size / 2, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + size / 2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr + size / 2);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::SetMemoryPermission(addr + size / 2, size / 2, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    }

    // TEST 2-40
    // RW -> N && size > 0x1000
    {
        result = nn::svc::SetMemoryPermission(addr, size / 2, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + size / 2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr + size / 2);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::SetMemoryPermission(addr + size / 2, size / 2, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    }

    // TEST 2-41
    // N -> R && size > 0x1000
    {
        result = nn::svc::SetMemoryPermission(addr, size / 2, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + size / 2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr + size / 2);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::SetMemoryPermission(addr + size / 2, size / 2, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    }

    // TEST 2-4
    // ヒープ領域外のアドレスを指定する
    result = nn::svc::SetMemoryPermission(addr - 0x1000, 0x2000, nn::svc::MemoryPermission_Read);
    EXPECT_TRUE(result.IsFailure());
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-5
    // 確保しているヒープサイズ以上の大きさを指定する
    result = nn::svc::SetMemoryPermission(addr, size + 0x1000, nn::svc::MemoryPermission_Read);
    EXPECT_TRUE(result.IsFailure());
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-6
    // 途中から確保しているヒープ領域を抜け出てしまうメモリ領域を指定する
    result = nn::svc::SetMemoryPermission(addr + size - 0x1000, 0x2000, nn::svc::MemoryPermission_Read);
    EXPECT_TRUE(result.IsFailure());
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-7
    // ヒープ領域の終端をアドレスに指定する
    result = nn::svc::SetMemoryPermission(addr + size, 0x1000, nn::svc::MemoryPermission_Read);
    EXPECT_TRUE(result.IsFailure());
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-4
    // ヒープ領域外のアドレスを指定する
    result = nn::svc::SetMemoryPermission(addr - 0x1000, 0x2000, nn::svc::MemoryPermission_ReadWrite);
    EXPECT_TRUE(result.IsFailure());
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-5
    // 確保しているヒープサイズ以上の大きさを指定する
    result = nn::svc::SetMemoryPermission(addr, size + 0x1000, nn::svc::MemoryPermission_ReadWrite);
    EXPECT_TRUE(result.IsFailure());
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-6
    // 途中から確保しているヒープ領域を抜け出てしまうメモリ領域を指定する
    result = nn::svc::SetMemoryPermission(addr + size - 0x1000, 0x2000, nn::svc::MemoryPermission_ReadWrite);
    EXPECT_TRUE(result.IsFailure());
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-7
    // ヒープ領域の終端をアドレスに指定する
    result = nn::svc::SetMemoryPermission(addr + size, 0x1000, nn::svc::MemoryPermission_ReadWrite);
    EXPECT_TRUE(result.IsFailure());
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-42
    // 0 をサイズに指定できない
    result = nn::svc::SetMemoryPermission(addr, 0, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 2-9
    // ReadExecute を指定できない
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
} // NOLINT (readability/fn_size)

TEST(SetHeapSizeTest, BssTest)
{
    uintptr_t addr;
    size_t size;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t rwTop;
    size_t rwSize;

    size = sizeof(g_Buffer);
    addr = reinterpret_cast<uintptr_t>(g_Buffer);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    rwTop = blockInfo.baseAddress;
    rwSize = blockInfo.size;
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

    // TEST 2-43
    // RW -> RW
    for (size_t i = 0; i < size; i += 0x1000)
    {
        result = nn::svc::SetMemoryPermission(addr + i, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
        TEST_RW(addr + i);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == rwTop);
        ASSERT_TRUE(blockInfo.size == rwSize);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    }

    // TEST 2-44
    // RW -> R
    for (size_t i = 0; i < size; i += 0x1000)
    {
        result = nn::svc::SetMemoryPermission(addr + i, 0x1000, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);
        TEST_RO(addr + i);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == (i + 0x1000));
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

        if (size != (i + 0x1000))
        {
            result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + i + 0x1000);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(blockInfo.baseAddress == addr + i + 0x1000);
            ASSERT_TRUE(blockInfo.size == rwTop + rwSize - (addr + i + 0x1000));
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        }
    }

    // TEST 2-45
    // R -> N
    for (size_t i = 0; i < size; i += 0x1000)
    {
        result = nn::svc::SetMemoryPermission(addr + i, 0x1000, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);
        TEST_NA(addr + i);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == (i + 0x1000));
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

        if (size != (i + 0x1000))
        {
            result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + i + 0x1000);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(blockInfo.baseAddress == addr + i + 0x1000);
            ASSERT_TRUE(blockInfo.size == size - (i + 0x1000));
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        }
    }

    // TEST 2-46
    // N -> RW && size > 0x1000
    {
        result = nn::svc::SetMemoryPermission(addr, size / 2, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == rwTop);
        ASSERT_TRUE(blockInfo.size == addr + (size / 2) - rwTop);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + size / 2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr + size / 2);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

        result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::SetMemoryPermission(addr + size / 2, size / 2, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == rwTop);
        ASSERT_TRUE(blockInfo.size == rwSize);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    }

    // TEST 2-47
    // RW -> N && size > 0x1000
    {
        result = nn::svc::SetMemoryPermission(addr, size / 2, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + size / 2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr + size / 2);
        ASSERT_TRUE(blockInfo.size == rwTop + rwSize - (addr + size / 2));
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

        result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::SetMemoryPermission(addr + size / 2, size / 2, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    }

    // TEST 2-48
    // N -> R && size > 0x1000
    {
        result = nn::svc::SetMemoryPermission(addr, size / 2, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr + size / 2);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr + size / 2);
        ASSERT_TRUE(blockInfo.size == size / 2);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

        result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::SetMemoryPermission(addr + size / 2, size / 2, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.baseAddress == addr);
        ASSERT_TRUE(blockInfo.size == size);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    }

    // TEST 2-13
    // addr/size で指定される領域に確保した領域の前方の使われていない領域が含まれていると失敗する
    result = nn::svc::SetMemoryPermission(addr - 0x1000, 0x2000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-14
    // 確保された領域の大きさより大きなサイズを指定すると失敗する
    result = nn::svc::SetMemoryPermission(addr, size + 0x1000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-15
    // addr/size で指定される領域に確保した領域の後方の使われていない領域が含まれていると失敗する
    result = nn::svc::SetMemoryPermission(addr + size - 0x1000, 0x2000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-49
    // addr/size で指定される領域に確保した領域の前方の使われていない領域が含まれていると失敗する
    result = nn::svc::SetMemoryPermission(addr - 0x1000, 0x2000, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-50
    // 確保された領域の大きさより大きなサイズを指定すると失敗する
    result = nn::svc::SetMemoryPermission(addr + size - 0x1000, 0x2000, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-51
    // addr/size で指定される領域に確保した領域の後方の使われていない領域が含まれていると失敗する
    result = nn::svc::SetMemoryPermission(addr, 0, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    // TEST 2-52
    // 0 をサイズに指定できない
    result = nn::svc::SetMemoryPermission(addr, 0, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    // TEST 2-53
    // 0 をサイズに指定できない
    result = nn::svc::SetMemoryPermission(addr, 0, nn::svc::MemoryPermission_None);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 2-18
    // MemoryPermission_ReadExecute を指定できない
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());
} // NOLINT (readability/fn_size)

TEST(SetMemoryPermissionTest, FailureTest)
{
    nn::Result result;

    // TEST 2-19
    result = nn::svc::SetMemoryPermission(g_FreeAreaBegin, 0x1000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-20
    result = nn::svc::SetMemoryPermission(reinterpret_cast<uintptr_t>(&nnMain) & ~(0x1000 - 1), 0x1000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-21
    // ReadWrite に指定された共有メモリに対してRead を指定する
    {
        nn::svc::Handle handle;
        size_t size = 1024 * 1024;
        result = nn::svc::CreateSharedMemory(&handle, size, nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SetMemoryPermission(g_FreeAreaBegin, 0x1000, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}
