﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>

TEST(SendSyncRequest, InvalidHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;

    // TEST 28-7
    // INVALID_HANDLE_VALUE を渡すと失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 28-8
    // PSEUDO_HANDLE_CURRENT_THREAD を渡すと失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 28-9
    // PSEUDO_HANDLE_CURRENT_PROCESS を渡すと失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(SendSyncRequest, InvalidLightIPCHandleTest)
{
    TestLightSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    // TEST 28-10
    // Light系のセッションを渡すと失敗する
    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, true, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequest, InvalidNamedPortHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    const char* PortName = "Test2811";

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 2);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-11
    // NamedPort のサーバーポートを渡すと失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-12
    // NamedPort のサーバーセッショを渡すと失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 28-13
    // 閉じられたNamedPort のクライアントセッションを渡すと失敗する
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequest, InvalidPortHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-14
    // Port のサーバーポートを渡すと失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-15
    // Port のサーバーセッションを渡すと失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 28-16
    // Port のクライアントポートを渡すと失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 28-17
    // 閉じられた Port のクライアントセッションを渡すと失敗する
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequest, InvalidSessionHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-18
    // サーバーセッションが渡されると失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-19
    // 閉じられた クライアントセッションを渡されると失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
}

