﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestIpc.h"
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <cstring>

namespace {

const int64_t SleepTime = 100 * 1000 * 1000; // 100 msec
char g_IpcServerBuffer[0x1000] __attribute__((aligned(0x1000)));
char g_IpcClientBuffer[0x1000] __attribute__((aligned(0x1000)));
char g_IpcMapBuffer[0x3000] __attribute__((aligned(0x1000)));
char g_IpcRecvBuffer[2][0x1000] __attribute__((aligned(0x1000)));
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));
int g_Sequence;

const char* PointData1 = "PointData1";
const char* PointData2 = "PointData2";
const char* RawData1 = "12345";
const char* RawData2 = "54321";

const int32_t SendPointDataNum = 2;
const char* SendPointData[2] = { PointData1, PointData2};
const size_t SendPointDataSize[2] = { sizeof(PointData1), sizeof(PointData2)};
const int32_t ReplyPointDataNum = 2;
const char* ReplyPointData[2] = { PointData2, PointData1};
const size_t ReplyPointDataSize[2] = { sizeof(PointData2), sizeof(PointData1)};
const uint8_t SendMapSendData = 0x1;
const uint8_t SendMapRecvData = 0x2;
const uint8_t SendMapExchData = 0x3;
const uint8_t ReplyMapRecvData = 0xd;
const uint8_t ReplyMapExchData = 0xe;
const size_t MapSendSize = 0x1001;
const size_t MapRecvSize = 0xfff;
const size_t MapExchSize = 0x1000;
const size_t MapDefaultSize = 0x1000;
const char* SendRawData = RawData1;
const size_t SendRawDataSize = sizeof(RawData1);
const char* ReplyRawData = RawData2;
const size_t ReplyRawDataSize = sizeof(RawData2);

enum PortType
{
    PortType_Port,
    PortType_NamedPort,
    PortType_Session,
};

const PortType PortTypeArray[] = {
    PortType_Port,
    PortType_NamedPort,
    PortType_Session,
};

enum DataType
{
    DataType_ProcessId,
    DataType_CopyHandle,
    DataType_MoveHandle,
    DataType_PointerWithIpc,
    DataType_PointerWithOne,
    DataType_PointerWithMulti,
    DataType_MapSend,
    DataType_MapRecv,
    DataType_MapExch,
    DataType_Raw,
    DataType_All,
};

const DataType DataTypeArray[] = {
    DataType_ProcessId,
    DataType_CopyHandle,
    DataType_MoveHandle,
    DataType_PointerWithIpc,
    DataType_PointerWithOne,
    DataType_PointerWithMulti,
    DataType_MapSend,
    DataType_MapRecv,
    DataType_MapExch,
    DataType_Raw,
    DataType_All,
};

enum TestState
{
    TestState_ServerNotReceive,
    TestState_ServerSuccessReceive,
    TestState_ServerMissReceive,
    TestState_ServerReply,
};

class TestCondition
{
    public:
        TestCondition() : mRecvMethod(-1), mPortIndex(-1), mDataIndex(-1), mEnd(false) {};

        bool Next()
        {
            if (mEnd)
            {
                return false;
            }

            if (mRecvMethod == -1)
            {
                mRecvMethod = 0;
                mPortIndex = 0;
                mDataIndex = 0;
                return true;
            }

            mRecvMethod++;

            if(mRecvMethod >= 2)
            {
                mRecvMethod = 0;
                mDataIndex++;
            }

            if (mDataIndex >= static_cast<int32_t>(sizeof(DataTypeArray) / sizeof(DataType)))
            {
                mDataIndex = 0;
                mPortIndex++;
            }

            if (mPortIndex >= static_cast<int32_t>(sizeof(PortTypeArray) / sizeof(PortType)))
            {
                mEnd = true;
                return false;
            }

            return true;
        }

        // Next 前に呼ばれるとデータアボートになるので注意
        DataType GetDataType() const
        {
            return DataTypeArray[mDataIndex];
        }
        PortType GetPortType() const
        {
            return PortTypeArray[mPortIndex];
        }
        bool GetServerBuffer() const
        {
            return mRecvMethod > 0;
        }

    private:
        int32_t mRecvMethod;
        int32_t mPortIndex;
        int32_t mDataIndex;
        bool mEnd;
};

#if 0 // For Debug
void ShowHeader(nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
    NN_LOG("tag:  0x%x\n", ipcHeader.GetTag());
    NN_LOG("ptr:  0x%x\n", ipcHeader.GetPointerNum());
    NN_LOG("send: 0x%x\n", ipcHeader.GetSendNum());
    NN_LOG("recv: 0x%x\n", ipcHeader.GetReceiveNum());
    NN_LOG("exch: 0x%x\n", ipcHeader.GetExchangeNum());
    NN_LOG("raw:  0x%x\n", ipcHeader.GetRawNum());
    NN_LOG("list: 0x%x\n", ipcHeader.GetReceiveListNum());
    NN_LOG("sp:   0x%x\n", ipcHeader.GetSpecialNum());
}
#endif

void MakeTestNamedPortSession(nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    const char* portName = "NamedPort";
    result = nn::svc::ManageNamedPort(&serverPort, portName, 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);

    result = nn::svc::ConnectToNamedPort(pOutClientSession, portName);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(pOutServerSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ManageNamedPort(&serverPort, portName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

void MakeTestPortSession(nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    result = nn::svc::ConnectToPort(pOutClientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(pOutServerSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

void MakeTestPureSession(nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession)
{
    nn::Result result;
    result = nn::svc::CreateSession(pOutServerSession, pOutClientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
}

void MakeTestSession(
        nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession, PortType port)
{
    switch(port)
    {
        case PortType_NamedPort:
            MakeTestNamedPortSession(pOutServerSession, pOutClientSession);
            break;
        case PortType_Port:
            MakeTestPortSession(pOutServerSession, pOutClientSession);
            break;
        case PortType_Session:
            MakeTestPureSession(pOutServerSession, pOutClientSession);
            break;
        default: FAIL();
    }
}

void MakeSendIpcAllData(
        nn::svc::Handle* pCopyHandle, nn::svc::Handle* pMoveHandle, nn::Bit32* pMsgBuffer)
{
    char* recvBuffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };
    size_t recvBufferSizes[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };

    MakeAllData(
            pCopyHandle, pMoveHandle,
            pMsgBuffer,
            SendPointData, SendPointDataSize, SendPointDataNum,
            g_IpcMapBuffer, MapDefaultSize, SendMapSendData,
            g_IpcMapBuffer + MapDefaultSize, MapDefaultSize, SendMapRecvData,
            g_IpcMapBuffer + MapDefaultSize * 2, MapDefaultSize, SendMapExchData,
            SendRawData, SendRawDataSize,
            recvBuffers, recvBufferSizes, 2
            );
}

void MakeReplyIpcAllData(
        nn::svc::Handle* pCopyHandle, nn::svc::Handle* pMoveHandle,
        nn::Bit32* pMsgBuffer, uintptr_t sendMapAddr, uintptr_t recvMapAddr, uintptr_t exchMapAddr,
        bool doMakeMap = true)
{
    char* recvBuffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };
    size_t recvBufferSizes[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };

    if (doMakeMap)
    {
        MakeAllData(
                pCopyHandle, pMoveHandle,
                pMsgBuffer,
                ReplyPointData, ReplyPointDataSize, ReplyPointDataNum,
                reinterpret_cast<char*>(sendMapAddr), MapDefaultSize,
                reinterpret_cast<char*>(recvMapAddr), MapDefaultSize, ReplyMapRecvData,
                reinterpret_cast<char*>(exchMapAddr), MapDefaultSize, ReplyMapExchData,
                ReplyRawData, ReplyRawDataSize,
                recvBuffers, recvBufferSizes, 2
                );
    }
    else
    {
        MakeAllData(
                pCopyHandle, pMoveHandle,
                pMsgBuffer,
                ReplyPointData, ReplyPointDataSize, ReplyPointDataNum,
                reinterpret_cast<char*>(sendMapAddr), 0,
                reinterpret_cast<char*>(recvMapAddr), 0, ReplyMapRecvData,
                reinterpret_cast<char*>(exchMapAddr), 0, ReplyMapExchData,
                ReplyRawData, ReplyRawDataSize,
                recvBuffers, recvBufferSizes, 2
                );
    }
}

void MakeSendProcessIdData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyProcessIdData(pMsgBuffer);
}

void MakeSendCopyHandleData(nn::svc::Handle* pOut, nn::Bit32* pMsgBuffer)
{
    MakeOnlyCopyHandleData(pOut, pMsgBuffer);
}

void MakeSendMoveHandleData(nn::svc::Handle* pOut, nn::Bit32* pMsgBuffer)
{
    MakeOnlyMoveHandleData(pOut, pMsgBuffer);
}

void MakeSendPointerDataWithIpcBuffer(nn::Bit32* pMsgBuffer, bool doSend)
{
    if (doSend)
    {
        MakeOnlyPointerWithIpcBufferData(
                pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum);
    }
    else
    {
        MakeOnlyPointerWithIpcBufferData(
                pMsgBuffer, ReplyPointData, ReplyPointDataSize, ReplyPointDataNum);
    }
}

void MakeSendPointerDataWithOneUserBufferData(nn::Bit32* pMsgBuffer, bool doSend)
{
    if (doSend)
    {
        MakeOnlyPointerWithOneUserBuffer(
                pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
                g_IpcRecvBuffer[0], sizeof(g_IpcRecvBuffer[0]));
    }
    else
    {
        MakeOnlyPointerWithOneUserBuffer(
                pMsgBuffer, ReplyPointData, ReplyPointDataSize, ReplyPointDataNum,
                g_IpcRecvBuffer[0], sizeof(g_IpcRecvBuffer[0]));
    }
}

void MakeSendPointerDataWithMultiUserBufferData(nn::Bit32* pMsgBuffer, bool doSend)
{
    size_t bufferSize[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };
    char* buffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };
    if (doSend)
    {
        MakeOnlyPointerWithMultiUserBuffer(
                pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
                buffers, bufferSize);
    }
    else
    {
        MakeOnlyPointerWithMultiUserBuffer(
                pMsgBuffer, ReplyPointData, ReplyPointDataSize, ReplyPointDataNum,
                buffers, bufferSize);
    }
}

void MakeSendMapSendData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyMapSendData(pMsgBuffer, g_IpcMapBuffer, MapSendSize, SendMapSendData);
}

void MakeSendMapRecvData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyMapReceiveData(pMsgBuffer, g_IpcMapBuffer, MapRecvSize, SendMapRecvData);
}

void MakeReplyMapRecvData(nn::Bit32* pMsgBuffer, uintptr_t addr, bool doMakeMap = true)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    if (doMakeMap)
    {
        WriteMapData(reinterpret_cast<char*>(addr), MapRecvSize, ReplyMapRecvData);
    }
}

void MakeSendMapExchData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyMapExchangeData(pMsgBuffer, g_IpcMapBuffer, MapExchSize, SendMapExchData);
}

void MakeReplyMapExchData(nn::Bit32* pMsgBuffer, uintptr_t addr, bool doMakeMap = true)
{
    if (doMakeMap)
    {
        WriteMapData(reinterpret_cast<char*>(addr), MapExchSize, ReplyMapExchData);
    }
}

void MakeSendRawData(nn::Bit32* pMsgBuffer, bool doSend)
{
    if (doSend)
    {
        MakeOnlyRawData(pMsgBuffer, SendRawData, SendRawDataSize);
    }
    else
    {
        MakeOnlyRawData(pMsgBuffer, ReplyRawData, ReplyRawDataSize);
    }
}

void CheckServerProcessId(nn::Bit32* pMsgBuffer)
{
    nn::Bit64 pid;
    nn::Result result = nn::svc::GetProcessId(&pid, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_SUCCESS(result);
    CheckOnlyProcessIdData(pid, pMsgBuffer);
}

void CheckServerCopyHandle(nn::Bit32* pMsgBuffer)
{
    CheckOnlyCopyHandleData(pMsgBuffer, true);
}

void CheckServerPointerWithIpc(nn::Bit32* pMsgBuffer, size_t msgSize)
{
    CheckOnlyPointerWithIpcBufferData(
            SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer, msgSize);
}

void CheckServerPointerWithOne(nn::Bit32* pMsgBuffer)
{
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_IpcRecvBuffer[0]);
    size_t bufSize = sizeof(g_IpcRecvBuffer[0]);
    CheckOnlyPointerWithOneUserBuffer(
            addr, bufSize,
            SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer);
}

void CheckServerPointerWithMulti(nn::Bit32* pMsgBuffer)
{
    size_t bufferSize[] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };
    uintptr_t buffers[] = {
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[0]),
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[1])
    };
    CheckOnlyPointerWithMultiUserBuffer(
            buffers, bufferSize,
            SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer);
}

void CheckServerMapSendData(uintptr_t* pOut, nn::Bit32* pMsgBuffer, bool doCheck = true)
{
    if (doCheck)
    {
        CheckOnlyMapSendData(pOut, MapSendSize, SendMapSendData, pMsgBuffer);
    }
}

void CheckServerMapRecvData(uintptr_t* pOut, nn::Bit32* pMsgBuffer, bool doCheck = true)
{
    if (doCheck)
    {
        CheckOnlyMapReceiveData(pOut, MapRecvSize, SendMapRecvData, pMsgBuffer);
    }
}

void CheckServerMapExchData(uintptr_t* pOut, nn::Bit32* pMsgBuffer, bool doCheck = true)
{
    if (doCheck)
    {
        CheckOnlyMapExchangeData(pOut, MapExchSize, SendMapExchData, pMsgBuffer);
    }
}

void CheckServerRawData(nn::Bit32* pMsgBuffer)
{
    CheckOnlyRawData(SendRawData, SendRawDataSize, pMsgBuffer);
}

void CheckServerAllData(
        uintptr_t* sendMapAddr, uintptr_t* receiveMapAddr, uintptr_t* exchangeMapAddr,
        nn::Bit32* pMsgBuffer, bool doCheckMap = true)
{
    uintptr_t recvBufferAddresses[2] = {
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[0]),
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[1]),
    };
    size_t recvBufferSizes[2] = {
        sizeof(g_IpcRecvBuffer[0]),
        sizeof(g_IpcRecvBuffer[1]),
    };

    if (doCheckMap)
    {
        CheckAllData(
                sendMapAddr, receiveMapAddr, exchangeMapAddr,
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, true, false,
                SendPointData, SendPointDataSize, SendPointDataNum,
                recvBufferAddresses, recvBufferSizes,
                MapDefaultSize, SendMapSendData,
                MapDefaultSize, SendMapRecvData,
                MapDefaultSize, SendMapExchData,
                SendRawData, SendRawDataSize,
                pMsgBuffer
                );
    }
    else
    {
        CheckAllData(
                sendMapAddr, receiveMapAddr, exchangeMapAddr,
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, true, false,
                SendPointData, SendPointDataSize, SendPointDataNum,
                recvBufferAddresses, recvBufferSizes,
                0, 0,
                0, 0,
                0, 0,
                SendRawData, SendRawDataSize,
                pMsgBuffer
                );
    }
}

struct TestData
{
    nn::svc::Handle copyHandle;
    nn::svc::Handle moveHandle;
    uintptr_t buffer;
    size_t bufferSize;
    uintptr_t sendAddr;
    uintptr_t recvAddr;
    uintptr_t exchAddr;
};

void MakeClientData(TestData* testData, DataType dataType)
{
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(testData->buffer);
    switch(dataType)
    {
    case DataType_ProcessId:
        MakeSendProcessIdData(pMsgBuffer);
        break;
    case DataType_CopyHandle:
        MakeSendCopyHandleData(&testData->copyHandle, pMsgBuffer);
        break;
    case DataType_MoveHandle:
        break;
    case DataType_PointerWithIpc:
        MakeSendPointerDataWithIpcBuffer(pMsgBuffer, true);
        break;
    case DataType_PointerWithOne:
        MakeSendPointerDataWithOneUserBufferData(pMsgBuffer, true);
        break;
    case DataType_PointerWithMulti:
        MakeSendPointerDataWithMultiUserBufferData(pMsgBuffer, true);
        break;
    case DataType_MapSend:
        MakeSendMapSendData(pMsgBuffer);
        break;
    case DataType_MapRecv:
        MakeSendMapRecvData(pMsgBuffer);
        break;
    case DataType_MapExch:
        MakeSendMapExchData(pMsgBuffer);
        break;
    case DataType_Raw:
        MakeSendRawData(pMsgBuffer, true);
        break;
    case DataType_All:
        MakeSendIpcAllData(&testData->copyHandle, nullptr, pMsgBuffer);
        break;
    default: FAIL();
    }
}

void MakeServerData(TestData* testData, DataType dataType, bool isBuffer, bool doMakeMap = true)
{
    nn::Bit32* pMsgBuffer;
    if (isBuffer)
    {
        pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_IpcServerBuffer);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    }

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    switch(dataType)
    {
    case DataType_ProcessId:
        MakeSendProcessIdData(pMsgBuffer);
        break;
    case DataType_CopyHandle:
        MakeSendCopyHandleData(&testData->copyHandle, pMsgBuffer);
        break;
    case DataType_MoveHandle:
        MakeSendMoveHandleData(&testData->moveHandle, pMsgBuffer);
        break;
    case DataType_PointerWithIpc:
        MakeSendPointerDataWithIpcBuffer(pMsgBuffer, false);
        break;
    case DataType_PointerWithOne:
        MakeSendPointerDataWithOneUserBufferData(pMsgBuffer, false);
        break;
    case DataType_PointerWithMulti:
        MakeSendPointerDataWithMultiUserBufferData(pMsgBuffer, false);
        break;
    case DataType_MapSend:
        break;
    case DataType_MapRecv:
        MakeReplyMapRecvData(pMsgBuffer, testData->recvAddr, doMakeMap);
        break;
    case DataType_MapExch:
        MakeReplyMapExchData(pMsgBuffer, testData->exchAddr, doMakeMap);
        break;
    case DataType_Raw:
        MakeSendRawData(pMsgBuffer, false);
        break;
    case DataType_All:
        MakeReplyIpcAllData(
                &testData->copyHandle, &testData->moveHandle, pMsgBuffer,
                testData->sendAddr, testData->recvAddr, testData->exchAddr, doMakeMap);
        break;
    default: FAIL();
    }
}

void CheckClientBufferState(const TestData* testData, TestState state)
{
    nn::svc::MemoryInfo info;
    uintptr_t addr = testData->buffer;

    GetMemoryInfo(&info, addr);
    switch(state)
    {
        case TestState_ServerNotReceive:
            NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
            NN_ASSERT(info.attribute == 0);
            break;
        case TestState_ServerMissReceive:
            NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
            NN_ASSERT(info.attribute == 0);
            break;
        case TestState_ServerReply:
            NN_ASSERT(info.attribute == 0);
            NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
            break;
        case TestState_ServerSuccessReceive:
            NN_ASSERT(info.attribute == nn::svc::MemoryAttribute_Locked);
            NN_ASSERT(info.permission == nn::svc::MemoryPermission_None);
            break;
        default: FAIL();
    }
}

void CheckServerMapArea(uintptr_t addr, size_t mapSize, TestState state, bool send)
{
    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, addr);
    NN_ASSERT(info.baseAddress <= addr);
    NN_ASSERT(info.baseAddress + info.size - 1 >= addr + mapSize - 1);
    switch(state)
    {
        case TestState_ServerNotReceive:
            NN_ASSERT(info.state == nn::svc::MemoryState_Free);
            NN_ASSERT(info.permission == nn::svc::MemoryPermission_None);
            break;
        case TestState_ServerMissReceive:
            NN_ASSERT(info.state == nn::svc::MemoryState_Free);
            NN_ASSERT(info.permission == nn::svc::MemoryPermission_None);
        case TestState_ServerReply:
            NN_ASSERT(info.state == nn::svc::MemoryState_Free);
            NN_ASSERT(info.permission == nn::svc::MemoryPermission_None);
            break;
        case TestState_ServerSuccessReceive:
            NN_ASSERT(info.state == nn::svc::MemoryState_Ipc);
            if (send)
            {
                NN_ASSERT(info.permission == nn::svc::MemoryPermission_Read);
            }
            else
            {
                NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
            }
            break;
        default: FAIL();
    }
}

void CheckClientState(const TestData* testData, DataType dataType, TestState state)
{
    nn::Result result;

    CheckClientBufferState(testData, state);

    switch(dataType)
    {
    case DataType_ProcessId:
        break;
    case DataType_CopyHandle:
        result = nn::svc::CloseHandle(testData->copyHandle);
        ASSERT_RESULT_SUCCESS(result);
        break;
    case DataType_MoveHandle:
        break;
    case DataType_PointerWithIpc:
        break;
    case DataType_PointerWithOne:
        break;
    case DataType_PointerWithMulti:
        break;
    case DataType_MapSend:
        break;
    case DataType_MapRecv:
        break;
    case DataType_MapExch:
        break;
    case DataType_Raw:
        break;
    case DataType_All:
        result = nn::svc::CloseHandle(testData->copyHandle);
        ASSERT_RESULT_SUCCESS(result);
        break;
    default: FAIL();
    }
}

void CheckServerData(TestData* testData, DataType dataType, bool isBuffer, bool doCheckMap = true)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (isBuffer)
    {
        pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_IpcServerBuffer);
        msgSize = sizeof(g_IpcServerBuffer);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    switch(dataType)
    {
    case DataType_ProcessId:
        CheckServerProcessId(pMsgBuffer);
        break;
    case DataType_CopyHandle:
        CheckServerCopyHandle(pMsgBuffer);
        break;
    case DataType_MoveHandle:
        break;
    case DataType_PointerWithIpc:
        CheckServerPointerWithIpc(pMsgBuffer, msgSize);
        break;
    case DataType_PointerWithOne:
        CheckServerPointerWithOne(pMsgBuffer);
        break;
    case DataType_PointerWithMulti:
        CheckServerPointerWithMulti(pMsgBuffer);
        break;
    case DataType_MapSend:
        CheckServerMapSendData(&testData->sendAddr, pMsgBuffer, doCheckMap);
        break;
    case DataType_MapRecv:
        CheckServerMapRecvData(&testData->recvAddr, pMsgBuffer, doCheckMap);
        break;
    case DataType_MapExch:
        CheckServerMapExchData(&testData->exchAddr, pMsgBuffer, doCheckMap);
        break;
    case DataType_Raw:
        CheckServerRawData(pMsgBuffer);
        break;
    case DataType_All:
        CheckServerAllData(
                &testData->sendAddr, &testData->recvAddr, &testData->exchAddr,
                pMsgBuffer, doCheckMap);
        break;
    default: FAIL();
    }
}

void CheckServerState(
        TestData* testData, DataType dataType, TestState state,
        bool isMoveClose, bool doCheckMap = true)
{
    nn::Result result;

    CheckClientBufferState(testData, state);

    switch(dataType)
    {
    case DataType_ProcessId:
        break;
    case DataType_CopyHandle:
        if (state == TestState_ServerReply)
        {
            result = nn::svc::CloseHandle(testData->copyHandle);
            NN_ASSERT_RESULT_SUCCESS(result);
        }
        break;
    case DataType_MoveHandle:
        if (state == TestState_ServerReply)
        {
            result = nn::svc::CloseHandle(testData->moveHandle);
            if (isMoveClose)
            {
                NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
            }
            else
            {
                NN_ASSERT_RESULT_SUCCESS(result);
            }
        }
        break;
    case DataType_PointerWithIpc:
        break;
    case DataType_PointerWithOne:
        break;
    case DataType_PointerWithMulti:
        break;
    case DataType_MapSend:
        if (doCheckMap)
        {
            CheckServerMapArea(testData->sendAddr, MapSendSize, state, true);
        }
        break;
    case DataType_MapRecv:
        if (doCheckMap)
        {
            CheckServerMapArea(testData->recvAddr, MapRecvSize, state, false);
        }
        break;
    case DataType_MapExch:
        if (doCheckMap)
        {
            CheckServerMapArea(testData->exchAddr, MapExchSize, state, false);
        }
        break;
    case DataType_Raw:
        break;
    case DataType_All:
        if (state == TestState_ServerReply)
        {
            result = nn::svc::CloseHandle(testData->copyHandle);
            NN_ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(testData->moveHandle);
            if (isMoveClose)
            {
                NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
            }
            else
            {
                NN_ASSERT_RESULT_SUCCESS(result);
            }
        }

        if (doCheckMap)
        {
            CheckServerMapArea(testData->sendAddr, MapDefaultSize, state, true);

            CheckServerMapArea(testData->recvAddr, MapDefaultSize, state, false);

            CheckServerMapArea(testData->exchAddr, MapDefaultSize, state, false);
        }
        break;
    default: FAIL();
    }
}

struct ReceiveInfo
{
    nn::svc::Handle session;
    nn::svc::Handle closeHandle;
    DataType dataType;
    uintptr_t buffer;
    size_t bufferSize;
    bool isBuffer;
    bool isMoveClose;
    int numReqeust;
    int receiveIndex;
};

void SetServerHeader(bool isBuffer, DataType dataType)
{
    nn::Bit32* pMsgBuffer;

    if (isBuffer)
    {
        pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_IpcServerBuffer);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    }

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    int offset;

    nn::svc::ipc::MessageBuffer::ReceiveListEntry
        recvEntry1(g_IpcRecvBuffer[0], sizeof(g_IpcRecvBuffer[0]));

    nn::svc::ipc::MessageBuffer::ReceiveListEntry
        recvEntry2(g_IpcRecvBuffer[1], sizeof(g_IpcRecvBuffer[1]));

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 1);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcOneHeader(0, 0, 0, 0, 0, 0, 0, 2);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcMultiHeader(0, 0, 0, 0, 0, 0, 0, 4);

    switch(dataType)
    {
    case DataType_ProcessId:
        ipcMsg.SetNull();
        break;
    case DataType_CopyHandle:
        ipcMsg.SetNull();
        break;
    case DataType_MoveHandle:
        ipcMsg.SetNull();
        break;
    case DataType_PointerWithIpc:
        offset = ipcMsg.Set(ipcHeader);
        break;
    case DataType_PointerWithOne:
        offset = ipcMsg.Set(ipcOneHeader);
        offset = ipcMsg.Set(offset, recvEntry1);
        break;
    case DataType_PointerWithMulti:
        offset = ipcMsg.Set(ipcMultiHeader);
        offset = ipcMsg.Set(offset, recvEntry1);
        offset = ipcMsg.Set(offset, recvEntry2);
        break;
    case DataType_MapSend:
        ipcMsg.SetNull();
        break;
    case DataType_MapRecv:
        ipcMsg.SetNull();
        break;
    case DataType_MapExch:
        ipcMsg.SetNull();
        break;
    case DataType_Raw:
        ipcMsg.SetNull();
        break;
    case DataType_All:
        offset = ipcMsg.Set(ipcMultiHeader);
        offset = ipcMsg.Set(offset, recvEntry1);
        offset = ipcMsg.Set(offset, recvEntry2);
        break;
    default: FAIL();
    }
}

void MissReceiveThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ReceiveInfo* info = reinterpret_cast<ReceiveInfo*>(arg);

    int32_t index;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_IpcServerBuffer);
    size_t msgSize = sizeof(g_IpcServerBuffer);;

    SetServerHeader(info->isBuffer, info->dataType);

    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }

    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(index == 0);

    uintptr_t bufferAddress = 0;
    for (int i = 0; i < info->numReqeust; i++)
    {
        nn::svc::MemoryInfo memoryInfo;
        GetMemoryInfo(&memoryInfo, info->buffer + info->bufferSize * i);
        if (memoryInfo.attribute == nn::svc::MemoryAttribute_Locked)
        {
            bufferAddress = info->buffer + info->bufferSize * i;
            info->receiveIndex = i;
            break;
        }
    }
    NN_ASSERT(bufferAddress != 0);

    result = nn::svc::CloseHandle(info->closeHandle);
    NN_ASSERT_RESULT_SUCCESS(result);

    TestData testData;
    testData.buffer = bufferAddress;
    testData.bufferSize = info->bufferSize;
    if (info->closeHandle == info->session)
    {
        CheckServerData(&testData, info->dataType, info->isBuffer, false);
        CheckServerState(&testData, info->dataType, TestState_ServerMissReceive, info->isMoveClose, false);
        MakeServerData(&testData, info->dataType, info->isBuffer, false);
    }
    else
    {
        CheckServerData(&testData, info->dataType, info->isBuffer);
        CheckServerState(&testData, info->dataType, TestState_ServerSuccessReceive, info->isMoveClose);
        MakeServerData(&testData, info->dataType, info->isBuffer);
    }


    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 0,
                    info->session, 0);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 0,
                    info->session, 0);
    }
    if (info->closeHandle == info->session)
    {
        NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }
    else
    {
        NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
    }
    CheckServerState(&testData, info->dataType, TestState_ServerReply, info->isMoveClose, info->closeHandle != info->session);
}

#ifdef SEND_ASYNC_REPLY_TEST
void MissReplyThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ReceiveInfo* info = reinterpret_cast<ReceiveInfo*>(arg);

    int32_t index;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_IpcServerBuffer);
    size_t msgSize = sizeof(g_IpcServerBuffer);;

    SetServerHeader(info->isBuffer, info->dataType);

    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }

    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(index == 0);

    TestData testData;
    testData.buffer = info->buffer;
    testData.bufferSize = info->bufferSize;
    CheckServerData(&testData, info->dataType, info->isBuffer);
    CheckServerState(&testData, info->dataType, TestState_ServerSuccessReceive, info->isMoveClose);

    MakeServerData(&testData, info->dataType, info->isBuffer);

    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 0,
                    info->session, 0);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 0,
                    info->session, 0);
    }

    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // 同じ優先度で動くので、相手が受信する前に閉じられるはず
    result = nn::svc::CloseHandle(info->closeHandle);
    NN_ASSERT_RESULT_SUCCESS(result);

    CheckServerState(&testData, info->dataType, TestState_ServerReply, info->isMoveClose);
}
#endif

void MissReceiveWithSequenceThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ReceiveInfo* info = reinterpret_cast<ReceiveInfo*>(arg);

    NN_ASSERT(g_Sequence == 0);
    int32_t index;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_IpcServerBuffer);
    size_t msgSize = sizeof(g_IpcServerBuffer);;

    SetServerHeader(info->isBuffer, info->dataType);

    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }

    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(index == 0);

    result = nn::svc::CloseHandle(info->closeHandle);
    NN_ASSERT_RESULT_SUCCESS(result);

    uintptr_t bufferAddress = 0;
    for (int i = 0; i < info->numReqeust; i++)
    {
        nn::svc::MemoryInfo memoryInfo;
        GetMemoryInfo(&memoryInfo, info->buffer + info->bufferSize * i);
        if (memoryInfo.attribute == nn::svc::MemoryAttribute_Locked)
        {
            bufferAddress = info->buffer + info->bufferSize * i;
            info->receiveIndex = i;
            break;
        }
    }
    NN_ASSERT(bufferAddress != 0);

    TestData testData;
    testData.buffer = bufferAddress;
    testData.bufferSize = info->bufferSize;
    if (info->closeHandle == info->session)
    {
        CheckServerData(&testData, info->dataType, info->isBuffer, false);
        CheckServerState(&testData, info->dataType, TestState_ServerMissReceive, info->isMoveClose, false);
        MakeServerData(&testData, info->dataType, info->isBuffer, false);
    }
    else
    {
        CheckServerData(&testData, info->dataType, info->isBuffer);
        CheckServerState(&testData, info->dataType, TestState_ServerSuccessReceive, info->isMoveClose);
        MakeServerData(&testData, info->dataType, info->isBuffer);
    }

    g_Sequence = 1;
    while(g_Sequence == 1)
    {
        nn::svc::SleepThread(SleepTime);
    }

    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 0,
                    info->session, 0);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 0,
                    info->session, 0);
    }
    if (info->closeHandle == info->session)
    {
        NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }
    else
    {
        NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
    }
    CheckServerState(&testData, info->dataType, TestState_ServerReply, info->isMoveClose, info->closeHandle != info->session);
}

void CloseReceivingThread(uintptr_t arg)
{
    NN_ASSERT(g_Sequence == 0);
    AutoThreadExit autoExit;
    nn::Result result;
    ReceiveInfo* info = reinterpret_cast<ReceiveInfo*>(arg);

    int32_t index;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_IpcServerBuffer);
    size_t msgSize = sizeof(g_IpcServerBuffer);;

    SetServerHeader(info->isBuffer, info->dataType);

    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }

    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(index == 0);

    TestData testData;
    testData.buffer = info->buffer;
    testData.bufferSize = info->bufferSize;

    result = nn::svc::CloseHandle(info->closeHandle);
    NN_ASSERT_RESULT_SUCCESS(result);

    CheckServerData(&testData, info->dataType, info->isBuffer);
    CheckServerState(&testData, info->dataType, TestState_ServerSuccessReceive, info->isMoveClose);
    MakeServerData(&testData, info->dataType, info->isBuffer);

    g_Sequence = 1;
    while(g_Sequence == 1)
    {
        nn::svc::SleepThread(SleepTime);
    }

    result = nn::svc::CloseHandle(info->session);
    NN_ASSERT_RESULT_SUCCESS(result);
    CheckServerState(&testData, info->dataType, TestState_ServerMissReceive, info->isMoveClose);

    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 0,
                    info->session, 0);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 0,
                    info->session, 0);
    }

    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    CheckServerState(&testData, info->dataType, TestState_ServerReply, info->isMoveClose);
}

#ifdef SEND_ASYNC_REPLY_TEST
void CloseReplyThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ReceiveInfo* info = reinterpret_cast<ReceiveInfo*>(arg);

    int32_t index;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_IpcServerBuffer);
    size_t msgSize = sizeof(g_IpcServerBuffer);;

    SetServerHeader(info->isBuffer, info->dataType);

    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 1,
                    nn::svc::INVALID_HANDLE_VALUE, -1);
    }

    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(index == 0);

    TestData testData;
    testData.buffer = info->buffer;
    testData.bufferSize = info->bufferSize;
    CheckServerData(&testData, info->dataType, info->isBuffer);
    CheckServerState(&testData, info->dataType, TestState_ServerSuccessReceive, info->isMoveClose);

    MakeServerData(&testData, info->dataType, info->isBuffer);

    if (info->isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, addr, msgSize, &info->session, 0,
                    info->session, 0);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &info->session, 0,
                    info->session, 0);
    }

    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // 同じ優先度で動くので、相手が受信する前に閉じられるはず
    result = nn::svc::CloseHandle(info->closeHandle);
    NN_ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(info->session);
    NN_ASSERT_RESULT_SUCCESS(result);

    CheckServerState(&testData, info->dataType, TestState_ServerReply, info->isMoveClose);
}
#endif

} // namespace

TEST(SendAsyncRequestWithUserBuffer, MissSendServer)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    uintptr_t bufferAddr = reinterpret_cast<uintptr_t>(g_IpcClientBuffer);
    size_t bufferSize = sizeof(g_IpcClientBuffer);

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        TestData data;
        data.buffer = bufferAddr;
        data.bufferSize = bufferSize;
        MakeClientData(&data, cond.GetDataType());

        nn::svc::Handle eventHandle;
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, bufferAddr, bufferSize, clientSession);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose eventCloser(eventHandle);

        result = nn::svc::CloseHandle(serverSession);
        ASSERT_RESULT_SUCCESS(result);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr));
        CheckClientState(&data, cond.GetDataType(), TestState_ServerNotReceive);
    }
}

TEST(SendAsyncRequestWithUserBuffer, MissReceiveServer)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    uintptr_t bufferAddr = reinterpret_cast<uintptr_t>(g_IpcClientBuffer);
    size_t bufferSize = sizeof(g_IpcClientBuffer);

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        ReceiveInfo info;
        info.session = serverSession;
        info.isBuffer = cond.GetServerBuffer();
        info.dataType = cond.GetDataType();
        info.closeHandle = serverSession;
        info.isMoveClose = false;
        info.buffer = bufferAddr;
        info.bufferSize = bufferSize;
        info.numReqeust = 1;
        info.receiveIndex = -1;

        uintptr_t pc = reinterpret_cast<uintptr_t>(MissReceiveThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
        uintptr_t param = reinterpret_cast<uintptr_t>(&info);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = g_ProcessIdealCore;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        TestData data;
        data.buffer = bufferAddr;
        data.bufferSize = bufferSize;
        MakeClientData(&data, cond.GetDataType());

        nn::svc::Handle eventHandle;
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, bufferAddr,bufferSize, clientSession);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose eventCloser(eventHandle);

        // このスレッドで受信後にサーバーセッションを閉じる
        thread.Wait();

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr));
        CheckClientState(&data, cond.GetDataType(), TestState_ServerMissReceive);
    }
}

#ifdef SEND_ASYNC_REPLY_TEST
TEST(SendAsyncRequestWithUserBuffer, MissReplyServer)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    uintptr_t bufferAddr = reinterpret_cast<uintptr_t>(g_IpcClientBuffer);
    size_t bufferSize = sizeof(g_IpcClientBuffer);

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        ReceiveInfo info;
        info.session = serverSession;
        info.isBuffer = cond.GetServerBuffer();
        info.dataType = cond.GetDataType();
        info.closeHandle = serverSession;
        info.isMoveClose = true;
        info.buffer = bufferAddr;
        info.bufferSize = bufferSize;
        info.numReqeust = 1;
        info.receiveIndex = -1;

        uintptr_t pc = reinterpret_cast<uintptr_t>(MissReplyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
        uintptr_t param = reinterpret_cast<uintptr_t>(&info);
        int32_t priority;
        int32_t idealCore = g_ProcessIdealCore;

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_SUCCESS(result);

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        TestData data;
        data.buffer = bufferAddr;
        data.bufferSize = bufferSize;
        MakeClientData(&data, cond.GetDataType());

        nn::svc::Handle eventHandle;
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, bufferAddr, bufferSize, clientSession);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose eventCloser(eventHandle);

        // このスレッドで受信後にサーバーセッションを閉じる
        thread.Wait();

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr));
        CheckClientState(&data, cond.GetDataType(), TestState_ServerReply);
    }
}
#endif

TEST(SendAsyncRequestWithUserBuffer, MissSendClient)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    uintptr_t bufferAddr = reinterpret_cast<uintptr_t>(g_IpcClientBuffer);
    size_t bufferSize = sizeof(g_IpcClientBuffer);

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        TestData data;
        data.buffer = bufferAddr;
        data.bufferSize = bufferSize;
        MakeClientData(&data, cond.GetDataType());

        nn::svc::Handle eventHandle;
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, bufferAddr,bufferSize, clientSession);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose eventCloser(eventHandle);

        result = nn::svc::CloseHandle(clientSession);
        ASSERT_RESULT_SUCCESS(result);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr));
        CheckClientState(&data, cond.GetDataType(), TestState_ServerNotReceive);
    }
}

TEST(SendAsyncRequestWithUserBuffer, MissReceiveClient)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    uintptr_t bufferAddr = reinterpret_cast<uintptr_t>(g_IpcClientBuffer);
    size_t bufferSize = sizeof(g_IpcClientBuffer);

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        ReceiveInfo info;
        info.session = serverSession;
        info.isBuffer = cond.GetServerBuffer();
        info.dataType = cond.GetDataType();
        info.closeHandle = clientSession;
        info.isMoveClose = true;
        info.buffer = bufferAddr;
        info.bufferSize = bufferSize;
        info.numReqeust = 1;
        info.receiveIndex = -1;

        uintptr_t pc = reinterpret_cast<uintptr_t>(MissReceiveThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
        uintptr_t param = reinterpret_cast<uintptr_t>(&info);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = g_ProcessIdealCore;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        TestData data;
        data.buffer = bufferAddr;
        data.bufferSize = bufferSize;
        MakeClientData(&data, cond.GetDataType());

        nn::svc::Handle eventHandle;
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, bufferAddr, bufferSize, clientSession);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose eventCloser(eventHandle);

        thread.Wait();

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr));
        CheckClientState(&data, cond.GetDataType(), TestState_ServerReply);
    }
}

#ifdef SEND_ASYNC_REPLY_TEST
TEST(SendAsyncRequestWithUserBuffer, MissReplyClient)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    uintptr_t bufferAddr = reinterpret_cast<uintptr_t>(g_IpcClientBuffer);
    size_t bufferSize = sizeof(g_IpcClientBuffer);

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        ReceiveInfo info;
        info.session = serverSession;
        info.isBuffer = cond.GetServerBuffer();
        info.dataType = cond.GetDataType();
        info.closeHandle = clientSession;
        info.isMoveClose = true;
        info.buffer = bufferAddr;
        info.bufferSize = bufferSize;
        info.numReqeust = 1;
        info.receiveIndex = -1;

        uintptr_t pc = reinterpret_cast<uintptr_t>(MissReplyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
        uintptr_t param = reinterpret_cast<uintptr_t>(&info);
        int32_t priority;
        int32_t idealCore = g_ProcessIdealCore;

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_SUCCESS(result);

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        TestData data;
        data.buffer = bufferAddr;
        data.bufferSize = bufferSize;
        MakeClientData(&data, cond.GetDataType());

        nn::svc::Handle eventHandle;
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, bufferAddr, bufferSize, clientSession);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose eventCloser(eventHandle);

        // このスレッドで受信後にクライアントセッションを閉じる
        thread.Wait();

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr));
        CheckClientState(&data, cond.GetDataType(), TestState_ServerReply);
    }
}
#endif

TEST(SendAsyncRequestWithUserBuffer, MultiSendClient)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    const int32_t sendNum = 3;
    size_t bufferSize = 0x1000;
    TestHeap heap(bufferSize * sendNum * 2);
    uintptr_t bufferAddr = heap.GetAddress();

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        nn::svc::Handle eventHandles[sendNum];
        TestData data[sendNum];
        for (int32_t i = 0; i < sendNum; i++)
        {
            data[i].buffer = bufferAddr + bufferSize * i;
            data[i].bufferSize = bufferSize;
            MakeClientData(&data[i], cond.GetDataType());

            result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandles[i], bufferAddr + bufferSize * i, bufferSize, clientSession);
            ASSERT_RESULT_SUCCESS(result);
        }

        cSessionCloser.Close();

        int32_t index;
        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
            ASSERT_RESULT_SUCCESS(result);
        }

        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::CloseHandle(eventHandles[i]);
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr + bufferSize * i));
            CheckClientState(&data[i], cond.GetDataType(), TestState_ServerNotReceive);
        }
    }
}

TEST(SendAsyncRequestWithUserBuffer, MultiSendAndReceiveClient)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    const int32_t sendNum = 3;
    size_t bufferSize = 0x1000;
    TestHeap heap(bufferSize * sendNum * 2);
    uintptr_t bufferAddr = heap.GetAddress();

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        ReceiveInfo info;
        info.session = serverSession;
        info.isBuffer = cond.GetServerBuffer();
        info.dataType = cond.GetDataType();
        info.closeHandle = clientSession;
        info.isMoveClose = true;
        info.buffer = bufferAddr;
        info.bufferSize = bufferSize;
        info.numReqeust = sendNum;
        info.receiveIndex = -1;

        uintptr_t pc = reinterpret_cast<uintptr_t>(MissReceiveWithSequenceThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
        uintptr_t param = reinterpret_cast<uintptr_t>(&info);
        int32_t priority;
        int32_t idealCore = g_ProcessIdealCore;

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle eventHandles[sendNum];
        TestData data[sendNum];
        for (int32_t i = 0; i < sendNum; i++)
        {
            data[i].buffer = bufferAddr + bufferSize * i;
            data[i].bufferSize = bufferSize;
            MakeClientData(&data[i], cond.GetDataType());

            result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandles[i], bufferAddr + bufferSize * i, bufferSize, clientSession);
            ASSERT_RESULT_SUCCESS(result);
        }

        g_Sequence = 0;
        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        while(g_Sequence == 0)
        {
            nn::svc::SleepThread(SleepTime);
        }
        ASSERT_TRUE(g_Sequence == 1);

        int32_t index;
        for (int32_t i = 0; i < sendNum - 1; i++)
        {
            result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::ClearEvent(eventHandles[index]);
            ASSERT_RESULT_SUCCESS(result);
        }

        // １つ要求を受信しているものは、reply されるまで保留される
        result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        g_Sequence = 2;

        thread.Wait();

        result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
        ASSERT_RESULT_SUCCESS(result);

        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::CloseHandle(eventHandles[i]);
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr + bufferSize * i));
            if (i == index)
            {
                CheckClientState(&data[i], cond.GetDataType(), TestState_ServerReply);
            }
            else
            {
                CheckClientState(&data[i], cond.GetDataType(), TestState_ServerNotReceive);
            }
        }
    }
}

TEST(SendAsyncRequestWithUserBuffer, MultiSendServer)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    const int32_t sendNum = 3;
    size_t bufferSize = 0x1000;
    TestHeap heap(bufferSize * sendNum * 2);
    uintptr_t bufferAddr = heap.GetAddress();

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        nn::svc::Handle eventHandles[sendNum];
        TestData data[sendNum];
        for (int32_t i = 0; i < sendNum; i++)
        {
            data[i].buffer = bufferAddr + bufferSize * i;
            data[i].bufferSize = bufferSize;
            MakeClientData(&data[i], cond.GetDataType());

            result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandles[i], bufferAddr + bufferSize * i, bufferSize, clientSession);
            ASSERT_RESULT_SUCCESS(result);
        }

        sSessionCloser.Close();

        int32_t index;
        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
            ASSERT_RESULT_SUCCESS(result);
        }

        for (int32_t i = 0; i < sendNum; i++)
        {
            CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr + bufferSize * i));
            result = nn::svc::CloseHandle(eventHandles[i]);
            ASSERT_RESULT_SUCCESS(result);
            CheckClientState(&data[i], cond.GetDataType(), TestState_ServerNotReceive);
        }
    }
}

TEST(SendAsyncRequestWithUserBuffer, MultiSendAndReceiveServer)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    const int32_t sendNum = 3;
    size_t bufferSize = 0x1000;
    TestHeap heap(bufferSize * sendNum * 2);
    uintptr_t bufferAddr = heap.GetAddress();

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        ReceiveInfo info;
        info.session = serverSession;
        info.isBuffer = cond.GetServerBuffer();
        info.dataType = cond.GetDataType();
        info.closeHandle = serverSession;
        info.isMoveClose = false;
        info.buffer = bufferAddr;
        info.bufferSize = bufferSize;
        info.numReqeust = sendNum;
        info.receiveIndex = -1;

        uintptr_t pc = reinterpret_cast<uintptr_t>(MissReceiveThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
        uintptr_t param = reinterpret_cast<uintptr_t>(&info);
        int32_t priority;
        int32_t idealCore = g_ProcessIdealCore;

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle eventHandles[sendNum];
        TestData data[sendNum];
        for (int32_t i = 0; i < sendNum; i++)
        {
            data[i].buffer = bufferAddr + bufferSize * i;
            data[i].bufferSize = bufferSize;
            MakeClientData(&data[i], cond.GetDataType());

            result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandles[i], bufferAddr + bufferSize * i, bufferSize, clientSession);
            ASSERT_RESULT_SUCCESS(result);
        }

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();
        thread.Wait();

        int32_t index;
        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::ClearEvent(eventHandles[index]);
            ASSERT_RESULT_SUCCESS(result);
        }

        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::CloseHandle(eventHandles[i]);
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr + bufferSize * i));
            if (i == info.receiveIndex)
            {
                CheckClientState(&data[i], cond.GetDataType(), TestState_ServerReply);
            }
            else
            {
                CheckClientState(&data[i], cond.GetDataType(), TestState_ServerNotReceive);
            }
        }
    }
}


TEST(SendAsyncRequestWithUserBuffer, BothCloseBeforeReceive)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    const int32_t sendNum = 3;
    size_t bufferSize = 0x1000;
    TestHeap heap(bufferSize * sendNum * 2);
    uintptr_t bufferAddr = heap.GetAddress();

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        nn::svc::Handle eventHandles[sendNum];
        TestData data[sendNum];
        for (int32_t i = 0; i < sendNum; i++)
        {
            data[i].buffer = bufferAddr + bufferSize * i;
            data[i].bufferSize = bufferSize;
            MakeClientData(&data[i], cond.GetDataType());

            result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandles[i], bufferAddr + bufferSize * i, bufferSize, clientSession);
            ASSERT_RESULT_SUCCESS(result);
        }

        cSessionCloser.Close();
        sSessionCloser.Close();

        int32_t index;
        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
            ASSERT_RESULT_SUCCESS(result);
        }

        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::CloseHandle(eventHandles[i]);
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr + bufferSize * i));
            CheckClientState(&data[i], cond.GetDataType(), TestState_ServerNotReceive);
        }
    }
}

TEST(SendAsyncRequestWithUserBuffer, BothCloseReceiving)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    const int32_t sendNum = 3;
    size_t bufferSize = 0x1000;
    TestHeap heap(bufferSize * sendNum * 2);
    uintptr_t bufferAddr = heap.GetAddress();

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        ReceiveInfo info;
        info.session = serverSession;
        info.isBuffer = cond.GetServerBuffer();
        info.dataType = cond.GetDataType();
        info.closeHandle = clientSession;
        info.isMoveClose = false;
        info.buffer = bufferAddr;
        info.bufferSize = bufferSize;
        info.numReqeust = sendNum;
        info.receiveIndex = -1;

        uintptr_t pc = reinterpret_cast<uintptr_t>(CloseReceivingThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
        uintptr_t param = reinterpret_cast<uintptr_t>(&info);
        int32_t priority;
        int32_t idealCore = g_ProcessIdealCore;

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle eventHandles[sendNum];
        TestData data[sendNum];
        for (int32_t i = 0; i < sendNum; i++)
        {
            data[i].buffer = bufferAddr + bufferSize * i;
            data[i].bufferSize = bufferSize;
            MakeClientData(&data[i], cond.GetDataType());

            result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandles[i], bufferAddr + bufferSize * i, bufferSize, clientSession);
            ASSERT_RESULT_SUCCESS(result);
        }

        g_Sequence = 0;
        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        while(g_Sequence == 0)
        {
            nn::svc::SleepThread(SleepTime);
        }
        ASSERT_TRUE(g_Sequence == 1);

        int32_t index;
        for (int32_t i = 0; i < sendNum - 1; i++)
        {
            result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::ClearEvent(eventHandles[index]);
            ASSERT_RESULT_SUCCESS(result);
        }

        // １つ要求を受信しているものは、サーバーセッションが閉じられるまで保留される
        result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        g_Sequence = 2;

        thread.Wait();

        result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
        ASSERT_RESULT_SUCCESS(result);

        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::CloseHandle(eventHandles[i]);
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr + bufferSize * i));
            if (i == index)
            {
                CheckClientState(&data[i], cond.GetDataType(), TestState_ServerReply);
            }
            else
            {
                CheckClientState(&data[i], cond.GetDataType(), TestState_ServerNotReceive);
            }
        }
    }
}

#ifdef SEND_ASYNC_REPLY_TEST
TEST(SendAsyncRequestWithUserBuffer, BothCloseAfterReply)
{
    TestThreadLeak testHandle;
    nn::Result result;
    TestCondition cond;
    const int32_t sendNum = 3;
    size_t bufferSize = 0x1000;
    TestHeap heap(bufferSize * sendNum * 2);
    uintptr_t bufferAddr = heap.GetAddress();

    while(cond.Next())
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        MakeTestSession(&serverSession, &clientSession, cond.GetPortType());
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        ReceiveInfo info;
        info.session = serverSession;
        info.isBuffer = cond.GetServerBuffer();
        info.dataType = cond.GetDataType();
        info.closeHandle = clientSession;
        info.isMoveClose = true;
        info.buffer = bufferAddr;
        info.bufferSize = bufferSize;
        info.numReqeust = sendNum;
        info.receiveIndex = -1;

        uintptr_t pc = reinterpret_cast<uintptr_t>(CloseReplyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
        uintptr_t param = reinterpret_cast<uintptr_t>(&info);
        int32_t priority;
        int32_t idealCore = g_ProcessIdealCore;

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle eventHandles[sendNum];
        TestData data[sendNum];
        for (int32_t i = 0; i < sendNum; i++)
        {
            data[i].buffer = bufferAddr + bufferSize * i;
            data[i].bufferSize = bufferSize;
            MakeClientData(&data[i], cond.GetDataType());

            result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandles[i], bufferAddr + bufferSize * i, bufferSize, clientSession);
            ASSERT_RESULT_SUCCESS(result);
        }

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();
        thread.Wait();

        int32_t index;
        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::WaitSynchronization(&index, eventHandles, sendNum, 0);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::ClearEvent(eventHandles[index]);
            ASSERT_RESULT_SUCCESS(result);
        }

        for (int32_t i = 0; i < sendNum; i++)
        {
            result = nn::svc::CloseHandle(eventHandles[i]);
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(reinterpret_cast<nn::Bit32*>(bufferAddr + bufferSize * i));
            if (i == info.receiveIndex)
            {
                CheckClientState(&data[i], cond.GetDataType(), TestState_ServerReply);
            }
            else
            {
                CheckClientState(&data[i], cond.GetDataType(), TestState_ServerNotReceive);
            }
        }
    }
}
#endif

