﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/nn_Assert.h>
#include <nn/svc/svc_Tcb.h>

extern "C" void IpcHeaderAsm();
namespace {

const int64_t WaitTime = 1000 * 1000 * 1000; // 1 sec
const int DefaultAlign = 0x1000;
#ifdef INVALID_POINTER_TEST
const int tmpVar = 0;
#endif
nn::Bit32 g_UserBuffer0[0x2000 / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));
char g_Buffer[0x2000] __attribute__((aligned(0x1000)));
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));
bool g_ThreadLoop = false;

// ただ返信をするだけのスレッド
void ReplyThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle handles[1] = { *reinterpret_cast<nn::svc::Handle*>(arg) };
    int32_t index;

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(&index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(&index, handles, 0, handles[0], 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

// 返信を繰り返す
void RepeatReplyThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);
    int32_t index;

    while(g_ThreadLoop)
    {
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handle, 1, nn::svc::INVALID_HANDLE_VALUE, WaitTime);
        if (!g_ThreadLoop)
        {
            break;
        }
        NN_ASSERT_RESULT_SUCCESS(result);
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        result = nn::svc::ReplyAndReceive(
                &index, handle, 0, *handle, 0);
        NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
}

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;
    nn::Bit32 msgBuffer[2];
    SetOnlyIpcTag(msgBuffer, DefaultIpcTag_Send);

    for (int i = 0; i < numState; i++)
    {
        size_t msgSize = 0x1000;

        ASSERT_TRUE(array[i]->GetSize() >= msgSize && ((array[i]->GetSize() & 0xfff) == 0));
        array[i]->Initialize(msgBuffer, sizeof(msgBuffer));

        uintptr_t msgBufferAddr = array[i]->GetAddress();

        nn::svc::Handle eventHandle;
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, msgBufferAddr, msgSize, handle);
        ASSERT_RESULT_SUCCESS(result);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(eventHandle);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;
    nn::Bit32 msgBuffer[2];
    SetOnlyIpcTag(msgBuffer, DefaultIpcTag_Send);

    for (int i = 0; i < numState; i++)
    {
        size_t msgSize = 0x1000;
        if (!(array[i]->GetSize() >= msgSize && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize(msgBuffer, sizeof(msgBuffer));

        uintptr_t msgBufferAddr = array[i]->GetAddress();
        if (array[i]->GetState() == nn::svc::MemoryState_Code)
        {
            msgBufferAddr = reinterpret_cast<uintptr_t>(IpcHeaderAsm);
            nn::Bit32* ptr = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
            ASSERT_EQ(ptr[0], 0x00000001);
            ASSERT_EQ(ptr[1], 0x00000000);

            nn::svc::MemoryInfo info;
            GetMemoryInfo(&info, msgBufferAddr);
            ASSERT_EQ(info.state, nn::svc::MemoryState_Code);
            ASSERT_EQ(info.permission, nn::svc::MemoryPermission_ReadExecute);
            ASSERT_LE(info.baseAddress, msgBufferAddr);
            ASSERT_GE(info.baseAddress + info.size - 1, msgBufferAddr + msgSize - 1);
        }

        nn::svc::Handle eventHandle;
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, msgBufferAddr, msgSize, handle);
        if (array[i]->CanWriteData() || array[i]->GetState() == nn::svc::MemoryState_Code)
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
        else
        {
            ASSERT_RESULT_FAILURE(result);
        }
        array[i]->Close();
    }
}


} // namespace

extern "C" void nnMain();

#ifdef INVALID_POINTER_TEST
TEST(SendAsyncRequestWithUserBuffer, InvalidPointerTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);

    // 前処理
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose serverCloser(serverSession);
    AutoHandleClose clientCloser(clientSession);

#ifdef INVALID_POINTER_TEST
    // TEST 30-8
    // NULL を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            NULL, pMessage, DefaultAlign, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 30-9
    // MemoryPermission_None の領域を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin),
            pMessage, DefaultAlign, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 30-10
    // MemoryPermission_Read の領域を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    uintptr_t tmpAddr = reinterpret_cast<uintptr_t>(&tmpVar);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            reinterpret_cast<nn::svc::Handle*>(tmpAddr),
            pMessage, DefaultAlign, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 30-11
    // MemoryPermission_ReadExecute の領域を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            reinterpret_cast<nn::svc::Handle*>(nnMain),
            pMessage, DefaultAlign, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST
}
#endif

TEST(SendAsyncRequestWithUserBuffer, InvalidAlignmentTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    std::memset(g_UserBuffer0, 0, sizeof(g_UserBuffer0));

    // 前処理
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose serverCloser(serverSession);
    AutoHandleClose clientCloser(clientSession);

    for (uintptr_t i = 1; i < DefaultAlign; i++)
    {
        // TEST 30-12
        // 4KB にアライメントされていないアドレスは受け付けない
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, pMessage + i, DefaultAlign, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());

        // TEST 30-13
        // 4KB にアライメントされていないサイズは受け付けない
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, pMessage, DefaultAlign + i, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }
}

TEST(SendAsyncRequestWithUserBuffer, InvalidMemoryAreaTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;
    uintptr_t heapPtr;

    // 前処理
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose serverCloser(serverSession);
    AutoHandleClose clientCloser(clientSession);

    TestHeap heap(HeapAlign);
    heapPtr = heap.GetAddress();
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(heapPtr);

    // TEST 30-14
    // メモリ領域のパーミッションが一様ではない
    {
        SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
        TestMemoryPermission testPermission(
                heapPtr + DefaultAlign, DefaultAlign, nn::svc::MemoryPermission_Read);

        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, heapPtr, DefaultAlign * 2, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 30-15
    // メモリ領域の状態が一様ではない
    {
        SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
        TestMapMemory testMap(g_FreeAreaBegin, heapPtr + DefaultAlign, DefaultAlign);

        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, heapPtr, DefaultAlign * 2, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

#if defined NN_BUILD_CONFIG_CPU_SVC_32
    // TEST 30-16
    // pMessage + Size が pMessage より小さいと失敗する
    {
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, 0xFFFFF000, 0x2000, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
#endif
#if defined NN_BUILD_CONFIG_CPU_SVC_64 && defined NN_BUILD_CONFIG_ABI_ILP32
    // TEST 30-17
    // pMessage + Size が pMessage より小さいと失敗する
    {
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, 0xFFFFF000, 0x2000, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
#endif
#if defined NN_BUILD_CONFIG_CPU_SVC_64 && defined NN_BUILD_CONFIG_ABI_LP64
    // TEST 30-18
    // pMessage + Size が pMessage より小さいと失敗する
    {
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, 0xFFFFFFFFFFFFF000, 0x2000, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
#endif

    // TEST 30-35
    // MemoryPermission_None の領域を渡すと失敗する
    {
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, g_FreeAreaBegin, DefaultAlign, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 30-36
    // MemoryPermission_Read の領域を渡すと失敗する
    {
        SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
        TestMemoryPermission testPermission(
                heapPtr, DefaultAlign, nn::svc::MemoryPermission_Read);

        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, heapPtr, DefaultAlign, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 30-37
    // MemoryPermission_ReadExecute の領域を渡すと失敗する
    {
        nn::svc::MemoryInfo blockInfo;
        nn::svc::PageInfo pageInfo;
        uintptr_t addr = (reinterpret_cast<uintptr_t>(nnMain) + 0xfff) & ~(0xfff);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);

        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadExecute);
        ASSERT_TRUE(blockInfo.size >= DefaultAlign);

        result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, addr, DefaultAlign, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
}

TEST(SendAsyncRequestWithUserBuffer, MemoryStateTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;

    // 前処理
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose serverCloser(serverSession);
    AutoHandleClose clientCloser(clientSession);

    // Thread
    uintptr_t pc = reinterpret_cast<uintptr_t>(RepeatReplyThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    g_ThreadLoop = true;
    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_CodeData,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState, clientSession);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid, clientSession);

    uint32_t successAttribute[] = {
        0,
    };
    const int NumSuccessAttr = sizeof(successAttribute) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, successAttribute, NumSuccessAttr);
    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr, clientSession);

    const int NumInvalidAttrs = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrs[NumInvalidAttrs];
    header->GetTestListExceptAttributes(invalidAttrs, NumInvalidAttrs, successAttribute, NumSuccessAttr);
    CheckInvalidMemoryArea(invalidAttrs, NumInvalidAttrs, clientSession);

    g_ThreadLoop = false;
    result = nn::svc::CancelSynchronization(thread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);
    thread.Wait();
}

TEST(SendAsyncRequestWithUserBuffer, InvalidHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle eventHandle;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    // TEST 30-19
    // INVALID_HANDLE_VALUE を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 30-20
    // PSEUDO_HANDLE_CURRENT_THREAD を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 30-21
    // PSEUDO_HANDLE_CURRENT_PROCESS を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(SendAsyncRequestWithUserBuffer, InvalidLightIPCHandleTest)
{
    TestLightSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    // TEST 30-22
    // Light系のセッションを渡すと失敗する
    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, true, 0);
    ASSERT_RESULT_SUCCESS(result);

    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(SendAsyncRequestWithUserBuffer, InvalidNamedPortHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;
    const char* PortName = "Test3023";
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    int32_t index;

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 2);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);

    // TEST 30-23
    // NamedPort のサーバーポートを渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    // TEST 30-24
    // NamedPort のサーバーセッショを渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 30-25
    // 閉じられたNamedPort のクライアントセッションを渡すと失敗する
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendAsyncRequestWithUserBuffer, InvalidPortHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    int32_t index;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    // TEST 30-26
    // Port のサーバーポートを渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    // TEST 30-27
    // Port のサーバーセッションを渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 30-28
    // Port のクライアントポートを渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 30-29
    // 閉じられた Port のクライアントセッションを渡すと失敗する
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(SendAsyncRequestWithUserBuffer, InvalidSessionHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    // TEST 30-30
    // サーバーセッションが渡されると失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 30-31
    // 閉じられた クライアントセッションを渡されると失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(SendAsyncRequestWithUserBuffer, ReceiveManualSignalWithNamedPortTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    uintptr_t pc = reinterpret_cast<uintptr_t>(ReplyThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param;
    const char* portName = "Test30169";
    int32_t index;

    result = nn::svc::ManageNamedPort(&serverPort, portName, 2);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);

    result = nn::svc::ConnectToNamedPort(&clientSession, portName);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    param = reinterpret_cast<uintptr_t>(&serverSession);
    TestThread thread(pc, param, sp, TestLowestThreadPriority, 0);

    thread.Start();

    // TEST 30-169
    // クリアするまで、何度もシグナルを受信できる
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, clientSession);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose eventCloser(eventHandle);

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ClearEvent(eventHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    thread.Wait();

    result = nn::svc::ManageNamedPort(&serverPort, portName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendAsyncRequestWithUserBuffer, ReceiveManualSignalWithPortTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    uintptr_t pc = reinterpret_cast<uintptr_t>(ReplyThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param;
    int32_t index;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    param = reinterpret_cast<uintptr_t>(&serverSession);
    TestThread thread(pc, param, sp, TestLowestThreadPriority, 0);

    thread.Start();

    // TEST 30-171
    // クリアするまで、何度もシグナルを受信できる
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, clientSession);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose eventCloser(eventHandle);

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ClearEvent(eventHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    thread.Wait();
}
TEST(SendAsyncRequestWithUserBuffer, ReceiveManualSignalWithSessionTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    nn::svc::Handle eventHandle;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);
    uintptr_t pc = reinterpret_cast<uintptr_t>(ReplyThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param;
    int32_t index;

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    param = reinterpret_cast<uintptr_t>(&serverSession);
    TestThread thread(pc, param, sp, TestLowestThreadPriority, 0);

    thread.Start();

    // TEST 30-171
    // クリアするまで、何度もシグナルを受信できる
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, pMessage, size, clientSession);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose eventCloser(eventHandle);

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Reply);

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ClearEvent(eventHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    thread.Wait();
}

