﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Compiler.h>
#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"

#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <cstring>

#if defined NN_BUILD_CONFIG_COMPILER_CLANG
#pragma clang diagnostic ignored "-Wtautological-compare"
#endif

extern "C" void nnMain();

namespace {

char g_Buffer[0x1000] __attribute__((aligned(0x1000)));

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uint64_t addr;

    for (int i = 0; i < numState; i++)
    {
        array[i]->Initialize();
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }
        addr = array[i]->GetAddress();

        result = nn::svc::QueryProcessMemory(
                &blockInfo, &pageInfo, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, addr);
        ASSERT_RESULT_SUCCESS(result);

        ASSERT_TRUE(blockInfo.state == array[i]->GetState());
        ASSERT_TRUE(blockInfo.permission == array[i]->GetPermission());
        ASSERT_TRUE(blockInfo.attribute == array[i]->GetAttribute());

        array[i]->Close();
    }
}

} // namespace

TEST(QueryProcessMemory, ReadOwnMemory)
{
    nn::Result result;

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uint64_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumTestMemoryState];
    header->GetTestListWithStates(successArea, AllMemoryState, NumTestMemoryState);

    CheckSuccessMemoryArea(successArea, NumTestMemoryState);

    TestMemoryInfo* successAttrArea[NumTestMemoryAttribute];
    const int NumSuccessAttr = NumTestMemoryAttribute;
    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uint64_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, AllMemoryAttribute, NumSuccessAttr);

    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr);
} // NOLINT (readability/fn_size)

TEST(QueryProcessMemory, ReadOtherMemory)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uint64_t addr;
    uint64_t size;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    for (int i = 0; i < NumTestMemoryState; i++)
    {
        nn::svc::MemoryState state = AllMemoryState[i];
        stateProcess.GetMemoryArea(&addr, &size, state);
        if (size == 0)
        {
            stateProcess.FreeMemoryArea();
            continue;
        }
        result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.state == state);
        ASSERT_TRUE(blockInfo.attribute == 0);

        stateProcess.FreeMemoryArea();
    }

    {
        stateProcess.GetMemoryArea(&addr, &size, nn::svc::MemoryState_Normal);
        stateProcess.SetUncached(addr, size);

        result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Uncached);

        stateProcess.UnsetUncached(addr, size);
    }

    // TEST 118-55
    // 全アドレス空間を読むことが出来る
    addr = 0;
    uint64_t endAddress = stateProcess.Is64Bit() ? AddressSpaceSize64 : AddressSpaceSize32;
    ASSERT_TRUE(endAddress != 0);
    do
    {
        result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
        ASSERT_RESULT_SUCCESS(result);

        addr += 0x1000;
    } while (addr != endAddress);

    // 対応しているアドレス空間外のアドレスも与えることが出来る
    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Inaccessible);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.attribute == 0);
} // NOLINT (readability/fn_size)

