﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <cstring>

namespace {

} // namespace


// TEST 116-54
// マップされた領域の命令を実行することが出来る
TEST(MapProcessMemory, ExecuteCodeTest)
{
    nn::Result result;

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum);
    nn::svc::Handle processHandle = process.GetHandle();

    // ただ、ExitProcess を呼ぶプロセスをプロセスを作成する
    AssignExitCode(processHandle, process.GetCodeAddress(), process.GetCodeAreaSize());

    // プロセスを実行する
    result = nn::svc::StartProcess(
            processHandle, TestLowestThreadPriority, g_ProcessIdealCore, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    // 終了確認をすることで、実行されたことを確認する
    WaitProcess(processHandle);
}

TEST(MapProcessMemory, MergeBlockTest)
{
    nn::Result result;

    TestProcess process(2);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr = process.GetCodeAddress();

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(toAddr + size, processHandle, fromAddr + size, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 116-60
    // 連続した領域をマップすると、ブロックがマージされる
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, fromAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == fromAddr);
    ASSERT_TRUE(blockInfo.size == process.GetCodeAreaSize());
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Code);

    result = nn::svc::QueryProcessMemory(
            &blockInfo, &pageInfo, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, toAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == toAddr);
    ASSERT_TRUE(blockInfo.size == size * 2);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedCode);

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size * 2);
    ASSERT_RESULT_SUCCESS(result);
}

