﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_MemoryState.h"
#include "util_TestProcess.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <cstring>

extern "C" void nnMain();

namespace {

const size_t BufferSize = 0x1000;
char g_Buffer[BufferSize] __attribute__((aligned(0x1000)));

void CheckSuccessToMemoryArea(TestMemoryInfo** array, int numState, TestProcess* process)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        ASSERT_TRUE(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0));

        array[i]->Initialize();
        uintptr_t fromAddr = process->GetCodeAddress();
        size_t size = BufferSize;
        uintptr_t toAddr = array[i]->GetAddress();
        nn::svc::Handle processHandle = process->GetHandle();

        CheckProcessMemory(
                processHandle, fromAddr, nn::svc::MemoryState_Code, nn::svc::MemoryPermission_None, 0);

        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                toAddr, nn::svc::MemoryState_SharedCode, nn::svc::MemoryPermission_ReadWrite, 0);

        result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();
        array[i]->Close();
    }
}

void CheckInvalidToMemoryArea(TestMemoryInfo** array, int numState, TestProcess* process)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize();
        uintptr_t fromAddr = process->GetCodeAddress();
        size_t size = BufferSize;
        uintptr_t toAddr =  array[i]->GetAddress();
        nn::svc::Handle processHandle = process->GetHandle();

        CheckProcessMemory(
                processHandle, fromAddr, nn::svc::MemoryState_Code, nn::svc::MemoryPermission_None, 0);

        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
        if (!IsInAslrRegion(toAddr))
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }

        CheckProcessMemory(
                processHandle, fromAddr, nn::svc::MemoryState_Code, nn::svc::MemoryPermission_None, 0);
        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}

} // namespace

TEST(MapProcessMemory, AddrTest)
{
    nn::Result result;

    TestProcess process(1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();

    // TEST 116-1
    // toAddr のアラインメントが4KBに揃っていると成功する
    // TEST 116-10
    // fromAddr のアラインメントが4KBに揃っていると成功する
    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 116-2
    // toAddr のアラインメントが4KBに揃っていないと失敗する
    for (uintptr_t i = toAddr + 1; i < toAddr + 0x1000; i++)
    {
        result = nn::svc::MapProcessMemory(i, processHandle, fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    // TEST 116-11
    // fromAddr のアラインメントが4KBに揃っていないと失敗する
    for (uintptr_t i = fromAddr + 1; i < fromAddr + 0x1000; i++)
    {
        result = nn::svc::MapProcessMemory(toAddr, processHandle, i, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }
}

TEST(MapProcessMemory, HandleTest)
{
    nn::Result result;

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    // TEST 116-3
    // 実行前のプロセスのハンドルを受け付ける
    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartProcess(
            processHandle, TestLowestThreadPriority, g_ProcessIdealCore, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 116-4
    // 実行中のプロセスのハンドルを受け付ける
    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    WaitProcess(processHandle);

    // TEST 116-5
    // 実行後のプロセスのハンドルを受け付ける
    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 116-6
    // プロセスの擬似ハンドルを受け付けない
    result = nn::svc::MapProcessMemory(
            toAddr, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 116-7
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::MapProcessMemory(
            toAddr, nn::svc::INVALID_HANDLE_VALUE, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 116-8
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::MapProcessMemory(
            toAddr, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 116-9
    // Close したプロセスのハンドルを受け付けない
    result = nn::svc::MapProcessMemory(
            toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(MapProcessMemory, SizeTest)
{
    nn::Result result;

    TestProcess process(1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();

    // TEST 116-12
    // サイズのアラインメントが4KBに揃っていると成功する
    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 116-13
    // サイズのアラインメントが4KBに揃っていないと失敗する
    for (size_t i = size + 1; i < size + 0x1000; i++)
    {
        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, i);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    // TEST 116-14
    // サイズは 0 を受け付けない
    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
}

TEST(MapProcessMemory, ToAddrAreaTest)
{
    nn::Result result;

    TestProcess process(1);
    nn::svc::Handle processHandle = process.GetHandle();

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    nn::svc::MemoryState successStates[] = {
        nn::svc::MemoryState_Free,
    };

    // 許可されたメモリ状態の領域を受け付ける
    const int NumSuccess = sizeof(successStates) / sizeof(nn::svc::MemoryState);
    TestMemoryInfo* successArea[NumSuccess];
    header->GetTestListWithStates(successArea, successStates, NumSuccess);

    CheckSuccessToMemoryArea(successArea, NumSuccess, &process);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccess;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successStates, NumSuccess);

    CheckInvalidToMemoryArea(invalidArea, NumInvalid, &process);

    TestHeap heap(HeapAlign);
    size_t size = 0x1000;
    uintptr_t fromAddr = process.GetCodeAddress();
    uintptr_t toAddr;

    // TEST 116-30
    // toAddr/size で示す領域 のメモリ状態が一様でないと失敗する (前方)
    {
        toAddr = g_FreeAreaBegin;
        TestMapMemory map(toAddr, heap.GetAddress(), size);

        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 116-31
    // toAddr/size で示す領域 のメモリ状態が一様でないと失敗する (後方)
    {
        toAddr = g_FreeAreaBegin;
        TestMapMemory map(toAddr + size, heap.GetAddress(), size);

        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 116-32
    // オーバーフローする組み合わせを指定できない
    toAddr = g_FreeAreaBegin + 0x1000;
    size = static_cast<size_t>(-0x1000);
    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    size = 0x1000;
    // TEST 116-51
    // toAddr/size で示す領域 のメモリ状態がAslr に含まれていないと失敗する
    // Reserved 領域
    {
        uintptr_t reservedAddr;
        size_t reservedSize;
        ASSERT_TRUE(GetReservedFreeMemoryArea(&reservedAddr, &reservedSize));
        ASSERT_TRUE(reservedSize >= size);

        result = nn::svc::MapProcessMemory(reservedAddr, processHandle, fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
    }

    // Large 領域
    {
        uint64_t largeAddr;
        bool check = GetLargeFreeArea(&largeAddr, size);
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
        ASSERT_TRUE(!check);
#else
        ASSERT_TRUE(check);
#endif
        if(check)
        {
            toAddr = static_cast<uintptr_t>(largeAddr);
            result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}

TEST(MapProcessMemory, FromAddrAreaTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    size_t mapSize = 0x1000;
    uint64_t size;
    uintptr_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    nn::svc::MemoryState state;

    CheckMemory(
            toAddr,
            nn::svc::MemoryState_Free,
            nn::svc::MemoryPermission_None,
            0);

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Code,
        nn::svc::MemoryState_CodeData,
        nn::svc::MemoryState_AliasCode,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    // 許可されたメモリ状態
    for (int i = 0; i < NumSuccessState; i++)
    {
        SCOPED_TRACE("Success Memory Area");
        state = successState[i];
        stateProcess.GetMemoryArea(&fromAddr, &size, state);
        ASSERT_TRUE(size >= mapSize);

        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, mapSize);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                toAddr,
                nn::svc::MemoryState_SharedCode,
                nn::svc::MemoryPermission_ReadWrite,
                0);

        result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, mapSize);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                toAddr,
                nn::svc::MemoryState_Free,
                nn::svc::MemoryPermission_None,
                0);

        stateProcess.FreeMemoryArea();
    }

    // 許可されていないメモリ状態
    for (int i = 0; i < NumTestMemoryState; i++)
    {
        SCOPED_TRACE("Invalid Memory Area");
        nn::svc::MemoryState state = AllMemoryState[i];
        bool isSuccess = false;
        for (int j = 0; j < NumSuccessState; j++)
        {
            if (state == successState[j])
            {
                isSuccess = true;
                break;
            }
        }
        if (isSuccess)
        {
            continue;
        }

        stateProcess.GetMemoryArea(&fromAddr, &size, state);
        if (size < mapSize)
        {
            continue;
        }

        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, mapSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        CheckMemory(
                toAddr,
                nn::svc::MemoryState_Free,
                nn::svc::MemoryPermission_None,
                0);

        stateProcess.FreeMemoryArea();
    }

    {
        uint64_t codeAddr;

        stateProcess.GetMemoryArea(&codeAddr, &size, nn::svc::MemoryState_Code);
        ASSERT_TRUE(size > mapSize * 2);

        // TEST 116-47
        // fromAddr/size で示す領域 のメモリ状態が一様でないと失敗する (前方)
        fromAddr = codeAddr - mapSize;
        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, mapSize * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        // TEST 116-48
        // fromAddr/size で示す領域 のメモリ状態が一様でないと失敗する (後方)
        fromAddr = codeAddr + size - mapSize;
        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, mapSize * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        // TEST 116-49
        // オーバーフローする組み合わせを指定できない
        fromAddr = codeAddr;
        size = static_cast<uint64_t>(-0x1000);
        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        stateProcess.FreeMemoryArea();
    }

    // メモリ属性が付加されていると失敗する
    {
        uint64_t codeDataAddr;
        stateProcess.GetMemoryArea(&codeDataAddr, &size, nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(size >= mapSize);
        stateProcess.SetUncached(codeDataAddr, mapSize);

        fromAddr = codeDataAddr;

        CheckProcessMemory(
                processHandle,
                fromAddr,
                nn::svc::MemoryState_CodeData,
                nn::svc::MemoryPermission_ReadWrite,
                nn::svc::MemoryAttribute_Uncached);

        result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, mapSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        stateProcess.UnsetUncached(codeDataAddr, mapSize);
    }
} // NOLINT (readability/fn_size)
