﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"

#include <nn/svc/svc_Dd.h>

namespace {

const size_t MapSize = 0x1000;

char g_Buffer[MapSize * 3] __attribute__((aligned(MapSize)));
char g_TestBuffer[0x1000] __attribute__((aligned(MapSize)));

void CheckSuccessFromMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;
    uintptr_t toAddr = g_FreeAreaEnd - MapSize;

    CheckMemory(
            toAddr,
            nn::svc::MemoryState_Free,
            nn::svc::MemoryPermission_None,
            0);

    for (int i = 0; i < numState; i++)
    {
        ASSERT_TRUE(array[i]->GetSize() >= MapSize&& ((array[i]->GetSize() & 0xfff) == 0));
        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::MapMemory(toAddr, addr, MapSize);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr,
                array[i]->GetState(),
                nn::svc::MemoryPermission_None,
                nn::svc::MemoryAttribute_Locked);

        CheckMemory(
                toAddr,
                nn::svc::MemoryState_Stack,
                nn::svc::MemoryPermission_ReadWrite,
                0);

        result = nn::svc::UnmapMemory(toAddr, addr, MapSize);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();

        CheckMemory(
                toAddr,
                nn::svc::MemoryState_Free,
                nn::svc::MemoryPermission_None,
                0);

        array[i]->Close();
    }
}

void CheckInvalidFromMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;
    uintptr_t toAddr = g_FreeAreaEnd - MapSize;

    CheckMemory(
            toAddr,
            nn::svc::MemoryState_Free,
            nn::svc::MemoryPermission_None,
            0);

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() >= MapSize && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::MapMemory(toAddr, addr, MapSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        array[i]->CheckDefaultState();

        CheckMemory(
                toAddr,
                nn::svc::MemoryState_Free,
                nn::svc::MemoryPermission_None,
                0);

        array[i]->Close();
    }
}

void CheckSuccessToMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;
    uintptr_t fromAddr = reinterpret_cast<uintptr_t>(g_Buffer);

    CheckMemory(
            fromAddr,
            nn::svc::MemoryState_CodeData,
            nn::svc::MemoryPermission_ReadWrite,
            0);

    for (int i = 0; i < numState; i++)
    {
        ASSERT_TRUE(array[i]->GetSize() >= MapSize&& ((array[i]->GetSize() & 0xfff) == 0));
        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::MapMemory(addr, fromAddr, MapSize);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr,
                nn::svc::MemoryState_Stack,
                nn::svc::MemoryPermission_ReadWrite,
                array[i]->GetAttribute());

        CheckMemory(
                fromAddr,
                nn::svc::MemoryState_CodeData,
                nn::svc::MemoryPermission_None,
                nn::svc::MemoryAttribute_Locked);

        result = nn::svc::UnmapMemory(addr, fromAddr, MapSize);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();

        CheckMemory(
                fromAddr,
                nn::svc::MemoryState_CodeData,
                nn::svc::MemoryPermission_ReadWrite,
                0);

        array[i]->Close();
    }
}

void CheckInvalidToMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;
    uintptr_t fromAddr = reinterpret_cast<uintptr_t>(g_Buffer);

    CheckMemory(
            fromAddr,
            nn::svc::MemoryState_CodeData,
            nn::svc::MemoryPermission_ReadWrite,
            0);

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() >= MapSize&& ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }
        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::MapMemory(addr, fromAddr, MapSize);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

        array[i]->CheckDefaultState();

        CheckMemory(
                fromAddr,
                nn::svc::MemoryState_CodeData,
                nn::svc::MemoryPermission_ReadWrite,
                0);

        array[i]->Close();
    }
}
}

extern "C" void nnMain();

TEST(MapMemoryTest, FromMemoryStateTest)
{
    nn::Result result;
    uintptr_t fromAddr;
    uintptr_t toAddr = g_FreeAreaBegin;

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_CodeData,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif // SUPPORT_ALIAS_CODE_DATA
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessFromMemoryArea(successArea, NumSuccessState);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidFromMemoryArea(invalidArea, NumInvalid);

    uint32_t successAttribute[] = {
        0,
    };
    const int NumSuccessAttr = sizeof(successAttribute) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, successAttribute, NumSuccessAttr);
    CheckSuccessFromMemoryArea(successAttrArea, NumSuccessAttr);

    const int NumInvalidAttrs = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrs[NumInvalidAttrs];
    header->GetTestListExceptAttributes(invalidAttrs, NumInvalidAttrs, successAttribute, NumSuccessAttr);
    CheckInvalidFromMemoryArea(invalidAttrs, NumInvalidAttrs);


    {
        TestNormalMemoryState normal;
        normal.Initialize();

        // メモリ状態が一様でないと失敗する
        // 前方
        fromAddr = normal.GetAddress() - MapSize;
        result = nn::svc::MapMemory(toAddr, fromAddr, MapSize * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        normal.CheckDefaultState();

        // 後方
        fromAddr = normal.GetAddress() + normal.GetSize() - MapSize;
        result = nn::svc::MapMemory(toAddr, fromAddr, MapSize * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        normal.CheckDefaultState();

        normal.Close();
    }
} // NOLINT (readability/fn_size)

TEST(MapMemoryTest, ToMemoryStateTest)
{
    nn::Result result;
    uintptr_t fromAddr = reinterpret_cast<uintptr_t>(g_Buffer);
    uintptr_t toAddr;

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Free,
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessToMemoryArea(successArea, NumSuccessState);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidToMemoryArea(invalidArea, NumInvalid);

    {
        // メモリ状態が一様でないと失敗する
        TestMapMemory map(g_FreeAreaBegin + MapSize, fromAddr, MapSize);

        // 前方
        toAddr = g_FreeAreaBegin;
        result = nn::svc::MapMemory(toAddr, fromAddr + MapSize, MapSize * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        // 後方
        toAddr = g_FreeAreaBegin + MapSize;
        result = nn::svc::MapMemory(toAddr, fromAddr + MapSize, MapSize * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // Reserved 領域だと成功する
    {
        toAddr = g_ReservedAreaBegin;
        result = nn::svc::MapMemory(toAddr, fromAddr, MapSize);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                toAddr,
                nn::svc::MemoryState_Alias,
                nn::svc::MemoryPermission_ReadWrite,
                0);
        CheckMemory(
                fromAddr,
                nn::svc::MemoryState_CodeData,
                nn::svc::MemoryPermission_None,
                nn::svc::MemoryAttribute_Locked);

        result = nn::svc::UnmapMemory(toAddr, fromAddr, MapSize);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                toAddr,
                nn::svc::MemoryState_Free,
                nn::svc::MemoryPermission_None,
                0);
        CheckMemory(
                fromAddr,
                nn::svc::MemoryState_CodeData,
                nn::svc::MemoryPermission_ReadWrite,
                0);
    }

    // Large 領域だと失敗する
    {
        uint64_t largeAreaAddr = 0;
        bool check = GetLargeFreeArea(&largeAreaAddr, MapSize);
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
        ASSERT_TRUE(!check);
#else
        ASSERT_TRUE(check);
#endif
        if (check)
        {
            toAddr = static_cast<uintptr_t>(largeAreaAddr);
            CheckMemory(
                    toAddr,
                    nn::svc::MemoryState_Free,
                    nn::svc::MemoryPermission_None,
                    0);
            CheckMemory(
                    fromAddr,
                    nn::svc::MemoryState_CodeData,
                    nn::svc::MemoryPermission_ReadWrite,
                    0);

            result = nn::svc::MapMemory(toAddr, fromAddr, MapSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

            CheckMemory(
                    toAddr,
                    nn::svc::MemoryState_Free,
                    nn::svc::MemoryPermission_None,
                    0);
            CheckMemory(
                    fromAddr,
                    nn::svc::MemoryState_CodeData,
                    nn::svc::MemoryPermission_ReadWrite,
                    0);
        }
    }
} // NOLINT (readability/fn_size)

TEST(MapMemoryTest, InvalidCurrentMemoryPermissionTest)
{
    nn::Result result;
    uintptr_t bssAddr;
    size_t size;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    bssAddr = reinterpret_cast<uintptr_t>(g_Buffer);
    size = sizeof(g_Buffer);

    // BSS領域が CodeData State であることを確認する
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, bssAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // FreeArea領域が 空いていて、Free State であることを確認する
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(blockInfo.attribute == 0);
    ASSERT_TRUE(g_FreeAreaEnd - 1 <= blockInfo.baseAddress + blockInfo.size - 1);

    // TEST 3-39
    // MemoryPermission_None の領域を受け付けない
    {
        TestMemoryPermission perm(bssAddr, size, nn::svc::MemoryPermission_None);
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, bssAddr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::MapMemory(g_FreeAreaBegin, bssAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 3-40
    // MemoryPermission_Read の領域を受け付けない
    {
        TestMemoryPermission perm(bssAddr, size, nn::svc::MemoryPermission_Read);
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, bssAddr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::MapMemory(g_FreeAreaBegin, bssAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 3-41
    // MemoryPermission_ReadWrite の領域を受け付ける
    {
        TestMemoryPermission perm(bssAddr, size, nn::svc::MemoryPermission_ReadWrite);
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, bssAddr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

        result = nn::svc::MapMemory(g_FreeAreaBegin, bssAddr, size);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, bssAddr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Stack);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::UnmapMemory(g_FreeAreaBegin, bssAddr, size);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(MapMemoryTest, OverFlowTest)
{
    nn::Result result;
    uintptr_t addr = static_cast<uintptr_t>(-1) & ~(0xFFF);

    // TEST 3-31
    // fromAddr, size の組み合わせで、オーバーフローする組み合わせは許容しない
    result = nn::svc::MapMemory(g_FreeAreaBegin, addr, 0x2000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 3-32
    // toAddr, size の組み合わせで、オーバーフローする組み合わせは許容しない
    result = nn::svc::MapMemory(addr, reinterpret_cast<uintptr_t>(g_Buffer), 0x2000);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());
}

