﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"

extern "C" void nnMain();

TEST(MapMemoryTest, HeapTest)
{
    uintptr_t addr;
    size_t size;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t freeAreaBase;
    size_t freeAreaSize;

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // Heap Area Check
    ASSERT_TRUE(g_HeapAreaBegin <= addr);
    ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);

    // テストに使うマッピング領域が空きかどうかチェック
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(g_FreeAreaEnd - 1 <= blockInfo.baseAddress + blockInfo.size - 1);
    freeAreaBase = blockInfo.baseAddress;
    freeAreaSize = blockInfo.size;

    // TEST 3-1
    // ヒープ領域をマップする
    result = nn::svc::MapMemory(g_FreeAreaBegin, addr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.size == 0x1000);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress <= g_FreeAreaBegin);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size - 1 >= g_FreeAreaBegin + 0x1000 - 1);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Stack);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 4-1
    // マップされていた領域を解除する
    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.size == HeapAlign);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.attribute == 0);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress <= g_FreeAreaBegin);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size - 1 >= g_FreeAreaBegin + 0x1000 - 1);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(blockInfo.attribute == 0);

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(MapMemoryTest, CodeDataTest)
{
    static char g_Buffer[0x2000] __attribute__((aligned(0x1000)));

    uintptr_t addr;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t freeAreaBase;
    size_t freeAreaSize;

    addr = reinterpret_cast<uintptr_t>(g_Buffer);

    // テストに使うマッピング領域が空きかどうかチェック
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(g_FreeAreaEnd - 1 <= blockInfo.baseAddress + blockInfo.size - 1);
    freeAreaBase = blockInfo.baseAddress;
    freeAreaSize = blockInfo.size;

    // TEST 3-1
    // CodeData 領域をマップする
    result = nn::svc::MapMemory(g_FreeAreaBegin, addr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.size == 0x1000);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress <= g_FreeAreaBegin);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size - 1 >= g_FreeAreaBegin + 0x1000 - 1);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Stack);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 4-1
    // マップされていた領域を解除する
    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.attribute == 0);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress <= g_FreeAreaBegin);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size - 1 >= g_FreeAreaBegin + 0x1000 - 1);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(blockInfo.attribute == 0);
}

TEST(MapMemoryTest, FailureTest)
{
    uintptr_t addr;
    size_t size;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // Heap Area Check
    ASSERT_TRUE(g_HeapAreaBegin <= addr);
    ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);

    // テストに使うマッピング領域が空きかどうかチェック
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(g_FreeAreaEnd - 1 <= blockInfo.baseAddress + blockInfo.size - 1);

    // TEST 3-2
    // 確保したヒープ領域外のメモリ領域をマップする
    // 領域外：確保したヒープ領域の先頭より前の領域
    result = nn::svc::MapMemory(g_FreeAreaBegin, addr - 0x1000, 0x1000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 3-3
    // 確保したヒープ領域外のメモリ領域を包含した領域をマップする
    // 領域外：確保したヒープ領域の先頭より前の領域
    result = nn::svc::MapMemory(g_FreeAreaBegin, addr - 0x1000, 0x2000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 3-2
    // 確保したヒープ領域外のメモリ領域をマップする
    // 領域外：確保したヒープ領域の終端より後ろの領域
    result = nn::svc::MapMemory(g_FreeAreaBegin, addr + HeapAlign, 0x1000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 3-3
    // 確保したヒープ領域外のメモリ領域を包含した領域をマップする
    // 領域外：確保したヒープ領域の終端より後ろの領域
    result = nn::svc::MapMemory(g_FreeAreaBegin, addr + HeapAlign - 0x1000, 0x2000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 3-4
    // fromAddr にFree State の領域を渡す。
    result = nn::svc::MapMemory(g_FreeAreaBegin + 0x1000, g_FreeAreaBegin, 0x1000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 3-5
    // toAddr と fromAddr が一致している
    result = nn::svc::MapMemory(g_FreeAreaBegin, g_FreeAreaBegin, 0x1000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 3-6
    // コード領域をマップしようとする
    result = nn::svc::MapMemory(g_FreeAreaBegin, reinterpret_cast<uintptr_t>(&nnMain) & ~(0x1000 - 1), 0x1000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    for (int i = 1; i < 0x1000; i++)
    {
        // TEST 3-7
        // fromAddr が 0x1000 にアライメントされていない
        result = nn::svc::MapMemory(g_FreeAreaBegin, addr + i, 0x1000);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());

        // TEST 3-8
        // toAddr が 0x1000 にアライメントされていない
        result = nn::svc::MapMemory(g_FreeAreaBegin + i, addr, 0x1000);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());

        // TEST 3-9
        // サイズが 0x1000 にアライメントされていない
        result = nn::svc::MapMemory(g_FreeAreaBegin, addr, 0x1000 + i);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

        // TEST 3-10
        // fromAddr と toAddr がアライメントされていない
        result = nn::svc::MapMemory(g_FreeAreaBegin + i, addr + i, 0x1000);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }
    for (size_t i = HeapAlign + 0x1000; i != CheckVirtualEnd; i += 0x1000)
    {
        // TEST 3-11
        // 確保しているヒープ領域のサイズ以上の大きさをマップしようとする
        result = nn::svc::MapMemory(g_FreeAreaBegin, addr, i);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());
    }
    // TEST 3-11
    // 境界値
    {
        size_t size = static_cast<size_t>(-0x1000);
        result = nn::svc::MapMemory(g_FreeAreaBegin + 0x1000, addr + 0x1000, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    result = nn::svc::MapMemory(g_FreeAreaBegin, addr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapMemory(g_FreeAreaBegin + 0x1000, addr + 0x1000, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 4-2
    // fromAddr とtoAddr にマップブロックが違うアドレスを入れる
    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr + 0x1000, 0x1000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

    result = nn::svc::UnmapMemory(g_FreeAreaBegin + 0x1000, addr, 0x1000);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

    // TEST 4-3
    // toAddr にマップしていない領域を指定する
    result = nn::svc::UnmapMemory(g_FreeAreaBegin - 0x1000, addr, 0x1000);
    ASSERT_TRUE(
            result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion()
            );

    // TEST 1-15
    // Aliasがある状態では解放できない
    uintptr_t tmp;
    result = nn::svc::SetHeapSize(&tmp, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    result = nn::svc::UnmapMemory(g_FreeAreaBegin + 0x1000, addr + 0x1000, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

