﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"

namespace {
const size_t OtherAppSize = HeapAlign * 16 * 2;
const size_t BaseAlign = 0x1000;
const size_t Test64kSize = 0x10000;
const size_t Test2mSize = 0x200000;
const size_t Test4mSize = 0x400000;
const size_t Test32mSize = 0x2000000;

const size_t TestSharedSizes[] = { Test64kSize, Test2mSize };
const size_t TestHeapSizes[] = { Test64kSize, Test2mSize, Test4mSize, Test32mSize };

const char BeginChar = 'b';
const char MiddleChar = 'm';
const char EndChar = 'e';

void CheckWrite(uintptr_t addr, size_t size)
{
    char* ptr = reinterpret_cast<char*>(addr);
    *ptr = BeginChar;

    ptr += size / 2;
    *ptr = MiddleChar;

    ptr = reinterpret_cast<char*>(addr + size - 1);
    *ptr = EndChar;
}

void CheckRead(uintptr_t addr, size_t size)
{
    char* ptr = reinterpret_cast<char*>(addr);
    ASSERT_TRUE(*ptr == BeginChar);

    ptr += size / 2;
    ASSERT_TRUE(*ptr == MiddleChar);

    ptr = reinterpret_cast<char*>(addr + size - 1);
    ASSERT_TRUE(*ptr == EndChar);
}

void TestHeapArea(uintptr_t addr, size_t size, uintptr_t heapAddr, size_t heapSize)
{
    nn::Result result;
    nn::svc::MemoryInfo info;

    CheckWrite(addr, size);
    result = nn::svc::SetMemoryPermission(
            addr, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&info, addr);
    ASSERT_TRUE(info.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_Read);
    ASSERT_TRUE(info.attribute == 0);
    ASSERT_TRUE(info.baseAddress == addr);
    ASSERT_TRUE(info.size == size);
    CheckRead(addr, size);

    result = nn::svc::SetMemoryPermission(
            addr, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&info, heapAddr);
    ASSERT_TRUE(info.baseAddress == heapAddr);
    ASSERT_TRUE(info.size == heapSize);
    ASSERT_TRUE(info.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(info.attribute == 0);

    CheckWrite(heapAddr, heapSize);
    CheckRead(heapAddr, heapSize);
}

} // namespace

TEST(LargePageTable, SharedMemory)
{
    ConsumeHeapMemory maxHeap;
    size_t heapSize = maxHeap.GetSize();
    heapSize -= OtherAppSize;
    TestHeap heap(heapSize);

    nn::Result result;
    nn::svc::MemoryInfo info;
    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_ReadWrite;

    for (int i = 0; i < static_cast<int32_t>(sizeof(TestSharedSizes) / sizeof(size_t)); ++i)
    {
        nn::svc::Handle sharedHandle;
        uintptr_t addr;
        size_t size;

        // TEST 902-1
        // 共有メモリをアラインに沿ってマップすることが出来る
        {
            addr = g_FreeAreaBegin;
            size = TestSharedSizes[i];

            result = nn::svc::CreateSharedMemory(&sharedHandle, size, perm, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(sharedHandle);

            result = nn::svc::MapSharedMemory(sharedHandle, addr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.baseAddress == addr);
            ASSERT_TRUE(info.size == size);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == perm);

            CheckWrite(addr, size);
            CheckRead(addr, size);

            result = nn::svc::UnmapSharedMemory(sharedHandle, addr, size);
            ASSERT_RESULT_SUCCESS(result);
        }

        // TEST 902-2
        // 共有メモリを両端がアラインに沿っていない状態でマップすることが出来る
        {
            addr = g_FreeAreaBegin + BaseAlign;
            size = TestSharedSizes[i];

            result = nn::svc::CreateSharedMemory(&sharedHandle, size, perm, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(sharedHandle);

            result = nn::svc::MapSharedMemory(sharedHandle, addr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.baseAddress == addr);
            ASSERT_TRUE(info.size == size);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == perm);

            CheckWrite(addr, size);
            CheckRead(addr, size);

            result = nn::svc::UnmapSharedMemory(sharedHandle, addr, size);
            ASSERT_RESULT_SUCCESS(result);
        }

        // TEST 902-3
        // 共有メモリの終端がアライメントに沿っていない状態でマップすることが出来る
        {
            addr = g_FreeAreaBegin;
            size = TestSharedSizes[i] + BaseAlign;

            result = nn::svc::CreateSharedMemory(&sharedHandle, size, perm, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(sharedHandle);

            result = nn::svc::MapSharedMemory(sharedHandle, addr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.baseAddress == addr);
            ASSERT_TRUE(info.size == size);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == perm);

            CheckWrite(addr, size);
            CheckRead(addr, size);

            result = nn::svc::UnmapSharedMemory(sharedHandle, addr, size);
            ASSERT_RESULT_SUCCESS(result);
        }

        // TEST 902-4
        // 共有メモリを始端がアライメントに沿っていない状態でマップすることが出来る
        {
            addr = g_FreeAreaBegin + BaseAlign;
            size = 2 * TestSharedSizes[i] - BaseAlign;

            result = nn::svc::CreateSharedMemory(&sharedHandle, size, perm, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(sharedHandle);

            result = nn::svc::MapSharedMemory(sharedHandle, addr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.baseAddress == addr);
            ASSERT_TRUE(info.size == size);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == perm);

            CheckWrite(addr, size);
            CheckRead(addr, size);

            result = nn::svc::UnmapSharedMemory(sharedHandle, addr, size);
            ASSERT_RESULT_SUCCESS(result);
        }

        // TEST 902-5
        // 共有メモリをアラインに沿ってマップすることが出来る
        // 間に1 ブロック
        {
            addr = g_FreeAreaBegin ;
            size = 3 * TestSharedSizes[i];

            result = nn::svc::CreateSharedMemory(&sharedHandle, size, perm, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(sharedHandle);

            result = nn::svc::MapSharedMemory(sharedHandle, addr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.baseAddress == addr);
            ASSERT_TRUE(info.size == size);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == perm);

            CheckWrite(addr, size);
            CheckRead(addr, size);

            result = nn::svc::UnmapSharedMemory(sharedHandle, addr, size);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
        }

        // TEST 902-6
        // 共有メモリを両端がアラインに沿っていない状態でマップすることが出来る
        // 間に1 ブロック
        {
            addr = g_FreeAreaBegin + BaseAlign;
            size = 3 * TestSharedSizes[i];

            result = nn::svc::CreateSharedMemory(&sharedHandle, size, perm, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(sharedHandle);

            result = nn::svc::MapSharedMemory(sharedHandle, addr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.baseAddress == addr);
            ASSERT_TRUE(info.size == size);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == perm);

            CheckWrite(addr, size);
            CheckRead(addr, size);

            result = nn::svc::UnmapSharedMemory(sharedHandle, addr, size);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
        }

        // TEST 902-7
        // 共有メモリの終端がアライメントに沿っていない状態でマップすることが出来る
        // 間に1 ブロック
        {
            addr = g_FreeAreaBegin;
            size = 2 * TestSharedSizes[i] + BaseAlign;

            result = nn::svc::CreateSharedMemory(&sharedHandle, size, perm, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(sharedHandle);

            result = nn::svc::MapSharedMemory(sharedHandle, addr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.baseAddress == addr);
            ASSERT_TRUE(info.size == size);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == perm);

            CheckWrite(addr, size);
            CheckRead(addr, size);

            result = nn::svc::UnmapSharedMemory(sharedHandle, addr, size);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
        }

        // TEST 902-8
        // 共有メモリを始端がアライメントに沿っていない状態でマップすることが出来る
        // 間に1 ブロック
        {
            addr = g_FreeAreaBegin + BaseAlign;
            size = 3 * TestSharedSizes[i] - BaseAlign;

            result = nn::svc::CreateSharedMemory(&sharedHandle, size, perm, perm);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sharedCloser(sharedHandle);

            result = nn::svc::MapSharedMemory(sharedHandle, addr, size, perm);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.baseAddress == addr);
            ASSERT_TRUE(info.size == size);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == perm);

            CheckWrite(addr, size);
            CheckRead(addr, size);

            result = nn::svc::UnmapSharedMemory(sharedHandle, addr, size);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
        }
    }
} // NOLINT(impl/function_size)

TEST(LargePageTable, HeapMemory)
{
    size_t heapSize;
    ConsumeHeapMemory maxHeap;
    heapSize = maxHeap.GetSize();
    heapSize -= OtherAppSize;
    TestHeap heap(heapSize);
    uintptr_t heapAddr = heap.GetAddress();

    nn::Result result;
    nn::svc::MemoryInfo info;
    nn::svc::MemoryPermission validPermission[] = {
        nn::svc::MemoryPermission_Read,
        nn::svc::MemoryPermission_None,
        nn::svc::MemoryPermission_ReadWrite,
    };
    int32_t permSize = sizeof(validPermission) / sizeof(nn::svc::MemoryPermission);

    for (int32_t sizeIndex = 0; sizeIndex < static_cast<int32_t>(sizeof(TestHeapSizes) / sizeof(size_t)); sizeIndex++)
    {
        size_t size = TestHeapSizes[sizeIndex];

        // TEST 902-9
        // 変更するパーミッションの領域がアラインに沿っている
        {
            uintptr_t beginAddr = RoundUp<uintptr_t>(heapAddr, size);

            int32_t testCount = sizeof(validPermission) / sizeof(nn::svc::MemoryPermission);
            for (int32_t i = 0; i < testCount; i++)
            {
                uintptr_t addr = beginAddr + i * size;
                nn::svc::MemoryPermission perm = validPermission[i % permSize];

                CheckWrite(addr, size);

                result = nn::svc::SetMemoryPermission(addr, size, perm);
                ASSERT_RESULT_SUCCESS(result);

                GetMemoryInfo(&info, addr);
                ASSERT_TRUE(info.state == nn::svc::MemoryState_Normal);
                ASSERT_TRUE(info.permission == perm);
                ASSERT_TRUE(info.attribute == 0);
                ASSERT_TRUE(info.baseAddress == addr);
                ASSERT_TRUE(info.size >= size);

                switch(perm)
                {
                    case nn::svc::MemoryPermission_ReadWrite:
                        CheckWrite(addr, size);
                        CheckRead(addr, size);
                        break;
                    case nn::svc::MemoryPermission_Read:
                        CheckRead(addr, size);
                        break;
                    case nn::svc::MemoryPermission_None:
                        TEST_NA(addr);
                        break;
                    default: FAIL();
                }
            }

            for (int32_t i = 0; i < testCount; i++)
            {
                uintptr_t addr = beginAddr + i * size;
                nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_ReadWrite;

                result = nn::svc::SetMemoryPermission(addr, size, perm);
                ASSERT_RESULT_SUCCESS(result);

                GetMemoryInfo(&info, addr);
                ASSERT_TRUE(info.baseAddress == heapAddr);
                size_t tmpSize = addr + size - heapAddr;
                ASSERT_TRUE(info.size >= tmpSize);
                ASSERT_TRUE(info.state == nn::svc::MemoryState_Normal);
                ASSERT_TRUE(info.permission == perm);
                ASSERT_TRUE(info.attribute == 0);

                CheckWrite(addr, size);
                CheckRead(addr, size);
            }

            GetMemoryInfo(&info, heapAddr);
            ASSERT_TRUE(info.baseAddress == heapAddr);
            ASSERT_TRUE(info.size == heapSize);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Normal);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(info.attribute == 0);
        }

        // TEST 902-10
        // 変更するパーミッションの領域の両端がアラインに沿っていない
        {
            uintptr_t beginAddr = static_cast<uintptr_t>(RoundUp(heapAddr, size)) + BaseAlign;
            size_t targetSize = size;

            SCOPED_TRACE("");
            ASSERT_NO_FATAL_FAILURE(TestHeapArea(beginAddr, targetSize, heapAddr, heapSize));
        }

        // TEST 902-11
        // 変更するパーミッションの領域の終端がアライメントに沿っていない
        {
            uintptr_t beginAddr = static_cast<uintptr_t>(RoundUp(heapAddr, size));
            size_t targetSize = size + BaseAlign;

            SCOPED_TRACE("");
            ASSERT_NO_FATAL_FAILURE(TestHeapArea(beginAddr, targetSize, heapAddr, heapSize));
        }

        // TEST 902-12
        // 変更するパーミッションの領域の始端がアライメントに沿っていない
        {
            uintptr_t beginAddr = static_cast<uintptr_t>(RoundUp(heapAddr, size)) + BaseAlign;
            size_t targetSize = size * 2 - BaseAlign;

            SCOPED_TRACE("");
            ASSERT_NO_FATAL_FAILURE(TestHeapArea(beginAddr, targetSize, heapAddr, heapSize));
        }

        // TEST 902-13
        // 変更するパーミッションの領域がアラインに沿っている
        // 間に1 ブロック
        {
            uintptr_t beginAddr = static_cast<uintptr_t>(RoundUp(heapAddr, size));
            size_t targetSize = 3 * size;

            SCOPED_TRACE("");
            ASSERT_NO_FATAL_FAILURE(TestHeapArea(beginAddr, targetSize, heapAddr, heapSize));
        }

        // TEST 902-14
        // 変更するパーミッションの領域の両端がアラインに沿っていない
        // 間に1 ブロック
        {
            uintptr_t beginAddr = static_cast<uintptr_t>(RoundUp(heapAddr, size)) + BaseAlign;
            size_t targetSize = size * 2;

            SCOPED_TRACE("");
            ASSERT_NO_FATAL_FAILURE(TestHeapArea(beginAddr, targetSize, heapAddr, heapSize));
        }

        // TEST 902-15
        // 変更するパーミッションの領域の終端がアライメントに沿っていない
        // 間に1 ブロック
        {
            uintptr_t beginAddr = static_cast<uintptr_t>(RoundUp(heapAddr, size));
            size_t targetSize = size * 2 + BaseAlign;

            SCOPED_TRACE("");
            ASSERT_NO_FATAL_FAILURE(TestHeapArea(beginAddr, targetSize, heapAddr, heapSize));
        }

        // TEST 902-16
        // 変更するパーミッションの領域の始端がアライメントに沿っていない
        // 間に1 ブロック
        {
            uintptr_t beginAddr = static_cast<uintptr_t>(RoundUp(heapAddr, size)) + BaseAlign;
            size_t targetSize = size * 3 - BaseAlign;

            SCOPED_TRACE("");
            ASSERT_NO_FATAL_FAILURE(TestHeapArea(beginAddr, targetSize, heapAddr, heapSize));
        }
    }
} // NOLINT(impl/function_size)
