﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_HardwareParamsSelect.h>

namespace {
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));

void TestThread0(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
}

void TestThread1(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t priority;

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(static_cast<int32_t>(arg) == priority);
}

} // namespace

extern "C" void nnMain();

/* TEST 10-3 */
/* PSEUDO_HANDLE_CURRENT_THREAD を渡し、自分のスレッドの優先度を取得できるかを確認する */
TEST(GetThreadPriorityTest, ValidHandleTest)
{
    TestThreadLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    int32_t priority;
    int32_t idealCore;
    uintptr_t sp;
    uintptr_t pc;

    sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    pc = reinterpret_cast<uintptr_t>(TestThread1);

    for (idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority; priority++)
        {
            result = nn::svc::CreateThread(&handle, pc, priority, sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}

/* TEST 10-4, 10-5, 10-6 */
/* 不正なハンドルを渡されたときに、エラーを返すかをチェックする */
TEST(GetThreadPriorityTest, InvalidHandleTest)
{
    TestThreadLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    int32_t priority;
    int32_t idealCore;
    int32_t tmpPriority;
    uintptr_t sp;
    uintptr_t pc;

    // TEST 10-4
    result = nn::svc::GetThreadPriority(&tmpPriority,
            nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 10-5
    result = nn::svc::GetThreadPriority(&tmpPriority,
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    pc = reinterpret_cast<uintptr_t>(TestThread0);

    for (idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority; priority++)
        {
            result = nn::svc::CreateThread(&handle, pc, priority, sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 10-6
            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::GetThreadPriority(&tmpPriority, handle);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        }
    }
}

#ifdef INVALID_POINTER_TEST
/* 不正なアドレスを渡されたとき、データアボートが発生することを確認する */
TEST(GetThreadPriorityTest, InvalidPointerTest)
{
    TestThreadLeak leakTest;
    nn::Result result;

#ifdef INVALID_POINTER_TEST
    // TEST 10-2
    // NULL を渡すと失敗する
    result = nn::svc::GetThreadPriority(NULL, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 10-13
    // MemoryPermission_None のアドレスを渡すと失敗する
    result = nn::svc::GetThreadPriority(
            reinterpret_cast<int32_t*>(g_FreeAreaBegin), nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 10-14
    // MemoryPermission_Read のアドレスを渡すと失敗する
    uintptr_t heapPtr;
    result = nn::svc::SetHeapSize(&heapPtr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetMemoryPermission(heapPtr, 0x1000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetThreadPriority(
            reinterpret_cast<int32_t*>(heapPtr), nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    result = nn::svc::SetMemoryPermission(heapPtr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 10-15
    // MemoryPermission_ReadExecute のアドレスを渡すと失敗する
    result = nn::svc::GetThreadPriority(
            reinterpret_cast<int32_t*>(nnMain), nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST
}
#endif // INVALID_POINTER_TEST

