﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>
#include <nn/nn_Common.h>

namespace {
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));
volatile bool g_StartFlag;
const int NumTest = 500;
const int32_t NumPriority = TestLowestThreadPriority - TestHighestThreadPriority + 1;
const int32_t StackSize = 0x1000;
const int64_t SleepTime = 100 * 1000 * 1000;
const int64_t ZeroIndex = 0xFFFFFFFFFFFFFFFF; // スレッドがこれだけ作られることはない想定

struct ThreadData
{
    nn::Bit64 id;
    nn::svc::Handle handle;
};

void TestThreadId()
{
    AutoThreadExit autoExit;
    while(!g_StartFlag)
    {
        nn::svc::SleepThread(SleepTime);
    }
}

/*
   データを挿入する箇所を見つける
   同じ値を持つデータがあれば、見つけた箇所を返す
   挿入する箇所は -1 を掛けた値になっている
 */
int64_t binarySearch(ThreadData *array, ThreadData *data, int64_t begin, int64_t end)
{
    if(array[begin].id > data->id)
    {
        return ZeroIndex;
    }
    else if(array[end - 1].id < data->id)
    {
        return -end;
    }

    int64_t middle = (end - begin) / 2 + begin;
    if(array[middle].id == data->id)
    {
        return middle;
    }
    if (end - begin < 2)
    {
        return -(begin + 1);
    }
    else if(array[middle].id < data->id)
    {
        return binarySearch(array, data, middle, end);
    }
    else
    {
        return binarySearch(array, data, begin, middle);
    }
}

/*
   データを挿入する
   同じ値を持つデータが既にあれば false
   挿入に成功すれば true
 */
bool InsertList(ThreadData *array, ThreadData *data, int64_t num)
{
    int64_t index;
    if(num == 0)
    {
        array[num] = *data;
        return true;
    }
    index = binarySearch(array, data, 0, num);
    if(index >= 0)
    {
        return false;
    }
    if (index == ZeroIndex)
    {
        index = 0;
    }
    else
    {
        index *= -1;
    }

    // 挿入
    array[num] = *data;
    for(int64_t i = index; i < num; i++)
    {
        ThreadData tmp = array[i];
        array[i] = array[num];
        array[num] = tmp;
    }

    return true;
}

} // namespace

/* TEST 32-5 */
/* 生成したスレッドの ID がユニークであることを確認する */
TEST(GetThreadId, CheckUniqueId)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t pc;
    uintptr_t sp;
    int32_t idealCore;
    int32_t priority;
    uintptr_t heapPtr;
    size_t wantSize = (sizeof(ThreadData) * NumTest + StackSize) * NN_KERN_SLAB_OBJ_NUM_THREAD;
    size_t heapSize = HeapAlign >= wantSize ? HeapAlign
        : (wantSize + HeapAlign - 1) & ~(HeapAlign - 1);
    ThreadData* threadArray;
    int64_t idLastIndex = 0;
    nn::Bit64 threadId = 0;
    int64_t i;
    int64_t handleIndex = 0;

    result = nn::svc::SetHeapSize(&heapPtr, heapSize);
    ASSERT_RESULT_SUCCESS(result);
    threadArray = reinterpret_cast<ThreadData*>(heapPtr);

    pc = reinterpret_cast<uintptr_t>(TestThreadId);
    sp = heapPtr + heapSize;

    for(int64_t testCount = 0; testCount < NumTest; testCount++)
    {
        for(i = 0;;i++)
        {
            idealCore = i % NumCore;
            priority = TestHighestThreadPriority + i % NumPriority;
            result = nn::svc::CreateThread(&handle, pc, 0, sp - i * StackSize, priority, idealCore);
            if (result.IsFailure())
            {
                ASSERT_TRUE(result <= nn::svc::ResultOutOfResource()
                        || result <= nn::svc::ResultMaxHandle());
                break;
            }

            result = nn::svc::GetThreadId(&threadId, handle);
            ASSERT_RESULT_SUCCESS(result);
            ThreadData data = {threadId, handle};
            ASSERT_TRUE(InsertList(threadArray, &data, idLastIndex));
            idLastIndex++;
        }

        for (i = handleIndex; i < idLastIndex; i++)
        {
            result = nn::svc::CloseHandle(threadArray[i].handle);
            ASSERT_RESULT_SUCCESS(result);
        }

        handleIndex = idLastIndex;
    }

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(GetThreadId, GetThreadIdAnyTime)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t sp;
    uintptr_t pc;
    nn::Bit64 threadId;
    nn::Bit64 tmpThreadId;

    sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    pc = reinterpret_cast<uintptr_t>(TestThreadId);

    for(int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for(int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            g_StartFlag = false;
            result = nn::svc::CreateThread(&handle, pc, 0, sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 32-6 (同じコア), 32-7 (違うコア)
            // 実行前にスレッドの ID が取得できる
            result = nn::svc::GetThreadId(&threadId, handle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 32-8 (同じコア), 32-9 (違うコア)
            // 実行中にスレッドの ID が取得できる
            result = nn::svc::GetThreadId(&tmpThreadId, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(threadId == tmpThreadId);

            g_StartFlag = true;

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 32-10 (同じコア), 32-11 (違うコア)
            // 実行後にスレッドの ID が取得できる
            result = nn::svc::GetThreadId(&tmpThreadId, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(threadId == tmpThreadId);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}

