﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>
#include <nn/nn_Common.h>

namespace {

char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));
volatile int g_Value;
#ifdef INVALID_POINTER_TEST
const size_t MemorySize = 0x1000;
#endif
const int64_t SleepTime = 100 * 1000 * 1000;

void LoopThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    while(g_Value == 0)
    {
        nn::svc::SleepThread(SleepTime);
    }
}

} // namespace

extern "C" void nnMain();

TEST(GetThreadCoreMask, NormalCase)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t sp;
    uintptr_t pc;
    int32_t tmpCore;
    nn::Bit64 tmpMask;

    pc = reinterpret_cast<uintptr_t>(LoopThread);
    sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));

    // TEST 12-8
    result = nn::svc::GetThreadCoreMask(&tmpCore, &tmpMask, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE((tmpMask & (1ULL << tmpCore)) > 0);

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority;
                priority++)
        {
            g_Value = 0;
            result = nn::svc::CreateThread(&handle, pc, 0, sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::GetThreadCoreMask(&tmpCore, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            // TEST 12-2 (同じコア), 12-5 (違うコア)
            ASSERT_TRUE(tmpCore == idealCore);
            // TEST 12-9 (同じコア), 12-12 (違うコア)
            ASSERT_TRUE((tmpMask & (1ULL << idealCore)) > 0);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::GetThreadCoreMask(&tmpCore, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            // TEST 12-3 (同じコア), 12-6 (違うコア)
            ASSERT_TRUE(tmpCore == idealCore);
            // TEST 12-10 (同じコア), 12-13 (違うコア)
            ASSERT_TRUE((tmpMask & (1ULL << idealCore)) > 0);

            g_Value = 1;

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::GetThreadCoreMask(&tmpCore, &tmpMask, handle);
            ASSERT_RESULT_SUCCESS(result);
            // TEST 12-4 (同じコア), 12-7 (違うコア)
            ASSERT_TRUE(tmpCore == idealCore);
            // TEST 12-11 (同じコア), 12-14 (違うコア)
            ASSERT_TRUE((tmpMask & (1ULL << idealCore)) > 0);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}

TEST(GetThreadCoreMask, InvalidHandle)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int32_t tmpCore;
    nn::Bit64 tmpMask;
    uintptr_t pc;
    uintptr_t sp;

    pc = reinterpret_cast<uintptr_t>(LoopThread);
    sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));

    // TEST 12-17
    result = nn::svc::GetThreadCoreMask(&tmpCore, &tmpMask, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 12-18
    result = nn::svc::GetThreadCoreMask(&tmpCore, &tmpMask, nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 12-19
    result = nn::svc::CreateThread(&handle, pc, 0, sp, 32, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::GetThreadCoreMask(&tmpCore, &tmpMask, handle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

#ifdef INVALID_POINTER_TEST
TEST(GetThreadCoreMask, InvalidPointer)
{
    TestThreadLeak leakTest;
    nn::Result result;
    int32_t tmpCore;
    nn::Bit64 tmpMask;

#ifdef INVALID_POINTER_TEST
    // TEST 12-15
    // NULL をpIdealCore に渡すと失敗する
    result = nn::svc::GetThreadCoreMask(NULL, &tmpMask,
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 12-16
    // NULL をpAffinityMask に渡すと失敗する
    result = nn::svc::GetThreadCoreMask(&tmpCore, NULL,
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 12-20
    // MemoryPermission_None の領域をpIdealCore に渡すと失敗する
    result = nn::svc::GetThreadCoreMask(reinterpret_cast<int32_t*>(g_FreeAreaBegin), &tmpMask,
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 12-21
    // MemoryPermission_None の領域をpAffinityMask に渡すと失敗する
    result = nn::svc::GetThreadCoreMask(&tmpCore, reinterpret_cast<nn::Bit64*>(g_FreeAreaBegin),
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif //INVALID_POINTER_TEST
    {
        uintptr_t heapPtr;
        size_t heapSize = HeapAlign;

        result = nn::svc::SetHeapSize(&heapPtr, heapSize);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SetMemoryPermission(heapPtr, MemorySize, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

#ifdef INVALID_POINTER_TEST
        // TEST 12-22
        // MemoryPermission_Read の領域をpIdealCore に渡すと失敗する
        result = nn::svc::GetThreadCoreMask(reinterpret_cast<int32_t*>(heapPtr), &tmpMask,
                nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif //INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 12-23
        // MemoryPermission_Read の領域をpAffinityMask に渡すと失敗する
        result = nn::svc::GetThreadCoreMask(&tmpCore, reinterpret_cast<nn::Bit64*>(heapPtr),
                nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif //INVALID_POINTER_TEST

        result = nn::svc::SetMemoryPermission
            (heapPtr, MemorySize, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SetHeapSize(&heapPtr, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef INVALID_POINTER_TEST
    // TEST 12-24
    // MemoryPermission_ReadExecute の領域をpIdealCore に渡すと失敗する
    result = nn::svc::GetThreadCoreMask(reinterpret_cast<int32_t*>(nnMain), &tmpMask,
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif //INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 12-25
    // MemoryPermission_ReadExecute の領域をpAffinityMask に渡すと失敗する
    result = nn::svc::GetThreadCoreMask(&tmpCore, reinterpret_cast<nn::Bit64*>(nnMain),
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif //INVALID_POINTER_TEST

}
#endif //INVALID_POINTER_TEST

