﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <cstring>

extern "C" void nnMain();

namespace {

#ifdef INVALID_POINTER_TEST
const int64_t ConstVar = 0;
#endif

void SetDefaultParam(nn::svc::CreateProcessParameter* param)
{
    std::strncpy(param->name, "test", 11);
    param->version        = 0xbabeface;
    param->programId      = 0xdeadbeef0badcafeull;
    param->memoryAddress  = 0x00200000;
    param->memoryNumPages = 0x00000100;
    param->flags          = 0x00000000;
}

void CreateAndSetupProcess(nn::svc::Handle* pOut, const nn::svc::CreateProcessParameter& param,
                           uint32_t* codePtr, size_t codeSize)
{
    nn::Result result;
    nn::svc::Handle handle;

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(
            g_FreeAreaBegin, handle, param.memoryAddress, param.memoryNumPages * 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    uint32_t* ptr = reinterpret_cast<uint32_t*>(g_FreeAreaBegin);
    ::std::memcpy(ptr, codePtr, codeSize);

    result = nn::svc::UnmapProcessMemory(
            g_FreeAreaBegin, handle, param.memoryAddress, param.memoryNumPages * 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetProcessMemoryPermission(
            handle, param.memoryAddress, param.memoryNumPages * 0x1000,
            nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_SUCCESS(result);

    *pOut = handle;
}

void CreateRunnableProcess(nn::svc::Handle* pOut)
{
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);
    uint32_t code = 0xef000000 | NN_SVC_ID_EXIT_PROCESS; // svc NN_SVC_ID_EXIT_PROCESS
    CreateAndSetupProcess(pOut, param, &code, sizeof(uint32_t));
}

} // namespace


TEST(GetProcessInfo, NormalCase)
{
    TestProcessLeak leakTest;
    nn::Result result;
    int64_t info;
    int64_t *pInfo;

    // TEST 124-1
    // MemoryPermission_ReadWrite を受け付ける

    // TEST 124-9
    // プロセスの擬似ハンドルを受け付ける

    // TEST 124-13
    // DebugRequested を受け付ける

    // TEST 124-17
    // DebugRequested を指定したときに、
    // 対象のプロセスが RUNNING 状態で、かつ JIT デバッグとなる状ではない場合には、
    // False が返ってくる


    // ローカル変数
    result = nn::svc::GetProcessInfo(
            &info, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    NN_LOG("info: %d\n", info);
    ASSERT_TRUE(
            static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Running ||
            static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Attached);

    TestHeap heap(sizeof(int64_t));
    pInfo = reinterpret_cast<int64_t*>(heap.GetAddress());

    // ヒープ
    result = nn::svc::GetProcessInfo(
            pInfo, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(
            static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Running ||
            static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Attached);
}

#ifdef INVALID_POINTER_TEST
TEST(GetProcessInfo, pOutTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    int64_t *pInfo;

#ifdef INVALID_POINTER_TEST
    // TEST 124-2
    // NULL を受け付けない
    pInfo = NULL;
    result = nn::svc::GetProcessInfo(
            pInfo, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 124-3
    // MemoryPermission_None を受け付けない
    pInfo = reinterpret_cast<int64_t*>(g_FreeAreaBegin);
    result = nn::svc::GetProcessInfo(
            pInfo, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 124-4
    // MemoryPermission_Read を受け付けない
    uintptr_t addr = reinterpret_cast<uintptr_t>(&ConstVar);
    pInfo = reinterpret_cast<int64_t*>(addr);
    result = nn::svc::GetProcessInfo(
            pInfo, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 124-5
    // MemoryPermission_ReadWrite を受け付けない
    pInfo = reinterpret_cast<int64_t*>(nnMain);
    result = nn::svc::GetProcessInfo(
            pInfo, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST
}
#endif // INVALID_POINTER_TEST

TEST(GetProcessInfo, HandleTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int64_t info;
    CreateRunnableProcess(&handle);

    // TEST 124-8
    // プロセスのハンドルを受け付ける
    result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Initializing);

    // TEST 124-10
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::GetProcessInfo(
            &info, nn::svc::INVALID_HANDLE_VALUE,
            nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 124-11
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::GetProcessInfo(
            &info, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
            nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 124-12
    // Close したプロセスのハンドルを受け付けない
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::GetProcessInfo(
            &info, handle,
            nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(GetProcessInfo, TypeTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    int64_t info;

    // TEST 124-14
    // ProcessInfoType ではない値を受け付けない
    result = nn::svc::GetProcessInfo(
            &info, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            static_cast<nn::svc::ProcessInfoType>(0xdeadbeaf));
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidEnum());
}

