﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Tcb.h>
#include <cstring>

namespace {
extern "C" void nnMain();
} // namespace

TEST(FlushProcessDataCache, NormalCase)
{
    nn::Result result;
    int32_t localVar = 0;

    // ヒープの設定
    TestHeap heap(HeapAlign);

    // TEST 85-1
    // addr/size が指し示す領域が書き込みできる領域の時、成功する
    // TEST 85-15
    // プロセスの擬似ハンドルを受け付ける

    nn::svc::Handle handle = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&localVar);

    result = nn::svc::FlushProcessDataCache(handle, addr, 0x10);
    ASSERT_RESULT_SUCCESS(result);

    addr = heap.GetAddress();

    result = nn::svc::InvalidateProcessDataCache(handle, addr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::InvalidateProcessDataCache(handle, addr, 0x10);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 85-33
    // 書き込み領域のある領域内であれば、ブロックの最初でなくても成功する
    result = nn::svc::FlushProcessDataCache(handle, addr + 0x10, 0x10);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 85-34
    // ページ境界をまたぐ領域でも成功する
    result = nn::svc::FlushProcessDataCache(handle, addr + 0xff0, 0x20);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::FlushProcessDataCache(handle, addr + 0xff0, 0x1010);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::FlushProcessDataCache(handle, addr + 0xff0, 0x1020);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 85-35
    // 領域の終端条件の確認
    result = nn::svc::FlushProcessDataCache(handle, addr + heap.GetSize() - 0x10, 0x10);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::FlushProcessDataCache(handle, addr + heap.GetSize() - 0x10, 0x10);
    ASSERT_RESULT_SUCCESS(result);

}

TEST(FlushProcessDataCache, MemoryAreaTest)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle handle = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    uintptr_t addr;
    size_t size = 0x1000;

    // ヒープの設定
    TestHeap heap(HeapAlign);

    // TEST 85-2
    // MemoryPermission_None の領域を指していると失敗する
    addr = g_FreeAreaBegin;
    result = nn::svc::FlushProcessDataCache(handle, addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 85-38
    // SetMemoryPermission で MemoryPermission_None を指定した領域を与えると成功する
    {
        addr = heap.GetAddress();
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_None);

        result = nn::svc::FlushProcessDataCache(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 85-3
    // MemoryPermission_Read の領域を指していると成功する
    {
        addr = heap.GetAddress();
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_Read);

        result = nn::svc::FlushProcessDataCache(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 85-4
    // MemoryPermission_ReadExecute の領域を指していると成功する
    addr = reinterpret_cast<uintptr_t>(nnMain);
    result = nn::svc::FlushProcessDataCache(handle, addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 85-7
    // addr/size の組み合わせがオーバーフローを起こす場合、失敗する
    addr = static_cast<uintptr_t>(-1);
    result = nn::svc::FlushProcessDataCache(handle, addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    addr = heap.GetAddress();
    // TEST 85-8
    // メモリ状態が一様でないと失敗する (前方部分)
    {
        addr = heap.GetAddress() - 0x10;
        TestMemoryBlock blockInfo(addr);
        ASSERT_TRUE(!blockInfo.CheckPermission(nn::svc::MemoryPermission_ReadWrite));
        result = nn::svc::FlushProcessDataCache(handle, addr, 0x20);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }


    // TEST 85-9
    // メモリ状態が一様でないと失敗する (後方部分)
    {
        addr = heap.GetAddress() + heap.GetSize() - 0x10;
        TestMemoryBlock blockInfo(addr + 0x10);
        ASSERT_TRUE(!blockInfo.CheckPermission(nn::svc::MemoryPermission_ReadWrite));
        result = nn::svc::FlushProcessDataCache(handle, addr, 0x20);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 85-10
    // addr に 0 を入れると失敗する
    addr = 0;
    result = nn::svc::FlushProcessDataCache(handle, addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 85-11
    // size に 0 を入れると失敗する
    addr = heap.GetAddress();
    size = 0;
    result = nn::svc::FlushProcessDataCache(handle, addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    for (int32_t i = 1; i < 0x1000; i++)
    {
        // TEST 85-12
        // addr に アラインされてないアドレスを入れることが出来る
        addr = heap.GetAddress() + i;
        size = 0x1000;
        result = nn::svc::FlushProcessDataCache(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 85-13
        // size に アラインされてない値を入れることが出来る
        addr = heap.GetAddress();
        size = 0x1000 + i;
        result = nn::svc::FlushProcessDataCache(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 85-14
        // アライメントされていない値を入れることが出来る
        addr = heap.GetAddress() + i;
        size = 0x1000 + i;
        result = nn::svc::FlushProcessDataCache(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 85-36
    // 作成したプロセスのMemoryPermission_None の領域を受け付けない
    {
        nn::svc::Handle handle;
        nn::svc::CreateProcessParameter param = {};
        std::strncpy(param.name, "test", sizeof(param.name));
        param.version        = 0xbabeface;
        param.programId      = 0xdeadbeef0badcafeull;
        param.memoryAddress  = 0x00200000;
        param.memoryNumPages = 0x00000100;
        param.flags          = 0x00000000;

        nn::Bit32 flags[DefaultCapabilityFlagNum];
        SetDefaultCapability(flags, DefaultCapabilityFlagNum);
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);

        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SetProcessMemoryPermission(
                handle, param.memoryAddress , 4 * 0x1000,
                nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SetProcessMemoryPermission(
                handle, param.memoryAddress + 4 * 0x1000, 4 * 0x1000,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        // 念のため確認
        addr = param.memoryAddress + 4 * 0x1000;
        size = 0x10;
        result = nn::svc::StoreProcessDataCache(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);

        addr = param.memoryAddress;
        size = 0x10;
        result = nn::svc::FlushProcessDataCache(handle, addr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
} // NOLINT (readability/fn_size)

TEST(FlushProcessDataCache, HandleTest)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&result);
    size_t size = 0x1000;

    // TEST 85-16
    // INVALID_HANDLE_VALUE を受け付けない
    handle = nn::svc::INVALID_HANDLE_VALUE;
    result = nn::svc::FlushProcessDataCache(handle, addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 85-17
    // スレッドの擬似ハンドルを受け付けない
    handle = nn::svc::PSEUDO_HANDLE_CURRENT_THREAD;
    result = nn::svc::FlushProcessDataCache(handle, addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 85-18
    // Close されたプロセスのハンドルを受け付けない
    {
        nn::svc::CreateProcessParameter param = {};
        std::strncpy(param.name, "test", sizeof(param.name));
        param.version        = 0xbabeface;
        param.programId      = 0xdeadbeef0badcafeull;
        param.memoryAddress  = 0x00200000;
        param.memoryNumPages = 0x00000100;
        param.flags          = 0x00000000;

        nn::Bit32 flags[DefaultCapabilityFlagNum];
        SetDefaultCapability(flags, DefaultCapabilityFlagNum);
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SetProcessMemoryPermission(
                handle, param.memoryAddress, 4 * 0x1000,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        // 念のため確認
        addr = param.memoryAddress;
        size = 0x10;
        result = nn::svc::StoreProcessDataCache(handle, addr, size);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::FlushProcessDataCache(handle, addr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }
}

TEST(FlushDataCache, NormalCase)
{
    nn::Result result;
    int32_t localVar = 0;

    // ヒープの設定
    TestHeap heap(HeapAlign);

    // TEST 85-1
    // addr/size が指し示す領域が書き込みできる領域の時、成功する
    // TEST 85-15
    // プロセスの擬似ハンドルを受け付ける

    nn::svc::Handle handle = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&localVar);

    result = nn::svc::FlushDataCache(addr, 0x10);
    ASSERT_RESULT_SUCCESS(result);

    addr = heap.GetAddress();

    result = nn::svc::InvalidateProcessDataCache(handle, addr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::InvalidateProcessDataCache(handle, addr, 0x10);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 85-33
    // 書き込み領域のある領域内であれば、ブロックの最初でなくても成功する
    result = nn::svc::FlushDataCache(addr + 0x10, 0x10);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 85-34
    // ページ境界をまたぐ領域でも成功する
    result = nn::svc::FlushDataCache(addr + 0xff0, 0x20);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::FlushDataCache(addr + 0xff0, 0x1010);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::FlushDataCache(addr + 0xff0, 0x1020);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 85-35
    // 領域の終端条件の確認
    result = nn::svc::FlushDataCache(addr + heap.GetSize() - 0x10, 0x10);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::FlushDataCache(addr + heap.GetSize() - 0x10, 0x10);
    ASSERT_RESULT_SUCCESS(result);

}

TEST(FlushDataCache, MemoryAreaTest)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    uintptr_t addr;
    size_t size = 0x1000;

    // ヒープの設定
    TestHeap heap(HeapAlign);

    // TEST 85-2
    // MemoryPermission_None の領域を指していると失敗する
    addr = g_FreeAreaBegin;
    result = nn::svc::FlushDataCache(addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 85-38
    // SetMemoryPermission で MemoryPermission_None を指定した領域を与えると成功する
    {
        addr = heap.GetAddress();
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_None);

        result = nn::svc::FlushDataCache(addr, size);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 85-3
    // MemoryPermission_Read の領域を指していると成功する
    {
        addr = heap.GetAddress();
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_Read);

        result = nn::svc::FlushDataCache(addr, size);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 85-4
    // MemoryPermission_ReadExecute の領域を指していると成功する
    addr = reinterpret_cast<uintptr_t>(nnMain);
    result = nn::svc::FlushDataCache(addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 85-7
    // addr/size の組み合わせがオーバーフローを起こす場合、失敗する
    addr = static_cast<uintptr_t>(-1);
    result = nn::svc::FlushDataCache(addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    addr = heap.GetAddress();
    // TEST 85-8
    // メモリ状態が一様でないと失敗する (前方部分)
    {
        addr = heap.GetAddress() - 0x10;
        TestMemoryBlock blockInfo(addr);
        ASSERT_TRUE(!blockInfo.CheckPermission(nn::svc::MemoryPermission_ReadWrite));
        result = nn::svc::FlushDataCache(addr, 0x20);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }


    // TEST 85-9
    // メモリ状態が一様でないと失敗する (後方部分)
    {
        addr = heap.GetAddress() + heap.GetSize() - 0x10;
        TestMemoryBlock blockInfo(addr + 0x10);
        ASSERT_TRUE(!blockInfo.CheckPermission(nn::svc::MemoryPermission_ReadWrite));
        result = nn::svc::FlushDataCache(addr, 0x20);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 85-10
    // addr に 0 を入れると失敗する
    addr = 0;
    result = nn::svc::FlushDataCache(addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 85-11
    // size に 0 を入れると失敗する
    addr = heap.GetAddress();
    size = 0;
    result = nn::svc::FlushDataCache(addr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    for (int32_t i = 1; i < 0x1000; i++)
    {
        // TEST 85-12
        // addr に アラインされてないアドレスを入れることが出来る
        addr = heap.GetAddress() + i;
        size = 0x1000;
        result = nn::svc::FlushDataCache(addr, size);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 85-13
        // size に アラインされてない値を入れることが出来る
        addr = heap.GetAddress();
        size = 0x1000 + i;
        result = nn::svc::FlushDataCache(addr, size);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 85-14
        // アライメントされていない値を入れることが出来る
        addr = heap.GetAddress() + i;
        size = 0x1000 + i;
        result = nn::svc::FlushDataCache(addr, size);
        ASSERT_RESULT_SUCCESS(result);
    }
} // NOLINT (readability/fn_size)

