﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_DebugProcess.h"

namespace {
const int64_t SleepTime = 100 * 1000 * 1000;
}


// アタッチ -> スタート -> 正常終了
TEST(DebugProcess, NormalCase)
{
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    nn::Bit64 pid;
    result = nn::svc::GetProcessId(&pid, processHandle);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_Initializing));

    nn::svc::Handle debug;
    result = nn::svc::DebugActiveProcess(&debug, pid);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_PreAttached));

    process.Start();

    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_Breaked));

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;
    bool isAttach = false;
    bool isCreateProcess = false;
    bool isCreateThread = false;

    while(!(isAttach && isCreateProcess && isCreateThread))
    {
        result = WaitAndGetDebugEvent(&debugInfo, debug, SleepTime);
        if (result <= nn::svc::ResultTimeout() && isAttach)
        {
            result = ContinueDebugProcess(debug);
            ASSERT_RESULT_SUCCESS(result);
        }

        if (debugInfo.event == nn::svc::DebugEvent_Exception)
        {
            except = debugInfo.info.exception;
            if (except.exceptionCode == nn::svc::DebugException_AttachBreak)
            {
                isAttach = true;
            }
            else
            {
                ASSERT_TRUE(false);
            }
        }
        else if (debugInfo.event == nn::svc::DebugEvent_CreateProcess)
        {
            isCreateProcess = true;
        }
        else if (debugInfo.event == nn::svc::DebugEvent_CreateThread)
        {
            isCreateThread = true;
        }
        else
        {
            ASSERT_TRUE(false);
        }
    }

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_Attached));

    for(;;)
    {
        result = WaitAndGetDebugEvent(&debugInfo, debug);
        ASSERT_RESULT_SUCCESS(result);

        if (debugInfo.event == nn::svc::DebugEvent_ExitThread)
        {
            continue;
        }
        else if(debugInfo.event == nn::svc::DebugEvent_ExitProcess)
        {
            break;
        }
        else
        {
            ASSERT_TRUE(false);
        }
    }

    result = nn::svc::CloseHandle(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

// アタッチ -> スタート -> ブレーク -> コンティニュー -> ブレーク -> コンティニュー -> 正常終了
TEST(DebugProcess, BreacAndContinue)
{
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    nn::svc::Handle debug;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;
    InitDebugProcess(&debug, nullptr, &process);

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debug, 1);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

// アタッチ -> スタート -> ブレーク -> Terminate
TEST(DebugProcess, TerminateDebugProcessWithBreak)
{
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    nn::svc::Handle debug;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    InitDebugProcess(&debug, nullptr, &process);

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    result = nn::svc::TerminateDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

// アタッチ -> スタート -> ブレーク -> Disconnect
TEST(DebugProcess, DisconnectDebugProcessWithBreak)
{
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    nn::svc::Handle debug;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    InitDebugProcess(&debug, nullptr, &process);

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    result = nn::svc::CloseHandle(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

// アタッチ -> スタート -> Terminate
TEST(DebugProcess, TerminateDebugProcess)
{
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    nn::svc::Handle debug;

    InitDebugProcess(&debug, nullptr, &process);

    result = nn::svc::TerminateDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

// アタッチ -> スタート -> Disconnect
TEST(DebugProcess, DisconnectDebugProcess)
{
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    nn::svc::Handle debug;
    InitDebugProcess(&debug, nullptr, &process);

    result = nn::svc::CloseHandle(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

// アタッチ -> スタート -> ブレーク -> コンティニュー -> Terminate
TEST(DebugProcess, TerminateDebugProcessWithBreakAndContinue)
{
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    nn::svc::Handle debug;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;
    InitDebugProcess(&debug, nullptr, &process);

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::TerminateDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

// アタッチ -> スタート -> ブレーク -> コンティニュー -> Disconnect
TEST(DebugProcess, DisconnectDebugProcessWithBreakAndContinue)
{
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    nn::svc::Handle debug;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;
    InitDebugProcess(&debug, nullptr, &process);

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

