﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "test_DASProcess.h"
#include "util_TestSdmmc.h"
#include "util_TestIpc.h"
#include "util_TestLoader.h"
#include <nn/svc/svc_Dd.h>
#include <cstring>

#ifdef SUPPORT_SDCARD_TEST

namespace {
    const uint64_t SpaceAddr = 0;
#ifdef SUPPORT_MMC_TEST
    const nn::sdmmc1::Port TestPort_Mmc = nn::sdmmc1::Port_Mmc0;    // sdmmc4a
#endif // SUPPORT_MMC_TEST
    const nn::sdmmc1::Port TestPort_SdCard = nn::sdmmc1::Port_SdCard0; // sdmmc3a

    const uint32_t TestSectorIndex = 0;
    const uint32_t TestSize = 32 * 1024;
    const uint32_t TestNumSectors = TestSize / nn::sdmmc1::SectorSize;
#ifdef SUPPORT_MMC_TEST
    NN_ALIGNAS(4096) uint8_t s_Buffer[TestSize];
#endif
    NN_ALIGNAS(4096) uint8_t g_BackupBuffer[TestSize];

} // namespace

extern char BinDasProcess_begin[];
extern char BinDasProcess_end[];
TEST(DeviceAddressSpace, MapMultiProcess)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    size_t align = 0x400000;
    uint64_t deviceAddress = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);
    AutoDetachDevice devCloser(handle, DeviceName_SdCard);

    uintptr_t heapAddr;
    size_t heapSize;

    nn::svc::Handle readableHandle;
    nn::svc::Handle writableHandle;
    result = nn::svc::CreateEvent(&writableHandle, &readableHandle);
    ASSERT_RESULT_SUCCESS(result);

    AutoHandleClose wEventCloser(writableHandle);
    NamedPortManager namedPort(PortName, 1);


    TestLoader loader(BinDasProcess_begin, reinterpret_cast<uintptr_t>(BinDasProcess_end) - reinterpret_cast<uintptr_t>(BinDasProcess_begin));
    nn::svc::Handle processHandle;
    loader.SpawnProcess(&processHandle);
    loader.StartProcess(processHandle);
    AutoHandleClose processCloser(processHandle);

    nn::svc::ResetSignal(processHandle);

    nn::svc::Handle portHandle = namedPort.GetHandle();
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &portHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, portHandle);
    ASSERT_RESULT_SUCCESS(result);

    AutoHandleClose sSessionCloser(serverSession);


    SendMoveHandleFromServer(readableHandle, serverSession);
    SendCopyHandleFromServer(handle, serverSession);

    {
        heapSize = align * 3;
        TestHeap heap(heapSize);
        ASSERT_TRUE(heapSize == heap.GetSize());
        heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
        deviceAddress = align;

        nn::svc::PhysicalMemoryInfo physInfo = { 0 };
        result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
        ASSERT_RESULT_SUCCESS(result);
        // 4MBアラインされるはず
        uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
        ASSERT_TRUE((physAddr & (align - 1)) == 0);

        heapSize -= heapAddr - heap.GetAddress();

        // Sdmmc が使えるか確認
        {
            char* buffer = reinterpret_cast<char*>(heapAddr);
            Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort_SdCard);
            ASSERT_TRUE(sdmmc.Activate());
        }

        result = nn::svc::MapDeviceAddressSpaceByForce(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
        ASSERT_TRUE(blockInfo.deviceCount == 1);

        result = nn::svc::SignalEvent(writableHandle);
        ASSERT_RESULT_SUCCESS(result);

        // 同期をとるためだけに、データを受信
        uintptr_t tmpAddr;
        ServerReceiveData(&tmpAddr, sizeof(tmpAddr), serverSession);

        {
            char* buffer = reinterpret_cast<char*>(heapAddr);
            Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort_SdCard);
            ASSERT_TRUE(sdmmc.Activate());

            // バックアップ
            nn::sdmmc1::Read(buffer, TestSize, TestPort_SdCard, TestSectorIndex, TestNumSectors);
            std::memcpy(g_BackupBuffer, buffer, TestSize);

            std::strcpy(buffer, TestMessage);
            size_t msgSize = std::strlen(TestMessage) + 1;
            uint32_t numSectors = msgSize / nn::sdmmc1::SectorSize;
            if (numSectors == 0)
            {
                msgSize = nn::sdmmc1::SectorSize;
                numSectors = 1;
            }
            nn::sdmmc1::Write(TestPort_SdCard, TestSectorIndex, numSectors, buffer, msgSize);

            std::memset(buffer, 0, msgSize);
            nn::sdmmc1::Read(buffer, msgSize, TestPort_SdCard, TestSectorIndex, numSectors);
            NN_ASSERT(::strcmp(TestMessage, buffer) == 0);
        }

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.deviceCount == 0);

        result = nn::svc::SignalEvent(writableHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        // 元に戻す
        {
            result = nn::svc::MapDeviceAddressSpaceByForce(
                    handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                    heapAddr, TestSize, deviceAddress,
                    nn::svc::MemoryPermission_ReadWrite);
            ASSERT_RESULT_SUCCESS(result);

            nn::svc::MemoryInfo blockInfo;
            GetMemoryInfo(&blockInfo, heapAddr);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
            ASSERT_TRUE(blockInfo.deviceCount == 1);

            char* buffer = reinterpret_cast<char*>(heapAddr);
            Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort_SdCard);
            ASSERT_TRUE(sdmmc.Activate());

            std::memcpy(buffer, g_BackupBuffer, TestSize);
            nn::sdmmc1::Write(TestPort_SdCard, TestSectorIndex, TestNumSectors, buffer, TestSize);

            // チェック
            std::memset(buffer, 0, TestSize);
            nn::sdmmc1::Read(buffer, TestSize, TestPort_SdCard, TestSectorIndex, TestNumSectors);
            ASSERT_TRUE(std::memcmp(g_BackupBuffer, buffer, TestSize) == 0);

            result = nn::svc::UnmapDeviceAddressSpace(
                    handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                    heapAddr, TestSize, deviceAddress);
            ASSERT_RESULT_SUCCESS(result);

            GetMemoryInfo(&blockInfo, heapAddr);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(blockInfo.attribute == 0);
            ASSERT_TRUE(blockInfo.deviceCount == 0);
        }
    }
} // NOLINT(readability/fn_size)

extern "C" void nnMain();

#ifdef SUPPORT_MMC_TEST
TEST(DeviceAddressSpace, MapOverlap)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle1;
    nn::svc::Handle handle2;
    size_t align = 0x400000;
    uint64_t deviceAddress = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle1, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser1(handle1);

    result = nn::svc::CreateDeviceAddressSpace(&handle2, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser2(handle2);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle1);
    ASSERT_RESULT_SUCCESS(result);
    AutoDetachDevice sdCloser(handle1, DeviceName_SdCard);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_Mmc, handle2);
    ASSERT_RESULT_SUCCESS(result);
    AutoDetachDevice mmcCloser(handle2, DeviceName_Mmc);

    uintptr_t heapAddr;
    size_t heapSize;

    {
        heapSize = align * 3;
        TestHeap heap(heapSize);
        ASSERT_TRUE(heapSize == heap.GetSize());
        heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);
        deviceAddress = align;

        nn::svc::PhysicalMemoryInfo physInfo = { 0 };
        result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
        ASSERT_RESULT_SUCCESS(result);
        // 4MBアラインされるはず
        uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
        ASSERT_TRUE((physAddr & (align - 1)) == 0);

        heapSize -= heapAddr - heap.GetAddress();

        // Sdmmc が使えるか確認
        {
            char* buffer = reinterpret_cast<char*>(heapAddr);
            Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort_SdCard);
            ASSERT_TRUE(sdmmc.Activate());
        }
        {
            char* buffer = reinterpret_cast<char*>(heapAddr);
            Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort_Mmc);
            ASSERT_TRUE(sdmmc.Activate());
        }

        // 重複する領域を作成
        result = nn::svc::MapDeviceAddressSpaceByForce(
                handle1, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapDeviceAddressSpaceByForce(
                handle2, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
        ASSERT_TRUE(blockInfo.deviceCount == 2);

        char* buffer = reinterpret_cast<char*>(heapAddr);
        {
            Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort_Mmc);
            ASSERT_TRUE(sdmmc.Activate());
            std::memset(buffer, 0, TestSize);
            nn::sdmmc1::Read(buffer, TestSize, TestPort_Mmc, TestSectorIndex, TestNumSectors);
            bool check =false;
            for (int i = 0; i < TestSize; i++)
            {
                if (buffer[i] != 0)
                {
                    check = true;
                    break;
                }
            }
            ASSERT_TRUE(check);
            std::memcpy(s_Buffer, buffer, TestSize);
        }
        {
            Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort_SdCard);
            ASSERT_TRUE(sdmmc.Activate());
            std::memset(buffer, 0, TestSize);
            nn::sdmmc1::Read(buffer, TestSize, TestPort_SdCard, TestSectorIndex, TestNumSectors);
            std::memcpy(g_BackupBuffer, buffer, TestSize);

            std::memcpy(buffer, reinterpret_cast<void*>(nnMain), TestSize);
            nn::sdmmc1::Write(TestPort_SdCard, TestSectorIndex, TestNumSectors, buffer, TestSize);

            std::memset(buffer, 0, TestSize);
            nn::sdmmc1::Read(buffer, TestSize, TestPort_SdCard, TestSectorIndex, TestNumSectors);
            ASSERT_TRUE(::memcmp(buffer, reinterpret_cast<void*>(nnMain), TestSize) == 0);
        }
        {
            Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort_Mmc);
            ASSERT_TRUE(sdmmc.Activate());
            std::memset(buffer, 0, TestSize);
            nn::sdmmc1::Read(buffer, TestSize, TestPort_Mmc, TestSectorIndex, TestNumSectors);
            ASSERT_TRUE(::memcmp(buffer, s_Buffer, TestSize) == 0);
        }
        {
            Sdmmc sdmmc(buffer, TestSize, deviceAddress, TestPort_SdCard);
            ASSERT_TRUE(sdmmc.Activate());
            std::memcpy(buffer, g_BackupBuffer, TestSize);
            nn::sdmmc1::Write(TestPort_SdCard, TestSectorIndex, TestNumSectors, buffer, TestSize);

            std::memset(buffer, 0, TestSize);
            nn::sdmmc1::Read(buffer, TestSize, TestPort_SdCard, TestSectorIndex, TestNumSectors);
            ASSERT_TRUE(::memcmp(buffer, g_BackupBuffer, TestSize) == 0);
        }

        result = nn::svc::UnmapDeviceAddressSpace(
                handle1, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
        ASSERT_TRUE(blockInfo.deviceCount == 1);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle2, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, TestSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.deviceCount == 0);
    }
} // NOLINT(impl/function_size)
#endif // SUPPORT_MMC_TEST
#endif // SUPPORT_SDCARD_TEST

